/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

/*!
 \file dtatoms.cc
 \brief iq̌qzuf[^̃NX
*/

#include <stdio.h>
#include "dtatoms.h"
#include "dtcif.h"
#include "dtcell.h"
#include "dtlattice.h"
#include "qtmisc.h"
#include "qtexception.h"
#include "symbol.h"
#include <algorithm>


DTElement::DTElement( const QString& name )
{ // TAPP3
  this->number = ElementSymbol::getAtomicNumber(name);
  this->name = ElementSymbol::getAtomicName(number);
  this->zo   = ElementSymbol::getAtomicValence(number);
  this->zn   = double(number);
  this->mass = ElementSymbol::getAtomicMass(number);

  this->file_pseudo = "ps-" + this->name;
  this->file_charge = "ps-" + this->name + ".ichr";
}

DTElement::DTElement( const double zo, const double zn )
{ // xTAPP
  this->number = (int) zn;
  this->name = ElementSymbol::getAtomicName(number);
  this->zo   = zo;
  this->zn   = zn;
  this->mass = ElementSymbol::getAtomicMass(number);

  this->file_pseudo = "ps-" + this->name;
  this->file_charge = "ps-" + this->name + ".ichr";
}

//-------------------------------------

DTAtom::DTAtom
( const DTElement& element, const Coordinates& coords, const int motion )
{
  this->element  = element;
  this->coords  = coords;
  this->velocity = Position(0.0,0.0,0.0);
  Coordinates::normalize(this->coords);
  this->motion = motion;

  copy.flag = false;
  duplicate.flag = false;
  special.flag = false;
  special.vmatrix.clear();
}

const QString& DTAtom::getElement( void ) const
{
  return this->element.name;
}
const int& DTAtom::getNumber( void ) const
{
  return this->element.number;
}

const Coordinates& DTAtom::getCoordinates( void ) const
{
  return coords;
}


void DTAtom::setAsOriginal( int index )
{
  copy.flag  = false;
  copy.index = index;
  copy.matrix     = IMatrix::getIdentity();
  copy.matrix_inv = IMatrix::getIdentity();
  duplicate.flag = false;
  special.flag = false;
  special.vmatrix.clear();
  special.matrix_const = IMatrix::getIdentity();
}

void DTAtom::setAsCopy( int index, const IMatrix& matrix )
{
  copy.flag  = true;
  copy.index = index;
  copy.matrix = matrix;
  copy.matrix_inv = matrix.getTransposed();
  duplicate.flag = false;
  special.flag = false;
  special.vmatrix.clear();
  special.matrix_const = IMatrix::getIdentity();
}

int DTAtom::getIndex( void ) const
{
  return copy.index;
}

const IMatrix& DTAtom::getMatrix( void ) const
{
  return copy.matrix;
}
const IMatrix& DTAtom::getMatrixInv( void ) const
{
  return copy.matrix_inv;
}

bool DTAtom::isCopy( void ) const
{
  return copy.flag;
}

void DTAtom::setAsDuplicate( void )
{
  duplicate.flag = true;
}

bool DTAtom::isDuplicate( void ) const
{
  return duplicate.flag;
}

void DTAtom::setAsSpecial( const IMatrix& matrix )
{
  special.flag = true;
  special.vmatrix.push_back(matrix);

  special.matrix_const = matrix.getConstraint() * special.matrix_const;
}

bool DTAtom::isSpecial( void ) const
{
  return special.flag;
}


const IMatrix& DTAtom::getConstraintMatrix( void ) const
{
  return special.matrix_const;
}

bool DTAtom::translateCoordinates( const Coordinates& coords )
{
  if( coords == Coordinates(0.0,0.0,0.0) ){
    return false;
  }
  this->coords += coords;
  Coordinates::normalize(this->coords);
  
  return true;
}

bool DTAtom::changeCoordinates( const Coordinates& coords )
{
  if( this->coords == coords ){
    return false;
  }
  this->coords = coords;
  Coordinates::normalize(this->coords);

  return true;
}

bool DTAtom::changeElement( const DTElement& element )
{
  if( this->element == element ){
    return false;
  }
  this->element = element;

  return true;
}

bool DTAtom::changeMotion( const int motion )
{
  if( this->motion == motion ){
    return false;
  }
  this->motion = motion;

  return true;
}

//-------------------------------------


DTAtoms::DTAtoms( DTCell& cell )
{
  this->cell = &cell;
  clear();
}

void DTAtoms::clear( void )
{
  vatom.clear();
  unselectAtom();
  alignment = ANY;
  velement_index_shown = 1;
  velement.clear();
  velement_value_shown = DTElement("H");
}

bool DTAtoms::load( DTLattice& lattice, const QString& fname )
{
  if( false );
  else if( fname.endsWith( ".xyz" ) ){
    return loadXYZ(fname);
  }
  else if( fname.endsWith( ".pdb" ) ){
    return loadPDB(fname);
  }
  else if( fname.endsWith( ".cif" ) ){
    return loadCIF(lattice,fname);
  }
  else{}

  return false;
}

bool DTAtoms::save( const QString& fname )
{
  if( false );
  else if( fname.endsWith( ".xyz" ) ){
    return saveXYZ(fname);
  }
  else{}

  return false;
}

bool DTAtoms::loadXYZ( const QString& fname )
{
  FILE* fptr = fopen( fname, "rb" );
  if( fptr == NULL ){
    throw MyException("can not open a molecule file.",fname);
    //    return false;
  }

  clear();

  char buf[256];

  int Na = 0;
  Position L = Position(0.0,0.0,0.0);
  char name[8];
  Position r;

  fgets( buf, sizeof(buf), fptr ); // SKIP total atoms line
  if( 1+3*3 == sscanf( buf, "%d %lf %lf %lf %lf %lf %lf %lf %lf %lf",
		       &Na, 
		       &cell->Ea.a,&cell->Ea.b,&cell->Ea.c,
		       &cell->Eb.a,&cell->Eb.b,&cell->Eb.c,
		       &cell->Ec.a,&cell->Ec.b,&cell->Ec.c ) ){
    cell->length = 1.0;
    cell->vectors_changed = true;
    cell->update();
  }
  if( 1+3 == sscanf( buf, "%d %lf %lf %lf",
		     &Na, &L.x, &L.y, &L.z ) ){
    cell->length = 1.0;
    cell->Ea = Position(L.x,0.0,0.0);
    cell->Eb = Position(0.0,L.y,0.0);
    cell->Ec = Position(0.0,0.0,L.z);
    cell->update();
  }
  else if( 1 == sscanf( buf, "%d", &Na ) ){
  }
  else{
    throw MyException("broken header in a molecule file.",fname);
  }

  fgets( buf, sizeof(buf), fptr ); // SKIP commnet line

  while( fgets( buf, sizeof(buf), fptr ) ){
    if( 4 != sscanf( buf, "%s %lf %lf %lf",
		     name, &r.x, &r.y, &r.z ) ) continue;

    const DTElement* elem = findElement(QString(name));
    if( elem == NULL ){
      velement.push_back(DTElement(QString(name)));
      elem = &velement.back();
    }
    
    vatom.push_back( DTAtom( *elem, cell->getCoordinatesNormalized(r) ) );
    vatom.back().setAsOriginal( vatom.size()-1 );
  }

  fclose(fptr);

  return true;
}

bool DTAtoms::loadCIF( DTLattice& lattice, const QString& fname )
{
  DTCIF cif;
  if( !cif.load(fname) ){
    throw MyException("can not open a molecule file.",fname);
  }

  clear();
  {
    cell->length = 1.0;
    cell->Ea = cif.unit_prim[0];
    cell->Eb = cif.unit_prim[1];
    cell->Ec = cif.unit_prim[2];
    cell->vectors_changed = true;
    cell->update();
  }

  for( int i=0; i<(int)cif.vatom_prim.size(); i++ ){
    const QString& name = cif.vatom_prim[i].element;

    const DTElement* elem = findElement(name);
    if( elem == NULL ){
      velement.push_back(DTElement(name));
      elem = &velement.back();
    }
    
    vatom.push_back( DTAtom( *elem, cif.vatom_prim[i].coords ) );
  }

  lattice.symmetry.name = cif.spacegroup.nameHM;
  lattice.symmetry.has_inversion = cif.has_inv;
  lattice.symmetry.vmatrix = cif.voperation_prim;
  lattice.symmetry.expand();

  /*
  {
    FILE* fptr = fopen("tmp.db","a");
    fprintf(fptr,"spacegroup_name: %s\n",
	    qPrintable(cif.spacegroup.nameHM) );
    fprintf(fptr,"spacegroup_number: %d\n",
	    cif.spacegroup.number );
    fprintf(fptr,"has_inversion: %d\n",
	    (int)lattice.symmetry.has_inversion );
    fprintf(fptr,"number_operations: %d\n",
	    lattice.symmetry.vmatrix.size() );
    fprintf(fptr,"each_operation:\n");
    for( int n=0; n<(int)lattice.symmetry.vmatrix.size(); n++ ){
      fprintf(fptr," %3d %s\n",
	      n+1, lattice.symmetry.vmatrix[n].name() );
    }
    fprintf(fptr,"\n");
    fclose(fptr);
  }
  exit(0);
  */

  if( !DTSymmetry::findDB(lattice.symmetry) ){
    MyException::warning("unknown new symmetry");
    //    return false;
    return true;
  }

  return true;
}

bool DTAtoms::saveXYZ( const QString& fname )
{
  FILE* fptr = fopen( fname, "wb" );
  if( fptr == NULL ){
    throw MyException("can not open a molecule file.",fname);
  }

  int natom=0;
  for( int n=0; n<(int)vatom.size(); n++ ){
    if( vatom[n].isCopy() ) continue;
    natom++;
  }

  fprintf(fptr, "%d %f %f %f %f %f %f %f %f %f\n",
	  natom,
	  cell->Ea.a,cell->Ea.b,cell->Ea.c,
	  cell->Eb.a,cell->Eb.b,cell->Eb.c,
	  cell->Ec.a,cell->Ec.b,cell->Ec.c );
  fprintf(fptr, "\n");
  for( int n=0; n<(int)vatom.size(); n++ ){
    if( vatom[n].isCopy() ) continue;

    const Position r = cell->getPositionLocal(vatom[n].coords);
    fprintf(fptr, "%s %+12.6f %+12.6f %+12.6f\n", qPrintable(vatom[n].element.name), r.x, r.y, r.z );
  }
  fclose(fptr);

  return true;
}

static int mscanf( char* buf, size_t pos, size_t len, char* s ){
  strncpy( s, buf+pos, len );
  s[len] = '\0';
  return 1;
}
static int mscanf( char* buf, size_t pos, size_t len, double* d ){
  char work[16];
  strncpy( work, buf+pos, len );
  work[len] = '\0';
  return sscanf( work, "%lf", d );
}


bool DTAtoms::loadPDB( const QString& fname )
{
  FILE* fptr = fopen( fname, "rb" );
  if( fptr == NULL ){
    throw MyException("can not open a molecule file.",fname);
    //    return false;
  }

  clear();

  const char* error_msg = NULL;
  char buf[1024];

  char name[8];
  Position r;

  while( fgets( buf, sizeof(buf), fptr ) ){
    if( strncmp( buf, "ATOM  ", 6 ) ) continue;

    if( mscanf( buf, 13, 1, name ) != 1 ||
	mscanf( buf, 30, 8, &r.x ) != 1 ||
	mscanf( buf, 38, 8, &r.y ) != 1 ||
	mscanf( buf, 46, 8, &r.z ) != 1 ){
      error_msg = "broken data.";
      goto error_block;
    }

    int ppid = findPPID(name);
    if( ppid == -1 ){
      ppid = velement.size();
      velement.push_back(DTElement(name));
    }
    vatom.push_back( DTAtom( velement[ppid], cell->getCoordinatesNormalized(r), 1 ) );
    vatom.back().setAsOriginal( vatom.size()-1 );
  }
  fclose(fptr);

  return true;

 error_block:
  fclose(fptr);

  throw MyException(error_msg,fname);
  return false; // dummy statement
}



void DTAtoms::update( void )
{
  switch( alignment ){
  case ANY : break; // nothing to do
  case SIMPLE : {
    if( velement.empty() ){
      velement.push_back( DTElement("H") );
    }
    vatom.clear();
    vatom.push_back( DTAtom(velement.front(),Coordinates(0.00,0.00,0.00),1) );
    vatom.back().setAsOriginal(0);
  } break;
  case FACE_CENTERED : {
    if( velement.empty() ){
      velement.push_back( DTElement("H") );
    }
    vatom.clear();
    vatom.push_back( DTAtom(velement.front(),Coordinates(0.00,0.00,0.00),1) );
    vatom.back().setAsOriginal(0);
    vatom.push_back( DTAtom(velement.front(),Coordinates(0.00,0.50,0.50),1) );
    vatom.back().setAsOriginal(1);
    vatom.push_back( DTAtom(velement.front(),Coordinates(0.50,0.00,0.50),1) );
    vatom.back().setAsOriginal(2);
    vatom.push_back( DTAtom(velement.front(),Coordinates(0.50,0.50,0.00),1) );
    vatom.back().setAsOriginal(3);
  } break;
  case BODY_CENTERED : {
    if( velement.empty() ){
      velement.push_back( DTElement("H") );
    }
    vatom.clear();
    vatom.push_back( DTAtom(velement.front(),Coordinates(0.00,0.00,0.00),1) );
    vatom.back().setAsOriginal(0);
    vatom.push_back( DTAtom(velement.front(),Coordinates(0.50,0.50,0.50),1) );
    vatom.back().setAsOriginal(1);
  } break;
  case BASE_CENTERED : {
    if( velement.empty() ){
      velement.push_back( DTElement("H") );
    }
    vatom.clear();
    vatom.push_back( DTAtom(velement.front(),Coordinates(0.00,0.00,0.00),1) );
    vatom.back().setAsOriginal(0);
    vatom.push_back( DTAtom(velement.front(),Coordinates(0.50,0.50,0.00),1) );
    vatom.back().setAsOriginal(1);
  } break;
  case DIAMOND : {
    if( velement.empty() ){
      velement.push_back( DTElement("H") );
    }
    vatom.clear();
    vatom.push_back( DTAtom(velement.front(),Coordinates(0.00,0.00,0.00),1) );
    vatom.back().setAsOriginal(0);
    vatom.push_back( DTAtom(velement.front(),Coordinates(0.00,0.50,0.50),1) );
    vatom.back().setAsOriginal(1);
    vatom.push_back( DTAtom(velement.front(),Coordinates(0.50,0.00,0.50),1) );
    vatom.back().setAsOriginal(2);
    vatom.push_back( DTAtom(velement.front(),Coordinates(0.50,0.50,0.00),1) );
    vatom.back().setAsOriginal(3);

    vatom.push_back( DTAtom(velement.front(),Coordinates(0.25,0.25,0.25),1) );
    vatom.back().setAsOriginal(4);
    vatom.push_back( DTAtom(velement.front(),Coordinates(0.25,0.75,0.75),1) );
    vatom.back().setAsOriginal(5);
    vatom.push_back( DTAtom(velement.front(),Coordinates(0.75,0.25,0.75),1) );
    vatom.back().setAsOriginal(6);
    vatom.push_back( DTAtom(velement.front(),Coordinates(0.75,0.75,0.25),1) );
    vatom.back().setAsOriginal(7);

  } break;
  }
}

bool DTAtoms::selectAtom( const int selected )
{
  // Iԍ͑IΏۊȎIӖ
  if( selected == -1 ) return false;

  // ̂ēxIƑI
  for( int s=0; s<(int)vselected.size(); s++ ){
    if( vselected[s] == selected ){
      unselectAtom();
      return true;
    }
  }
  // Iԍ̍XV
  vselected.clear();
  vselected.push_back(selected);

  // Iꂽ̌qԍƍW̍XV
  viatom.clear();
  vcoords_selected.clear();

  for( int s=0; s<(int)vselected.size(); s++ ){
    // Iԍ猴qł̒o
    int ia = vselected[s] % vatom.size();
    // IԍxNg̒o
    Coordinates L;
    int ix = (vselected[s]/vatom.size())/1%3;
    int iy = (vselected[s]/vatom.size())/3%3;
    int iz = (vselected[s]/vatom.size())/9%3;
    L.a = ix==0 ? 0.0 : ix==1 ? +1.0 : -1.0;
    L.b = iy==0 ? 0.0 : iy==1 ? +1.0 : -1.0;
    L.c = iz==0 ? 0.0 : iz==1 ? +1.0 : -1.0;

    viatom.push_back(ia);
    vcoords_selected.push_back( vatom[ia].coords + L );
  }

  // ŌɑIꂽqԍƍW̋L^
  this-> iatom = viatom.back();
  this->coords_selected = vcoords_selected.back();

  updateBond();

  return true;
}

bool DTAtoms::addselectAtom( const int selected )
{
  if( selected == -1 ) return false;

  // ̂ēxIƑI
  for( int s=0; s<(int)vselected.size(); s++ ){
    if( vselected[s] == selected ){
      vselected.erase( vselected.begin()+s );
      s--;
      if( vselected.empty() ){
	unselectAtom();
	return true;
      }
    }
  }
  // Iԍ̒ǉ
  vselected.push_back(selected);

  // Iꂽ̌qԍƍW̍XV
  viatom.clear();
  vcoords_selected.clear();

  for( int s=0; s<(int)vselected.size(); s++ ){
    // Iԍ猴qł̒o
    int ia = vselected[s] % vatom.size();
    // IԍxNg̒o
    Coordinates L;
    int ix = (vselected[s]/vatom.size())/1%3;
    int iy = (vselected[s]/vatom.size())/3%3;
    int iz = (vselected[s]/vatom.size())/9%3;
    L.a = ix==0 ? 0.0 : ix==1 ? +1.0 : -1.0;
    L.b = iy==0 ? 0.0 : iy==1 ? +1.0 : -1.0;
    L.c = iz==0 ? 0.0 : iz==1 ? +1.0 : -1.0;

    viatom.push_back(ia);
    vcoords_selected.push_back( vatom[ia].coords + L );
  }

  // ŌɑIꂽqԍƍW̋L^
  this-> iatom = viatom.back();
  this->coords_selected = vcoords_selected.back();

  updateBond();

  return true;
}

bool DTAtoms::selectAtoms( const vector<int> viatom )
{
  if( viatom.empty() ){
    unselectAtom();
    return true;
  }

  for( int i=0; i<(int)viatom.size(); i++ ){
    if( viatom[i] >= (int)vatom.size() ){
      return false;
    }
  }

  this->viatom = viatom;
  this-> iatom = viatom.back();

  vcoords_selected.clear();
  vselected.clear();
  for( int i=0; i<(int)viatom.size(); i++ ){
    int ia = viatom[i];
    vselected.push_back(ia);
    vcoords_selected.push_back( vatom[ia].coords );
  }

  // ŌɑIꂽqԍƍW̋L^
  this-> iatom = viatom.back();
  this->coords_selected = vcoords_selected.back();

  updateBond();

  return true;
}

void DTAtoms::unselectAtom( void )
{
  this->viatom.clear();
  this->vselected.clear();
  this-> iatom = -1;
  this->vcoords_selected.clear();

  bond_length = 0.0;
  bond_angle = 0.0;
  bond_dihedral = 0.0;
}

int DTAtoms::getAtomSelected( void )
{
  return this->iatom;
}

const vector<int>& DTAtoms::getAtomsSelected( void )
{
  return this->viatom;
}

bool DTAtoms::isAtomSelected( void ) const
{
  return !viatom.empty();
}

bool DTAtoms::moveBond( void )
{
  bool status = false;

  switch( getDragMode() ){
  case 0 : { // IĂȂƂ͂肦ȂBȂB
  } break;
  case 2 : { // ύXB
    // _̓W̃Rs[(`̋tԂɒ)
    const Coordinates c0 = getDragCoordinate(1); // inversed order
    const Coordinates c1 = getDragCoordinate(0); // inversed order
    // _̍W
    const Position p0 = cell->getPositionLocal(c0);
    const Position p1 = cell->getPositionLocal(c1);

    // p0: ړq̌̍W
    // p1: ړqq̍W
    // ea xNg p1 --> p0 ƂĒ`
    const Position ea = (p0 - p1).normal();


    // ړΏیq݂̌̍W̓
    const Coordinates c = coords_selected;//vatom[iatom].getCoordinates();
    const Position p = cell->getPositionLocal(c);
    const Position pnew = p1 + ea*bond_length;

    // ړxNg̐ݒ
    const Position t0 = pnew - p;
    
    // ړxNgWɕϊ
    Coordinates coords0 = cell->getCoordinatesLocal(t0);
    // IWiqړWɕϊ
    coords0 = vatom[iatom].getMatrixInv() * coords0;
    Coordinates coords = coords0;

    // ړΏۂ̃IWiq̃CfbNX
    const int n = vatom[iatom].getIndex();
    // Ώ̖̐̐ɕϊ
    if( vatom[n].isSpecial() ){
      coords = vatom[n].getConstraintMatrix() * coords;
    }
    if( coords != coords0 ){
      coords = Coordinates(0.0,0.0,0.0);
    }
    // IWiqɑ΂Ĉړ̎s
    if( vatom[n].translateCoordinates(coords) ){
      coords_selected += coords;
      status = true;
    }
  } break;

  case 3 : { // pύXB
    // _̓W̃Rs[(`̋tԂɒ)
    const Coordinates c0 = getDragCoordinate(2); // inversed order
    const Coordinates c1 = getDragCoordinate(1); // inversed order
    const Coordinates c2 = getDragCoordinate(0); // inversed order
    // _̍W
    const Position p0 = cell->getPositionLocal(c0);
    const Position p1 = cell->getPositionLocal(c1);
    const Position p2 = cell->getPositionLocal(c2);

    // p0: ړq̌̍W
    // p1: ړqq̍W
    // p2: ɐ̌q̍W
    // ea xNgŒ茋 p1 --> p2 ƂĒ`
    // eb xNgɐp0,p1,p2ʓ p0ɒ`
    const Position ea = (p2 - p1).normal();
    const Position p0projected = ((p0 - p1)*ea)*ea + p1;
    const Position eb = (p0 - p0projected).normal();

    // ړΏیq݂̌̍W̓
    const Coordinates c = coords_selected;//vatom[iatom].getCoordinates();
    const Position p = cell->getPositionLocal(c);

    // WvZ
    const Position pnew = p1 + bond_length*
      ( + cos(bond_angle*M_PI/180.0)*ea
	+ sin(bond_angle*M_PI/180.0)*eb );
    // ړxNgČvZ
    const Position t0 = pnew - p;

    // ړxNgWɕϊ
    Coordinates coords0 = cell->getCoordinatesLocal(t0);
    // IWiqړWɕϊ
    coords0 = vatom[iatom].getMatrixInv() * coords0;
    Coordinates coords = coords0;

    // ړΏۂ̃IWiq̃CfbNX
    const int n = vatom[iatom].getIndex();
    // Ώ̖̐̐ɕϊ
    if( vatom[n].isSpecial() ){
      coords = vatom[n].getConstraintMatrix() * coords;
    }
    if( coords != coords0 ){
      coords = Coordinates(0.0,0.0,0.0);
    }
    // IWiqɑ΂Ĉړ̎s
    if( vatom[n].translateCoordinates(coords) ){
      coords_selected += coords;
      status = true;
    }
  } break;

  case 4 : { // ʊpύXB
    // _̓W̃Rs[(`̋tԂɒ)
    //    const Coordinates c0 = getDragCoordinate(3); // inversed order
    const Coordinates c1 = getDragCoordinate(2); // inversed order
    const Coordinates c2 = getDragCoordinate(1); // inversed order
    const Coordinates c3 = getDragCoordinate(0); // inversed order
    // _̍W
    //    const Position p0 = cell->getPositionLocal(c0);
    const Position p1 = cell->getPositionLocal(c1);
    const Position p2 = cell->getPositionLocal(c2);
    const Position p3 = cell->getPositionLocal(c3);

    // p0: ړq̌̍W
    // p1: ړqq̍W
    // p2: ɐ̌q̍W
    // p3: ɐ̌q̍W
    // ea xNgŒ茋 p1 --> p2 ƂĒ`
    // eb xNgɐp1,p2,p3ʓ p3ɒ`
    // ec xNgɐɒ`
    const Position ea = (p2 - p1).normal();
    const Position p3projected = ((p3 - p2)*ea)*ea + p2;
    const Position eb = (p3 - p3projected).normal();
    const Position ec = Position::outer_product(ea,eb);

    // ړΏیq݂̌̍W̓
    const Coordinates c = coords_selected;//vatom[iatom].getCoordinates();
    const Position p = cell->getPositionLocal(c);

    // WvZ
    const Position pnew = p1 + bond_length*
      ( + cos(bond_angle*M_PI/180.0)*ea
	+ sin(bond_angle*M_PI/180.0)*
	( + cos(bond_dihedral*M_PI/180.0)*eb
	  + sin(bond_dihedral*M_PI/180.0)*ec ) );
    // ړxNgČvZ
    const Position t0 = pnew - p;

    // ړxNgWɕϊ
    Coordinates coords0 = cell->getCoordinatesLocal(t0);
    // IWiqړWɕϊ
    coords0 = vatom[iatom].getMatrixInv() * coords0;
    Coordinates coords = coords0;

    // ړΏۂ̃IWiq̃CfbNX
    const int n = vatom[iatom].getIndex();
    // Ώ̖̐̐ɕϊ
    if( vatom[n].isSpecial() ){
      coords = vatom[n].getConstraintMatrix() * coords;
    }
    if( coords != coords0 ){
      coords = Coordinates(0.0,0.0,0.0);
    }
    // IWiqɑ΂Ĉړ̎s
    if( vatom[n].translateCoordinates(coords) ){
      coords_selected += coords;
      status = true;
    }
  } break;
  }

  return status;
}

bool DTAtoms::updateBond( void )
{
  bond_length   = 0.0;
  bond_angle    = 0.0;
  bond_dihedral = 0.0;

  if( !isAtomSelected() ) return false;

  switch( getDragMode() ){
  case 0 : { // IĂȂƂ͂肦ȂBȂB
  } break;
  case 2 : { // 
    // _̓W̃Rs[(`̋tԂɒ)
    //    const Coordinates c0 = getDragCoordinate(1); // inversed order
    const Coordinates c1 = getDragCoordinate(0); // inversed order
    // _̍W
    //    const Position p0 = cell->getPositionLocal(c0);
    const Position p1 = cell->getPositionLocal(c1);

    // p0: ړq̌̍W
    // p1: ړqq̍W
    // ea xNg p1 --> p0 ƂĒ`
    //    const Position ea = (p0 - p1).normal();

    // ړΏیq݂̌̍W̓
    const Coordinates c = coords_selected;//vatom[iatom].getCoordinates();
    const Position p = cell->getPositionLocal(c);

    bond_length   = Position::length(p-p1);
    bond_angle    = 0.0;
    bond_dihedral = 0.0;
  } break;

  case 3 : { // pύXB
    // _̓W̃Rs[(`̋tԂɒ)
    const Coordinates c0 = getDragCoordinate(2); // inversed order
    const Coordinates c1 = getDragCoordinate(1); // inversed order
    const Coordinates c2 = getDragCoordinate(0); // inversed order
    // _̍W
    const Position p0 = cell->getPositionLocal(c0);
    const Position p1 = cell->getPositionLocal(c1);
    const Position p2 = cell->getPositionLocal(c2);

    // p0: ړq̌̍W
    // p1: ړqq̍W
    // p2: ɐ̌q̍W
    // ea xNgŒ茋 p1 --> p2 ƂĒ`
    // eb xNgɐp0,p1,p2ʓ p0ɒ`
    const Position ea = (p2 - p1).normal();
    const Position p0projected = ((p0 - p1)*ea)*ea + p1;
    const Position eb = (p0 - p0projected).normal();

    // ړΏیq݂̌̍W̓
    const Coordinates c = coords_selected;//vatom[iatom].getCoordinates();
    const Position p = cell->getPositionLocal(c);

    bond_angle  = Position::angle( p-p1, p2-p1 );
    if( (p-p1)*eb < 0.0 ){
      bond_angle *= -1.0;// ̊pxɂ
    }
    // ͌̒lɌŒ
    bond_length = Position::length( p0-p1 );
    bond_dihedral = 0.0;
  } break;

  case 4 : { // ʊpύXB
    // _̓W̃Rs[(`̋tԂɒ)
    const Coordinates c0 = getDragCoordinate(3); // inversed order
    const Coordinates c1 = getDragCoordinate(2); // inversed order
    const Coordinates c2 = getDragCoordinate(1); // inversed order
    const Coordinates c3 = getDragCoordinate(0); // inversed order
    // _̍W
    const Position p0 = cell->getPositionLocal(c0);
    const Position p1 = cell->getPositionLocal(c1);
    const Position p2 = cell->getPositionLocal(c2);
    const Position p3 = cell->getPositionLocal(c3);

    // p0: ړq̌̍W
    // p1: ړqq̍W
    // p2: ɐ̌q̍W
    // p3: ɐ̌q̍W
    // ea xNgŒ茋 p1 --> p2 ƂĒ`
    // eb xNgɐp1,p2,p3ʓ p3ɒ`
    // ec xNgɐɒ`
    const Position ea = (p2 - p1).normal();
    const Position p3projected = ((p3 - p2)*ea)*ea + p2;
    const Position eb = (p3 - p3projected).normal();
    const Position ec = Position::outer_product(ea,eb);

    // ړΏیq݂̌̍W̓
    const Coordinates c = coords_selected;//vatom[iatom].getCoordinates();
    const Position p = cell->getPositionLocal(c);

    // ʊpvZ
    bond_dihedral = atan2( (p-p1)*ec, (p-p1)*eb )*180.0/M_PI;

    // px͌̒lɌŒ
    bond_angle  = Position::angle( p0-p1, p2-p1 );
    if( (p0-p1)*eb < 0.0 ){
	bond_angle *= -1.0;    // ̊pxɂ
    }
    // ͌̒lɌŒ
    bond_length = Position::length( p0-p1 );
  } break;
  }

  return true;
}

// 1̌qw̕Ɉړ֐
bool DTAtoms::moveAtomCoords( const Position& t )
{
  bool status = false;

  if( !isAtomSelected() ) return false;
  alignment = ANY;

  switch( getDragMode() ){
  case 0 : { // IĂȂƂ͂肦ȂBȂB
  } break;
  case 2 : { // ύXB
    // _̓W̃Rs[(`̋tԂɒ)
    const Coordinates c0 = getDragCoordinate(1); // inversed order
    const Coordinates c1 = getDragCoordinate(0); // inversed order
    // _̍W
    const Position p0 = cell->getPositionLocal(c0);
    const Position p1 = cell->getPositionLocal(c1);

    // p0: ړq̌̍W
    // p1: ړqq̍W
    // ea xNg p1 --> p0 ƂĒ`
    const Position ea = (p0 - p1).normal();
    // ړxNg̐ݒ
    const Position t0 = (t*ea)*ea;

    // ړΏیq݂̌̍W̓
    const Coordinates c = coords_selected;//vatom[iatom].getCoordinates();
    const Position p = cell->getPositionLocal(c);

    const Position pnew = p + t0;
    bond_length   = Position::length(pnew-p1);

    // ړxNgWɕϊ
    Coordinates coords0 = cell->getCoordinatesLocal(t0);
    // IWiqړWɕϊ
    coords0 = vatom[iatom].getMatrixInv() * coords0;
    Coordinates coords = coords0;

    // ړΏۂ̃IWiq̃CfbNX
    const int n = vatom[iatom].getIndex();
    // Ώ̖̐̐ɕϊ
    if( vatom[n].isSpecial() ){
      coords = vatom[n].getConstraintMatrix() * coords;
    }
    if( coords != coords0 ){
      coords = Coordinates(0.0,0.0,0.0);
    }
    // IWiqɑ΂Ĉړ̎s
    if( vatom[n].translateCoordinates(coords) ){
      coords_selected += coords;
      status = true;
    }
  } break;

  case 3 : { // pύXB
    // _̓W̃Rs[(`̋tԂɒ)
    const Coordinates c0 = getDragCoordinate(2); // inversed order
    const Coordinates c1 = getDragCoordinate(1); // inversed order
    const Coordinates c2 = getDragCoordinate(0); // inversed order
    // _̍W
    const Position p0 = cell->getPositionLocal(c0);
    const Position p1 = cell->getPositionLocal(c1);
    const Position p2 = cell->getPositionLocal(c2);

    // p0: ړq̌̍W
    // p1: ړqq̍W
    // p2: ɐ̌q̍W
    // ea xNgŒ茋 p1 --> p2 ƂĒ`
    // eb xNgɐp0,p1,p2ʓ p0ɒ`
    const Position ea = (p2 - p1).normal();
    const Position p0projected = ((p0 - p1)*ea)*ea + p1;
    const Position eb = (p0 - p0projected).normal();

    // ړxNgea,ebŒʓɐݒ
    Position t0 = (t*ea)*ea+(t*eb)*eb;

    // ړΏیq݂̌̍W̓
    const Coordinates c = coords_selected;//vatom[iatom].getCoordinates();
    const Position p = cell->getPositionLocal(c);

    // ɍWړāApxvZ
    Position pnew = p+t0;
    bond_angle  = Position::angle( pnew-p1, p2-p1 );
    if( (pnew-p1)*eb < 0.0 ){
      bond_angle *= -1.0;// ̊pxɂ
    }
    // ͌̒lɌŒ
    bond_length = Position::length( p0-p1 );

    // ͕ςɌpxςWvZ
    pnew = p1 + bond_length*
      ( + cos(bond_angle*M_PI/180.0)*ea
	+ sin(bond_angle*M_PI/180.0)*eb );
    // ړxNgČvZ
    t0 = pnew - p;

    // ړxNgWɕϊ
    Coordinates coords0 = cell->getCoordinatesLocal(t0);
    // IWiqړWɕϊ
    coords0 = vatom[iatom].getMatrixInv() * coords0;
    Coordinates coords = coords0;

    // ړΏۂ̃IWiq̃CfbNX
    const int n = vatom[iatom].getIndex();
    // Ώ̖̐̐ɕϊ
    if( vatom[n].isSpecial() ){
      coords = vatom[n].getConstraintMatrix() * coords;
    }
    if( coords != coords0 ){
      coords = Coordinates(0.0,0.0,0.0);
    }
    // IWiqɑ΂Ĉړ̎s
    if( vatom[n].translateCoordinates(coords) ){
      coords_selected += coords;
      status = true;
    }
  } break;

  case 4 : { // ʊpύXB
    // _̓W̃Rs[(`̋tԂɒ)
    const Coordinates c0 = getDragCoordinate(3); // inversed order
    const Coordinates c1 = getDragCoordinate(2); // inversed order
    const Coordinates c2 = getDragCoordinate(1); // inversed order
    const Coordinates c3 = getDragCoordinate(0); // inversed order
    // _̍W
    const Position p0 = cell->getPositionLocal(c0);
    const Position p1 = cell->getPositionLocal(c1);
    const Position p2 = cell->getPositionLocal(c2);
    const Position p3 = cell->getPositionLocal(c3);

    // p0: ړq̌̍W
    // p1: ړqq̍W
    // p2: ɐ̌q̍W
    // p3: ɐ̌q̍W
    // ea xNgŒ茋 p1 --> p2 ƂĒ`
    // eb xNgɐp1,p2,p3ʓ p3ɒ`
    // ec xNgɐɒ`
    const Position ea = (p2 - p1).normal();
    const Position p3projected = ((p3 - p2)*ea)*ea + p2;
    const Position eb = (p3 - p3projected).normal();
    const Position ec = Position::outer_product(ea,eb);

    // ړxNgeb,ecŒʓɐݒ
    Position t0 = (t*ec)*ec+(t*eb)*eb;

    // ړΏیq݂̌̍W̓
    const Coordinates c = coords_selected;//vatom[iatom].getCoordinates();
    const Position p = cell->getPositionLocal(c);

    // ɍWړāAʊpvZ
    Position pnew = p+t0;
    bond_dihedral = atan2( (pnew-p1)*ec, (pnew-p1)*eb )*180.0/M_PI;

    // px͌̒lɌŒ
    bond_angle  = Position::angle( p0-p1, p2-p1 );
    if( (p0-p1)*eb < 0.0 ){
	bond_angle *= -1.0;    // ̊pxɂ
    }
    // ͌̒lɌŒ
    bond_length = Position::length( p0-p1 );

    // Epx͕ςɓʊpςWvZ
    pnew = p1 + bond_length*
      ( + cos(bond_angle*M_PI/180.0)*ea
	+ sin(bond_angle*M_PI/180.0)*
	( + cos(bond_dihedral*M_PI/180.0)*eb
	  + sin(bond_dihedral*M_PI/180.0)*ec ) );
    // ړxNgČvZ
    t0 = pnew - p;

    // ړxNgWɕϊ
    Coordinates coords0 = cell->getCoordinatesLocal(t0);
    // IWiqړWɕϊ
    coords0 = vatom[iatom].getMatrixInv() * coords0;
    Coordinates coords = coords0;

    // ړΏۂ̃IWiq̃CfbNX
    const int n = vatom[iatom].getIndex();
    // Ώ̖̐̐ɕϊ
    if( vatom[n].isSpecial() ){
      coords = vatom[n].getConstraintMatrix() * coords;
    }
    if( coords != coords0 ){
      coords = Coordinates(0.0,0.0,0.0);
    }
    // IWiqɑ΂Ĉړ̎s
    if( vatom[n].translateCoordinates(coords) ){
      coords_selected += coords;
      status = true;
    }
  } break;

  default : { // RɍWύXB
    // ړxNgWɕϊ
    Coordinates coords0 = cell->getCoordinatesLocal(t);
    // IWiqړWɕϊ
    coords0 = vatom[iatom].getMatrixInv() * coords0;

    bond_length = 0.0;
    bond_angle  = 0.0;
    bond_dihedral = 0.0;

    // A,B,Cǂꂩ݂̂Ɉړ
    if( fabs(coords0.x) < fabs(coords0.y) ){
      if( fabs(coords0.y) < fabs(coords0.z) ){
	coords0.x = 0.0, coords0.y = 0.0;
      }
      else{
	coords0.x = 0.0, coords0.z = 0.0;
      }
    }
    else{
      if( fabs(coords0.x) < fabs(coords0.z) ){
	coords0.x = 0.0, coords0.y = 0.0;
      }
      else{
	coords0.y = 0.0, coords0.z = 0.0;
      }
    }
    Coordinates coords = coords0;

    // ړΏۂ̃IWiq̃CfbNX
    const int n = vatom[iatom].getIndex();
    // Ώ̖̐̐ɕϊ
    if( vatom[n].isSpecial() ){
      coords = vatom[n].getConstraintMatrix() * coords;
    }
    // IWiqɑ΂Ĉړ̎s
    if( vatom[n].translateCoordinates(coords) ){
      status = true;
    }
  } break;
  }

  return status;
}

bool DTAtoms::moveAtomsCoords( const Position& t )
{
  const Coordinates coords = cell->getCoordinatesLocal(t);

  bool status = false;
  if( !isAtomSelected() ) return status;
  alignment = ANY;

  Coordinates coords0 = coords;
  for( int i=0; i<(int)viatom.size(); i++ ){
    const int iatom=viatom[i];

    coords0 = vatom[iatom].getMatrixInv() * coords0;

    // A,B,Cǂꂩ݂̂Ɉړ
    if( fabs(coords0.x) < fabs(coords0.y) ){
      if( fabs(coords0.y) < fabs(coords0.z) ){
	coords0.x = 0.0, coords0.y = 0.0;
      }
      else{
	coords0.x = 0.0, coords0.z = 0.0;
      }
    }
    else{
      if( fabs(coords0.x) < fabs(coords0.z) ){
	coords0.x = 0.0, coords0.y = 0.0;
      }
      else{
	coords0.y = 0.0, coords0.z = 0.0;
      }
    }

    const int n = vatom[iatom].getIndex();
    if( vatom[n].isSpecial() ){
      coords0 = vatom[n].getConstraintMatrix() * coords0;
    }
  }

  for( int i=0; i<(int)viatom.size(); i++ ){
    const int iatom=viatom[i];
    const int n = vatom[iatom].getIndex();

    if( vatom[n].translateCoordinates(coords0) ){
      status = true;
    }
  }
  for( int s=0; s<(int)vcoords_selected.size(); s++ ){
    vcoords_selected[s] += coords0;
  }
  coords_selected += coords0;

  return status;
}

bool DTAtoms::changeAtomCoordinates( const Coordinates& coords )
{
  if( !isAtomSelected() ) return false;
  alignment = ANY;

  const Coordinates diff = coords - vatom[iatom].coords;

  Coordinates coords0 = vatom[iatom].getMatrixInv() * diff;

  const int n = vatom[iatom].getIndex();
  if( vatom[n].isSpecial() ){
    coords0 = vatom[n].getConstraintMatrix() * coords0;
  }

  return vatom[n].translateCoordinates(coords0);
}

bool DTAtoms::changeAtomCoordA( const double a )
{
  if( !isAtomSelected() ) return false;
  alignment = ANY;

  const Coordinates coords =
    Coordinates( a - vatom[iatom].coords.a, 0.0, 0.0 );
  Coordinates coords0 = vatom[iatom].getMatrixInv() * coords;

  const int n = vatom[iatom].getIndex();
  if( vatom[n].isSpecial() ){
    coords0 = vatom[n].getConstraintMatrix() * coords0;
  }

  return vatom[n].translateCoordinates(coords0);
}


bool DTAtoms::changeAtomCoordB( const double b )
{
  if( !isAtomSelected() ) return false;
  alignment = ANY;

  const Coordinates coords =
    Coordinates( 0.0, b - vatom[iatom].coords.b, 0.0 );

  Coordinates coords0 = vatom[iatom].getMatrixInv() * coords;

  const int n = vatom[iatom].getIndex();
  if( vatom[n].isSpecial() ){
    coords0 = vatom[n].getConstraintMatrix() * coords0;
  }

  return vatom[n].translateCoordinates(coords0);
}

bool DTAtoms::changeAtomCoordC( const double c )
{
  if( !isAtomSelected() ) return false;
  alignment = ANY;

  const Coordinates coords =
    Coordinates( 0.0, 0.0, c - vatom[iatom].coords.c );

  Coordinates coords0 = vatom[iatom].getMatrixInv() * coords;

  const int n = vatom[iatom].getIndex();
  if( vatom[n].isSpecial() ){
    coords0 = vatom[n].getConstraintMatrix() * coords0;
  }

  return vatom[n].translateCoordinates(coords0);
}

bool DTAtoms::changeAtomMotion( const int motion )
{
  if( !isAtomSelected() ) return false;
  alignment = ANY;

  const int n = vatom[iatom].getIndex();

  return vatom[n].changeMotion(motion);
}

bool DTAtoms::changeAtomVelocityX( const double vx )
{
  if( !isAtomSelected() ) return false;
  alignment = ANY;

  const int n = vatom[iatom].getIndex();
  vatom[n].velocity.x = vx;

  return true;
}

bool DTAtoms::changeAtomVelocityY( const double vy )
{
  if( !isAtomSelected() ) return false;
  alignment = ANY;

  const int n = vatom[iatom].getIndex();
  vatom[n].velocity.y = vy;

  return true;
}

bool DTAtoms::changeAtomVelocityZ( const double vz )
{
  if( !isAtomSelected() ) return false;
  alignment = ANY;

  const int n = vatom[iatom].getIndex();
  vatom[n].velocity.z = vz;

  return true;
}


bool DTAtoms::changeAtomElement( const QString& name )
{
  if( !isAtomSelected() ) return false;

  const int n = vatom[iatom].getIndex();

  int ppid = findPPID(name);
  if( ppid == -1 ){
    ppid = velement.size();
    velement.push_back(DTElement(name));
  }
  return vatom[n].changeElement(velement[ppid]);
}

bool DTAtoms::removeAtoms( void )
{
  if( !isAtomSelected() ) return false;

  alignment = ANY;

  vector<DTAtom> vatom_bak = vatom;

  vatom.clear();
  for( int n=0; n<(int)vatom_bak.size(); n++ ){

    bool found=false;
    for( int i=0; i<(int)viatom.size(); i++ ){
      const int iatom = viatom[i];
      const int index = vatom_bak[iatom].getIndex();

      if( n==index ){
	found = true;
	break;
      }
    }
    if( !found ){
      vatom.push_back(vatom_bak[n]);
    }
  }
  unselectAtom();

  return true;
}

bool DTAtoms::addAtoms( void )
{
  alignment = ANY;

  if( velement.empty() ){
    velement.push_back( DTElement("H") );
  }
  if( !isAtomSelected() ){
    if( vatom.empty() ){
      vatom.push_back
	(DTAtom(velement.front(),Coordinates(0.0,0.0,0.0),1));
    }else{
      vatom.push_back(vatom.front());
    }
    return true;
  }

  for( int i=0; i<(int)viatom.size(); i++ ){
    const int iatom = viatom[i];
    vatom.push_back( vatom[iatom] );
  }

  return true;
}


void DTAtoms::change_shown( void )
{
  if( velement.empty() ) return;
  if( velement_index_shown > (int)velement.size() ) velement_index_shown = 1;

  if( ElementSymbol::getAtomicNumber(velement_value_shown.name) == 0 ){
    return;
  }

  velement_value_shown = DTElement(velement_value_shown.name);

  const QString old_name = velement[velement_index_shown-1].name;
  const QString new_name = velement_value_shown.name;

  if( old_name != new_name ){
    velement[velement_index_shown-1] = velement_value_shown;

    for( int n=0; n<(int)vatom.size(); n++ ){
      if( vatom[n].element.name == old_name ){
	vatom[n].element = velement_value_shown;
      }
    }
  }
}

void DTAtoms::update_shown( void )
{
  if( velement.empty() ) return;
  if( velement_index_shown > (int)velement.size() ) velement_index_shown = 1;

  velement_value_shown  = velement[velement_index_shown-1];
}

int DTAtoms::size_shown( void )
{
  return (int)velement.size();
}

void DTAtoms::addElement( void )
{
  if( velement.empty() ){
    velement.push_back( DTElement("H") );
  }else{
    velement.push_back( velement.front() );
  }
  velement_index_shown = (int)velement.size();
  update_shown();
}

void DTAtoms::delElement( void )
{
  if( velement.size()<=1 ) return;
  if( velement_index_shown > (int)velement.size() ) velement_index_shown = 1;

  const QString new_name = velement_value_shown.name;

  for( int n=0; n<(int)vatom.size(); n++ ){
    if( vatom[n].element.name == new_name ){
      return;
    }
  }

  velement.erase( velement.begin()+velement_index_shown-1 );
  update_shown();
}

const DTElement* DTAtoms::findElement( const QString& name ) const
{
  for( int i=0; i<(int)velement.size(); i++ ){
    if( velement[i].name == name ) return &velement[i];
  }
  return NULL;
}

const DTElement* DTAtoms::findElement( const int number ) const
{
  for( int i=0; i<(int)velement.size(); i++ ){
    if( velement[i].number == number ) return &velement[i];
  }
  return NULL;
}

int DTAtoms::findPPID( const QString& name ) const
{
  for( int i=0; i<(int)velement.size(); i++ ){
    if( velement[i].name == name ) return i;
  }
  return -1;
}

int DTAtoms::findPPID( const int number ) const
{
  for( int i=0; i<(int)velement.size(); i++ ){
    if( velement[i].number == number ) return i;
  }
  return -1;
}
