/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

/*!
 \file dtband.cc
 \brief ohf[^̃NX
*/

#include <stdio.h>
#include "dtband.h"
#include "qtexception.h"
#include "glspline.h"
#include "position.h"
#include <math.h>
#include <algorithm>

DTBand::DTBand( void )
{
  clear();
}

void DTBand::clear( void )
{
  data[0].clear();
  data[1].clear();
  order = DEFAULT_ORDER;
}

double round( double& a, double& b )
{
  double d = a-b;
  double u = pow( 10.0, rint(log10(d))-1.0 );

  while( d/u >= 8.0 ){
    u *= 2.0;
  }

  b = floor(b/u)*u;
  a = ceil (a/u)*u;

  return u;
}

bool DTBand::load( const QString& fname )
{
  FILE* fptr = fopen( fname, "r" );
  if( fptr == NULL ){
    throw MyException("can not open a field file.",fname);
  }

  clear();

  const char* error_msg = NULL;
  char  buf[1024];
  int   offset;
  int   ivalue;
  char  svalue[32];

  int   nbk, nw;
  int   total_kpoints;
  int   total_bands;
  vector<QString> vlabel;
  vector<int>    vindex;
  bool  tapp3;
  double dummy;
  Position K, Kprev;
  double  dKlen;

  fgets(buf,sizeof(buf),fptr);
  if( 4 == sscanf(buf,"%d %d %d %lf", &nspin, &nbk, &nw, &dummy ) ){
    tapp3 = true; // old style format
  }
  else if( 3 == sscanf(buf,"%d %d %d", &nbk, &nw, &nspin ) ){
    tapp3 = false; // current format
  }
  else{
    error_msg = "Band file is broken in header section.";
    goto error_block;
  }

  vlabel.resize(nbk+1);
  vindex.resize(nbk+1);

  fgets(buf,sizeof(buf),fptr);
  {
    char* ptr = buf;
    for( int i=0; i<nbk+1; i++ ){
      if( 1 != sscanf( ptr, "%s%n", svalue, &offset ) ){
	error_msg = "Band file is broken in header section.";
	goto error_block;
      }
      vlabel[i] = QString(svalue);

      ptr += offset;
    }
  }

  fgets(buf,sizeof(buf),fptr);
  {
    char* ptr = buf;
    vindex.front() = 0;
    for( int i=0; i<nbk; i++ ){
      if( 1 != sscanf( ptr, "%d%n", &ivalue, &offset ) ){
	error_msg = "Band file is broken in header section.";
	goto error_block;
      }
      vindex[i+1] = vindex[i] + ivalue;

      ptr += offset;
    }
    total_kpoints = vindex.back()+1;
    total_bands   = nw;
  }

  data[0].resize(total_kpoints);
  if( nspin == 2 ){
    data[1].resize(total_kpoints);
  }
  for( int i=0; i<nbk+1; i++ ){
    data[0][ vindex[i] ].xlabel = vlabel[i];
  }

  // k-point coordinates
  data[0][0].x = 0.0;
  for( int i=0; i<nbk+1; i++ ){
    fgets(buf,sizeof(buf),fptr);
    if( 3 == sscanf(buf,"%lf %lf %lf", &K.x, &K.y, &K.z ) ){
      if( i==0 ){
      }
      else{
	dKlen = Position::length(K-Kprev)/(vindex[i]-vindex[i-1]);
	for( int k=vindex[i-1]+1; k<=vindex[i]; k++ ){
	  data[0][k].x = data[0][k-1].x + dKlen;
	}
      }
      Kprev = K;
    }

  }

  if( tapp3 ){
    for( int s=0; s<nspin; s++ ){
      fgets(buf,sizeof(buf),fptr); // skip spin line

      for( int k=0; k<total_kpoints; k++ ){
	data[s][k].resize(total_bands);

	for( int m=0; m<total_bands; m++ ){
	  double value;
	  if( 1 != fscanf( fptr, "%lf", &value )){
	    error_msg = "Band file is broken in data section.";
	    goto error_block;
	  }
	  else{
	    data[s][k].setValue(m,value);
	  }
	} // m
	fgets(buf,sizeof(buf),fptr); // read end-of-line
      } // k
    } // s
  }
  else{
    for( int k=0; k<total_kpoints; k++ ){
      for( int s=0; s<nspin; s++ ){
	data[s][k].resize(total_bands);

	for( int m=0; m<total_bands; m++ ){
	  double value;
	  if( 1 != fscanf( fptr, "%lf", &value )){
	    error_msg = "Band file is broken in data section.";
	    goto error_block;
	  }
	  else{
	    data[s][k].setValue(m,value);
	  }
	} // m
	fgets(buf,sizeof(buf),fptr); // read end-of-line
      } // s
    } // k
  }

  fclose(fptr);

  emin=emax=data[0].front().vy.front();

  for( int s=0; s<nspin; s++ ){
    for( int i=0; i<(int)data[0].size(); i++ ){
      for( int j=0; j<(int)data[0].front().size(); j++ ){
	if( emin > data[s][i].vy[j] ) emin = data[s][i].vy[j];
	if( emax < data[s][i].vy[j] ) emax = data[s][i].vy[j];
      }
    }
  }
  round(emax,emin);

  kmin = 0.0;
  kmax = data[0].back().x;

  update();

  return true;

 error_block:
  fclose(fptr);
  clear();
  throw MyException(error_msg,fname);
  return false; // dummy statement
}

bool DTBand::update( void )
{
  if( !isset() ) return false;

  switch( order ){
  case DEFAULT_ORDER :
    for( int s=0; s<nspin; s++ ){
      swapBandByDefaultOrder(data[s]);
    }
    break;
  case ASCENDING_ORDER :
    for( int s=0; s<nspin; s++ ){
      swapBandByAscendingOrder(data[s]);
    }
    break;
  case NATUAL_ORDER :
    for( int s=0; s<nspin; s++ ){
      swapBandByNatualOrder(data[s]);
    }
    break;
  }

  return true;
}


void DTBand::swapBandByDefaultOrder( vector<datum>& data )
{
  const int N = (int)data.front().size();

  for( int i=1; i<(int)data.size(); i++ ){
    for( int j=0; j<N; j++ ){
      data[i].vy[j] = data[i].vy_orig[j];
    }
  }
}

void DTBand::swapBandByAscendingOrder( vector<datum>& data )
{
  const int N = (int)data.front().size();
  for( int i=1; i<(int)data.size(); i++ ){
    for( int j=0; j<N; j++ ){
      data[i].vy[j] = data[i].vy_orig[j];
    }
    std::sort( data[i].vy.begin(), data[i].vy.end() );
  }
}

void DTBand::swapBandByNatualOrder( vector<datum>& data )
{
  const int N = (int)data.front().size();

  vector<double> df, ddf, fnext;

  df.resize(N);
  ddf.resize(N);
  fnext.resize(N);

  for( int i=1; i<(int)data.size(); i++ ){
    for( int j=0; j<N; j++ ){
      if( i==1 ){
	df [j] = 0.0;
	ddf[j] = 0.0;
      }
      else if( i==2 ){
	df [j] = data[i-1].vy[j] - data[i-2].vy[j];
	ddf[j] = 0.0;
      }
      else{
	df [j] = (data[i-1].vy[j] - data[i-3].vy[j])*0.5;
	ddf[j] = data[i-1].vy[j] - 2.0*data[i-2].vy[j] + data[i-3].vy[j];
      }
      fnext[j] = data[i-1].vy[j] + df[j] + 0.5*ddf[j];
    }

    for( int j1=0; j1<N; j1++ ){
      for( int j2=j1; j2<N; j2++ ){
	df[j2] = fabs(data[i].vy[j2] - fnext[j1]);
      }

      int index=j1;
      double min=df[j1];
      for( int j2=j1; j2<N; j2++ ){
	if( min > df[j2] ){
	  min = df[j2];
	  index = j2;
	}
      }
      // swap elements [index] and [j2]
      const double work = data[i].vy[j1];
      data[i].vy[j1] = data[i].vy[index];
      data[i].vy[index] = work;
    }
  }
}
