/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

/*!
 \file dtcell.cc
 \brief iq̒PʖEf[^̃NX
*/

#include <stdio.h>
#include <stdlib.h>
#include "dtcell.h"
#include "qtmisc.h"
#include "qtxml.h"

#ifndef M_SQRT3
const double M_SQRT3 = sqrt(3.0);
#endif

double DTCell::length_min=0.0, DTCell::length_max=1000.0;
double DTCell::E_min=-10.0, DTCell::E_max=+10.0;

const double facet_err = 1.0e-5;
//const double facet_err = 1.0e-8;

DTCell::DTCell( void )
{
  clear();
  loadKPathDefaults();
}

void DTCell::clear( void )
{
  shape = CUBIC;
  length = 10.0;
  Ea = Position(1.0,0.0,0.0);
  Eb = Position(0.0,1.0,0.0);
  Ec = Position(0.0,0.0,1.0);
  vectors_changed = true;
  shape_changed = false;
  alignment_changed = false;

  vplane.clear();
  vfacet.clear();
  vsymmL.clear();
  vsymmLselected.clear();
  vsymmLselected_loaded = false;
}

void DTCell::update( void )
{
  if( vectors_changed ){
    updateShape();
  }
  if( shape_changed ){
    updateVectors();
  }

  La = length*Ea;
  Lb = length*Eb;
  Lc = length*Ec;
  Lo = (La+Lb+Lc)*(-0.50);

  V  = La*(Lb%Lc);
  Ka = (Lb%Lc)*(1.0/V);
  Kb = (Lc%La)*(1.0/V);
  Kc = (La%Lb)*(1.0/V);

  if( vectors_changed || shape_changed || alignment_changed ){
    checkBravais();

    double V_conv  = La_conv*(Lb_conv%Lc_conv);
    Ka_conv = (Lb_conv%Lc_conv)*(1.0/V_conv);
    Kb_conv = (Lc_conv%La_conv)*(1.0/V_conv);
    Kc_conv = (La_conv%Lb_conv)*(1.0/V_conv);

    constructBrillouin();

    setupKPathDefault();
    if( !vsymmLselected_loaded ){
      clearKPathSelect();
      defaultKPathSelect();
    }
    vsymmLselected_loaded = false;
  }

  vectors_changed = false;
  shape_changed = false;
  alignment_changed = false;
}

void DTCell::updateVectors( void )
{
  switch( shape ){
  case CUBIC : {
    Ea = Position( 1.0, 0.0, 0.0 );
    Eb = Position( 0.0, 1.0, 0.0 );
    Ec = Position( 0.0, 0.0, 1.0 );
  } break;
  case TETRAGONAL_A : {
    Ea = Position( 2.0, 0.0, 0.0 );
    Eb = Position( 0.0, 1.0, 0.0 );
    Ec = Position( 0.0, 0.0, 1.0 );
  } break;
  case TETRAGONAL_B : {
    Ea = Position( 1.0, 0.0, 0.0 );
    Eb = Position( 0.0, 2.0, 0.0 );
    Ec = Position( 0.0, 0.0, 1.0 );
  } break;
  case TETRAGONAL_C : {
    Ea = Position( 1.0, 0.0, 0.0 );
    Eb = Position( 0.0, 1.0, 0.0 );
    Ec = Position( 0.0, 0.0, 2.0 );
  } break;
  case ORTHORHOMBIC : {
    Ea = Position( 3.0, 0.0, 0.0 );
    Eb = Position( 0.0, 2.0, 0.0 );
    Ec = Position( 0.0, 0.0, 1.0 );
  } break;
  case MONOCLINIC_A : {
    Ea = Position( 0.5, 0.5, 0.0 );
    Eb = Position( 0.0, 1.0, 0.0 );
    Ec = Position( 0.0, 0.0, 1.0 );
  } break;
  case MONOCLINIC_B : {
    Ea = Position( 1.0, 0.0, 0.0 );
    Eb = Position( 0.0, 0.5, 0.5 );
    Ec = Position( 0.0, 0.0, 1.0 );
  } break;
  case MONOCLINIC_C : {
    Ea = Position( 1.0, 0.0, 0.0 );
    Eb = Position( 0.0, 1.0, 0.0 );
    Ec = Position( 0.5, 0.0, 0.5 );
  } break;
  case TRIGONAL : {
    Ea = Position( 0.5, 0.5, 0.0 );
    Eb = Position( 0.0, 0.5, 0.5 );
    Ec = Position( 0.5, 0.0, 0.5 );
  } break;
  case TRICLINIC : {
    Ea = Position( 1.0, 0.0, 0.0 );
    Eb = Position( 0.0, 1.0, 0.0 );
    Ec = Position( 0.0, 0.0, 1.0 );
  } break;    
  case HEXAGONAL_A : {
    Ea = Position( 1.0, 0.0, 0.0 );
    Eb = Position( 0.0, +M_SQRT3*0.5, 0.5 );
    Ec = Position( 0.0, -M_SQRT3*0.5, 0.5 );
  } break;
  case HEXAGONAL_B : {
    Ea = Position( 0.5, 0.0, -M_SQRT3*0.5 );
    Eb = Position( 0.0, 1.0, 0.0 );
    Ec = Position( 0.5, 0.0, +M_SQRT3*0.5 );
  } break;
  case HEXAGONAL_C : {
    Ea = Position( +M_SQRT3*0.5, 0.5, 0.0 );
    Eb = Position( -M_SQRT3*0.5, 0.5, 0.0 );
    Ec = Position( 0.0, 0.0, 1.0 );
  } break;
  }
}

inline bool match( const double a, const double b )
{
  const double err = 1.0e-8;
  //  const double err = 1.0e-5;
  return fabs(a-b)<err;
}

void DTCell::updateShape( void )
{
  const double Da = Position::length(Ea);
  const double Db = Position::length(Eb);
  const double Dc = Position::length(Ec);
  const double Aa = Position::angle(Eb,Ec);
  const double Ab = Position::angle(Ec,Ea);
  const double Ac = Position::angle(Ea,Eb);

  if( false ){
  }
  else if( match(Da,Db) && match(Db,Dc) && match(Dc,Da) &&
	   match(Ac,90.0) && match(Aa,90.0) && match(Ab,90.0) ){
    shape = CUBIC;
  }
  else if( match(Db,Dc) &&
	   match(Ac,90.0) && match(Aa,90.0) && match(Ab,90.0) ){
    shape = TETRAGONAL_A;
  }
  else if( match(Dc,Da) &&
	   match(Ac,90.0) && match(Aa,90.0) && match(Ab,90.0) ){
    shape = TETRAGONAL_B;
  }
  else if( match(Da,Db) &&
	   match(Ac,90.0) && match(Aa,90.0) && match(Ab,90.0) ){
    shape = TETRAGONAL_C;
  }
  else if( !match(Da,Db) && !match(Db,Dc) && !match(Dc,Da) &&
	   match(Ac,90.0) && match(Aa,90.0) && match(Ab,90.0) ){
    shape = ORTHORHOMBIC;
  }
  else if( match(Db,Dc) &&
	   match(Aa,120.0) && match(Ab,90.0) && match(Ac,90.0) ){
    shape = HEXAGONAL_A;
  }
  else if( match(Dc,Da) &&
	   match(Aa,90.0) && match(Ab,120.0) && match(Ac,90.0) ){
    shape = HEXAGONAL_B;
  }
  else if( match(Da,Db) &&
	   match(Aa,90.0) && match(Ab,90.0) && match(Ac,120.0) ){
    shape = HEXAGONAL_C;
  }
  else if( match(Ab,90.0) && match(Ac,90.0) ){
    shape = MONOCLINIC_A;
  }
  else if( match(Ac,90.0) && match(Aa,90.0) ){
    shape = MONOCLINIC_B;
  }
  else if( match(Aa,90.0) && match(Ab,90.0) ){
    shape = MONOCLINIC_C;
  }
  else if( match(Da,Db) && match(Db,Dc) && match(Dc,Da) &&
	   match(Ac,Aa) && match(Aa,Ab) ){
    shape = TRIGONAL;
  }
  else{
    shape = TRICLINIC;
  }
}



Position DTCell::getPosition( const Coordinates& q ) const
{
  return q.a*La + q.b*Lb + q.c*Lc + Lo;
}

Position DTCell::getPositionLocal( const Coordinates& q ) const
{
  return q.a*La + q.b*Lb + q.c*Lc;
}

Coordinates DTCell::getCoordinates( const Position& p ) const
{
  return Coordinates( (p-Lo)*Ka, (p-Lo)*Kb, (p-Lo)*Kc );
}

Coordinates DTCell::getCoordinatesLocal( const Position& p ) const
{
  return Coordinates( p*Ka, p*Kb, p*Kc );
}

Coordinates DTCell::getCoordinatesNormalized( const Position& p ) const
{
  
  return Coordinates::normalize( Coordinates( (p-Lo)*Ka, (p-Lo)*Kb, (p-Lo)*Kc ) );
}

Coordinates DTCell::getCoordinatesLocalNormalized( const Position& p ) const
{
  return Coordinates::normalize( Coordinates( p*Ka, p*Kb, p*Kc ) );
}


void DTCell::checkBravais( void )
{
  bravais.center = "";
  double angle_a=0.0, angle_b=0.0, angle_c=0.0;

  const double eps=1.0e-5;

  if( bravais.center == "" ){
      La_conv = La;
      Lb_conv = Lb;
      Lc_conv = Lc;

    angle_a = Position::angle( Lb_conv, Lc_conv );
    angle_b = Position::angle( Lc_conv, La_conv );
    angle_c = Position::angle( La_conv, Lb_conv );

    if( fabs(angle_a-90.00)<eps &&
	fabs(angle_b-90.0)<eps &&
	fabs(angle_c-90.0)<eps ){
      // simple
      bravais.center = "simple";
    }
    else if( fabs(angle_a-90.00)<eps &&
	fabs(angle_b-90.0)<eps &&
	fabs(angle_c-120.0)<eps ){
      // simple
      bravais.center = "simple";
    }
  }

  if( bravais.center == "" ){
    La_conv = - La + Lb + Lc;
    Lb_conv = + La - Lb + Lc;
    Lc_conv = + La + Lb - Lc;

    angle_a = Position::angle( Lb_conv, Lc_conv );
    angle_b = Position::angle( Lc_conv, La_conv );
    angle_c = Position::angle( La_conv, Lb_conv );

    if( fabs(angle_a-90.00)<eps &&
	fabs(angle_b-90.0)<eps &&
	fabs(angle_c-90.0)<eps ){
      // face center
      bravais.center = "face";
    }
  }

  if( bravais.center == "" ){
    La_conv = + Lb + Lc;
    Lb_conv = + Lc + La;
    Lc_conv = + La + Lb;

    angle_a = Position::angle( Lb_conv, Lc_conv );
    angle_b = Position::angle( Lc_conv, La_conv );
    angle_c = Position::angle( La_conv, Lb_conv );

    if( fabs(angle_a-90.00)<eps &&
	fabs(angle_b-90.0)<eps &&
	fabs(angle_c-90.0)<eps ){
      // body center
      bravais.center = "body";
    }
  }

  if( bravais.center == "" ){
    La_conv = + La - Lb;
    Lb_conv = + La + Lb;
    Lc_conv = + Lc;

    angle_a = Position::angle( Lb_conv, Lc_conv );
    angle_b = Position::angle( Lc_conv, La_conv );
    angle_c = Position::angle( La_conv, Lb_conv );

    if( fabs(angle_a-90.00)<eps &&
	fabs(angle_c-90.0)<eps ){
      // base center
      bravais.center = "base";
    }
  }

  if( bravais.center == "" ){
    La_conv = Lb - Lc;
    Lb_conv = Lc - La;
    Lc_conv = La + Lb + Lc;

    angle_a = Position::angle( Lb_conv, Lc_conv );
    angle_b = Position::angle( Lc_conv, La_conv );
    angle_c = Position::angle( La_conv, Lb_conv );

    if( fabs(angle_a-90.00)<eps &&
	fabs(angle_b-90.0)<eps &&
	fabs(angle_c-120.0)<eps ){
      // treat this lattice as trigonal.
      bravais.shape  = "trigonal";
      bravais.center = "simple";
      /*
      // treat this lattice as hexagonal.
      // it does not work.
      bravais.shape  = "hexagonal";
      bravais.center = "simple";
      */
    }
  }

  if( bravais.center == "" ){
    La_conv = La;
    Lb_conv = Lb;
    Lc_conv = Lc;

    angle_a = Position::angle( Lb_conv, Lc_conv );
    angle_b = Position::angle( Lc_conv, La_conv );
    angle_c = Position::angle( La_conv, Lb_conv );

    bravais.center = "simple";
  }

  double length_a, length_b, length_c;
  length_a = Position::length( La_conv );
  length_b = Position::length( Lb_conv );
  length_c = Position::length( Lc_conv );

  if( fabs(angle_a-90.0)<eps &&
      fabs(angle_b-90.0)<eps &&
      fabs(angle_c-90.0)<eps ){
    // cubic, tetragonal, orthorhombic
    if( fabs(length_a-length_b)<eps && fabs(length_b-length_c)<eps ){
      bravais.shape = "cubic";
      bravais.subtype = "none";
    }
    else if( fabs(length_a-length_b)<eps && bravais.center != "face" ){
      bravais.shape = "tetragonal";
      if( bravais.center == "simple" ){
	bravais.subtype = "none";
      }
      else if( bravais.center == "body" ){
	if( length_a < length_c ){
	  bravais.subtype = "type1";
	}
	else{
	  bravais.subtype = "type2";
	}
      }
    }
    else{
      bravais.shape = "orthorhombic";
      if( bravais.center == "simple" ){
	bravais.subtype = "none";
      }
      else if( bravais.center == "face" ){
	const double ka2 = 1.0/(length_a*length_a);
	const double kb2 = 1.0/(length_b*length_b);
	const double kc2 = 1.0/(length_c*length_c);
	if( ka2 < kb2 + kc2 &&
	    kb2 < ka2 + kc2 &&
	    kc2 < ka2 + kb2 ){
	  bravais.subtype = "type1";
	}
	else if( ka2 > kb2 + kc2 ){
	  bravais.subtype = "type2";
	}
	else if( kb2 > ka2 + kc2 ){
	  bravais.subtype = "type3";
	}
	else if( kc2 > ka2 + kb2 ){
	  bravais.subtype = "type4";
	}
      }
      else if( bravais.center == "body" ){
	if( false );
	else if( length_a>length_c &&
		 length_a>length_b ){
	  bravais.subtype = "type1";
	}
	else if( length_b>length_a &&
		 length_b>length_c ){
	  bravais.subtype = "type2";
	}
	else if( length_c>length_a &&
		 length_c>length_b ){
	  bravais.subtype = "type3";
	}
      }
      else if( bravais.center == "base" ){
	// suppose C base
	if( length_a<length_b ){
	  bravais.subtype = "type1";
	}
	else{
	  bravais.subtype = "type2";
	}
      }
    }
  }
  else if( fabs(angle_a-90.0)<eps &&
	fabs(angle_b-90.0)<eps &&
	fabs(angle_c-120.0)<eps ){

    if( bravais.shape == "trigonal" ){
     if( length_a*M_SQRT3<length_c*M_SQRT2 ){
	bravais.subtype = "type1";
      }
      else{
	bravais.subtype = "type2";
      }
    }
    else{
      // hexagonal
      if( fabs(length_a-length_b)<eps ){
	bravais.shape = "hexagonal";
	bravais.subtype = "none";
      }
      else{
	bravais.shape = "triclinic";
	bravais.subtype = "none";
      }
    }
  }
  else if( fabs(angle_a-90.0)<eps &&
      fabs(angle_c-90.0)<eps ){
    bravais.shape = "monoclinic";
    if( bravais.center == "simple" ){
      bravais.subtype = "none";
    }
    else if( bravais.center == "base" ){ // AB plane is the base
      if( length_a<length_b ){
	bravais.subtype = "type1";
      }
      else{
	bravais.subtype = "type2";
      }
    }
  }
  else if( fabs(angle_a-angle_b)<eps && fabs(angle_b-angle_c)<eps ){
    // trigonal
    if( fabs(length_a-length_b)<eps && fabs(length_b-length_c)<eps ){
      bravais.shape = "trigonal";
      if( angle_a<90.0 ){
	bravais.subtype = "type1";
      }
      else{
	bravais.subtype = "type2";
      }
    }
    else{
      bravais.shape = "triclinic";
    }
  }
  else{
    bravais.shape = "triclinic";
  }
}





BrillouinPlane::BrillouinPlane( void )
{
}

BrillouinPlane::BrillouinPlane( const Position& normal )
{
  this->normal = normal;
  this->distance = (-1.0)*Position::inner_product(normal,normal);
}

double BrillouinPlane::equation( const Position& position ) const
{
  return Position::inner_product(normal,position) + distance;
}




BrillouinSegment::BrillouinSegment( void )
{
}

BrillouinSegment::BrillouinSegment
( const QString& labelL,
  const QString& labelPs,
  const double  xs,
  const double  ys,
  const double  zs,
  const QString& labelPe,
  const double  xe,
  const double  ye,
  const double  ze )
{
  this->labelL = labelL;
  this->labelPs = labelPs;
  this->positions = Position( xs, ys, zs );
  this->labelPe = labelPe;
  this->positione = Position( xe, ye, ze );
  this->npoint = 10;
}

BrillouinSegment::BrillouinSegment
( const Position& positions,
  const Position& positione )
{
  this->positions = positions;
  this->positione = positione;
}

BrillouinSegment BrillouinSegment::reverse( void ) const
{
  return BrillouinSegment
    ( labelL,
      labelPe, positione.x, positione.y, positione.z,
      labelPs, positions.x, positions.y, positions.z );
}

bool BrillouinSegment::match
( const BrillouinSegment& segment1, const BrillouinSegment& segment2 )
{
  return
    ( Position::match(segment1.positions,segment2.positions,facet_err) &&
      Position::match(segment1.positione,segment2.positione,facet_err) ) ||
    ( Position::match(segment1.positions,segment2.positione,facet_err) &&
      Position::match(segment1.positione,segment2.positions,facet_err) );
}

bool BrillouinSegment::find
( const vector<BrillouinSegment>& vs, const BrillouinSegment& s )
{
  for( int n=0; n<(int)vs.size(); n++ ){
    if( match(vs[n],s) ) return true;
  }
  return false;
}

bool BrillouinSegment::cut( const BrillouinPlane& plane )
{
  const double vs = plane.equation(positions);
  const double ve = plane.equation(positione);

  const double eps = facet_err;

  // the both point is on the plane
  if( fabs(vs)<=eps && fabs(ve)<=eps ){
    cutted = false;//ADD
    return true;
  }

  // the end point is on the plane, the segment is inside
  if( vs< 0.0 && fabs(ve)<=eps ){
    positionc = positione;
    cutted = true;//ADD
    return true;
  }

  // the end point is on the plane, the segment is outside
  if( vs> 0.0 && fabs(ve)<=eps ){
    positionc = positione;
    cutted = true;//ADD
    return false;
  }

  // the start point is on the plane, the segment is outside
  if( fabs(vs)<=eps && ve>0.0 ){
    positionc = positions;
    cutted = true;//ADD
    return false;
  }

  // the start point is on the plane, the segment is inside
  if( fabs(vs)<=eps && ve<0.0 ){
    positionc = positions;
    cutted = true;//ADD
    return true;
  }

  // the segment is inside the plane
  if( vs< 0.0 && ve<0.0 ){
    cutted = false;//ADD
    return true;
  }

  // the segment crosses the plane. vs outside, ve inside.
  if( vs> 0.0 && ve<0.0 ){
    const double t  = (vs)/(vs-ve);
    for( int i=0; i<3; i++ ){
      positionc[i] = positione[i]*t + positions[i]*(1.0-t);
    }
    positions = positionc;
    cutted = true;//ADD
    return true;
  }

  // the segment crosses the plane.
  if( vs< 0.0 && ve>0.0 ){
    const double t = (ve)/(ve-vs);
    for( int i=0; i<3; i++ ){
      positionc[i] = positions[i]*t + positione[i]*(1.0-t);
    }
    positione = positionc;
    cutted = true;//ADD
    return true;
  }

  // the segment is outside the plane
  if( vs> 0.0 && ve>0.0 ){
    cutted = false;//ADD
    return false;
  }

  fprintf(stderr,"# Abnormal error: BrillouinSegment::cut.\n");
  exit(0);
  return false;
}



BrillouinFacet::BrillouinFacet
( const Position& position0,
  const Position& position1,
  const Position& position2,
  const Position& position3 )
{
  vsegment.push_back( BrillouinSegment(position0,position1) );
  vsegment.push_back( BrillouinSegment(position1,position2) );
  vsegment.push_back( BrillouinSegment(position2,position3) );
  vsegment.push_back( BrillouinSegment(position3,position0) );
}

BrillouinFacet::BrillouinFacet( const vector<BrillouinSegment>& vs )
{
  vsegment = vs;
  order();
}

bool BrillouinFacet::cut( const BrillouinPlane& plane )
{
  int cutted_size = 0;

  for( int s=0; s<(int)vsegment.size(); s++ ){
    bool inside = vsegment[s].cut(plane);

    if( vsegment[s].cutted ){
      if( cutted_size == 0 ){
	segmentc.positions = vsegment[s].positionc;
	cutted_size++;
      }
      else if( cutted_size == 1 ){
	if( !Position::match(segmentc.positions,vsegment[s].positionc,facet_err) ){
	  segmentc.positione = vsegment[s].positionc;
	  cutted_size++;
	}
      }
    }
    if( !inside ){
      vsegment.erase( vsegment.begin()+s-- );
    }
  }

  if( cutted_size == 2 ){
    vsegment.push_back(segmentc);
    cutted = true;
  }
  else{
    cutted = false;
  }

  if( vsegment.size()<3 ){
    return false;
  }

  if( cutted ){
    order();
  }

  return true;
}

void BrillouinFacet::order( void )
{
  for( int s=0; s<(int)vsegment.size(); s++ ){
    for( int t=s+1; t<(int)vsegment.size(); t++ ){
      if( BrillouinSegment::match(vsegment[s],vsegment[t]) ){
	vsegment.erase( vsegment.begin()+t-- );
      }
    }
  }

  for( int s=0; s<(int)vsegment.size(); s++ ){
    if( Position::match(vsegment[s].positions,vsegment[s].positione) ){
      fprintf(stderr,"# Abnormal error: BrillouinFacet::order. s==e.\n");

    for( int s=0; s<(int)vsegment.size(); s++ ){
      fprintf(stderr,
	      "segment from %+f,%+f,%+f to %+f,%+f,%+f\n",
	      vsegment[s].positions[0],
	      vsegment[s].positions[1],
	      vsegment[s].positions[2],
	      vsegment[s].positione[0],
	      vsegment[s].positione[1],
	      vsegment[s].positione[2] );
    }

      exit(0);
    }
  }

  for( int s=0; s<(int)vsegment.size(); s++ ){
    for( int t=s+1; t<(int)vsegment.size(); t++ ){
      if( Position::match(vsegment[s].positione,vsegment[t].positions,facet_err) ){
	if(s+1!=t) std::swap(vsegment[s+1],vsegment[t]);
	break;
      }
      if( Position::match(vsegment[s].positione,vsegment[t].positione,facet_err) ){
	Position positionw;
	positionw = vsegment[t].positione;
	vsegment[t].positione = vsegment[t].positions;
	vsegment[t].positions = positionw;

	if(s+1!=t){
	  std::swap(vsegment[s+1],vsegment[t]);
	}
	break;
      }
    }
  }

  if( !Position::match(vsegment.front().positions,vsegment.back().positione,facet_err) ){
    fprintf(stderr,"# Abnormal error: BrillouinFacet::order. not loop.\n");
    for( int s=0; s<(int)vsegment.size(); s++ ){
      fprintf(stderr,
	      "segment from %+f,%+f,%+f to %+f,%+f,%+f\n",
	      vsegment[s].positions[0],
	      vsegment[s].positions[1],
	      vsegment[s].positions[2],
	      vsegment[s].positione[0],
	      vsegment[s].positione[1],
	      vsegment[s].positione[2] );
      exit(0);
    }
  }
}




void DTCell::constructBrillouin( void )
{
  // initial trial points, which make a large box, large enough to involve the zone
  //
  //      6------7
  //     /|     /|
  //    4-+----5 |
  //    | |    | |
  //    | 2----+-3
  //    |/     |/
  //    0------1
  //

  Position position[8];

  position[0] = -Ka-Kb-Kc;
  position[1] = +Ka-Kb-Kc;
  position[2] = -Ka+Kb-Kc;
  position[3] = +Ka+Kb-Kc;
  position[4] = -Ka-Kb+Kc;
  position[5] = +Ka-Kb+Kc;
  position[6] = -Ka+Kb+Kc;
  position[7] = +Ka+Kb+Kc;

  vfacet.clear();
  vfacet.push_back( BrillouinFacet(position[0],position[1],position[3],position[2]) );
  vfacet.push_back( BrillouinFacet(position[2],position[3],position[7],position[6]) );
  vfacet.push_back( BrillouinFacet(position[6],position[7],position[5],position[4]) );
  vfacet.push_back( BrillouinFacet(position[4],position[5],position[1],position[0]) );
  vfacet.push_back( BrillouinFacet(position[1],position[5],position[7],position[3]) );
  vfacet.push_back( BrillouinFacet(position[2],position[6],position[4],position[0]) );

  // wigner ceitz planes
  vplane.clear();
  for( int kx=-1; kx<=+1; kx++ ){
    for( int ky=-1; ky<=+1; ky++ ){
      for( int kz=-1; kz<=+1; kz++ ){
	if( kx==0 && ky==0 && kz==0 ) continue;
	Position normal = Ka*(kx*0.5) + Kb*(ky*0.5) + Kc*(kz*0.5);
	vplane.push_back(BrillouinPlane(normal));
      }
    }
  }

  // cut zone
  for( int p=0; p<(int)vplane.size(); p++ ){
    vector<BrillouinSegment> vsegmentc;

    for( int f=0; f<(int)vfacet.size(); f++ ){
      if( !vfacet[f].cut( vplane[p] ) ){
	vfacet.erase( vfacet.begin()+f-- );
	continue;
      }
      if( vfacet[f].cutted ){
	if( !BrillouinSegment::find( vsegmentc, vfacet[f].segmentc ) ){
	  vsegmentc.push_back(vfacet[f].segmentc);
	}
      }
    }
    if( vsegmentc.size()>=3 ){
      vfacet.push_back(vsegmentc);
    }
  }
}



void DTCell::loadKPathDefaults( void )
{
  kpath_defaults.clear();

  QDomDocument doc;
  QDomElement group, node;

  if( !XML::load( doc, "kpath.xml") ){
    fprintf(stderr,"kpath not found\n");
    return;
  }

  QDomNode root = XML::getRoot(doc);

  //---- <kpath>
  vector<QDomElement> vnode_kpath = XML::getElementsByTagName(root,"kpath");

  char shape[32], center[32], subtype[32];
  char labelL[32], labelPs[32], labelPe[32];
  Position Ks, Ke;

  for( int i=0; i<(int)vnode_kpath.size(); i++ ){
    if( 3 != XML::sscanf( XML::getAttribute(vnode_kpath[i],"bravais"), "%s %s %s",
			  shape, center, subtype ) ){
      fprintf(stderr,"<kpath bravais> not found\n");
      continue;
    }

    kpath_defaults.push_back(KPathDefaults());

    kpath_defaults.back().shape   = QString(shape);
    kpath_defaults.back().center  = QString(center);
    kpath_defaults.back().subtype = QString(subtype);

    vector<QDomElement> vnode_segment = XML::getElementsByTagName(vnode_kpath[i],"segment");
    for( int j=0; j<(int)vnode_segment.size(); j++ ){
      if( 1 != XML::sscanf( XML::getAttribute(vnode_segment[j],"name"), "%s",
			    labelL ) ){
	fprintf(stderr,"<segment name> not found\n");
	continue;
      }
      vector<QDomElement> vnode_point = XML::getElementsByTagName(vnode_segment[j],"point");

      if( (int)vnode_point.size() != 2 ){
	fprintf(stderr,"<point> not found\n");
	continue;
      }

      if( 1 != XML::sscanf( XML::getAttribute(vnode_point[0],"name"), "%s",
			    labelPs ) ){
	fprintf(stderr,"<point name> not found\n");
	continue;
      }
      if( 1 != XML::sscanf( XML::getAttribute(vnode_point[1],"name"), "%s",
			    labelPe ) ){
	fprintf(stderr,"<point name> not found\n");
	continue;
      }

      if( 3 != XML::sscanf( XML::getChildValue(vnode_point[0]), "%lf %lf %lf",
			    &Ks.x, &Ks.y, &Ks.z ) ){
	fprintf(stderr,"<point> broken\n");
	continue;
      }
      if( 3 != XML::sscanf( XML::getChildValue(vnode_point[1]), "%lf %lf %lf",
			    &Ke.x, &Ke.y, &Ke.z ) ){
	fprintf(stderr,"<point> broken\n");
	continue;
      }

      kpath_defaults.back().kpath.push_back
	( BrillouinSegment( labelL,
			    labelPs,Ks.x,Ks.y,Ks.z,
			    labelPe,Ke.x,Ke.y,Ke.z));
    }
  }
}

void DTCell::setupKPathDefault( void )
{
  vsymmL.clear();

  bool found=false;
  for( int i=0; i<(int)kpath_defaults.size(); i++ ){
    if( kpath_defaults[i].shape  != bravais.shape ) continue;
    if( kpath_defaults[i].center != bravais.center ) continue;
    if( kpath_defaults[i].subtype != bravais.subtype ) continue;

    vsymmL = kpath_defaults[i].kpath;
    found=true;
    break;
  }
  if( !found ){
    fprintf(stderr,"Error: unknown cell type. %s %s %s\n",
	    qPrintable(bravais.shape), qPrintable(bravais.center),
	    qPrintable(bravais.subtype) );
  }

  // cut symmetric lines
  for( int s=0; s<(int)vsymmL.size(); s++ ){
    Position ks =
      + Ka_conv * vsymmL[s].positions.a
      + Kb_conv * vsymmL[s].positions.b 
      + Kc_conv * vsymmL[s].positions.c;
    Position ke =
      + Ka_conv * vsymmL[s].positione.a
      + Kb_conv * vsymmL[s].positione.b 
      + Kc_conv * vsymmL[s].positione.c;

    BrillouinSegment segment = BrillouinSegment(ks,ke);
    // begin modified 2014/11/11
    bool cutted = false;
    for( int p=0; p<(int)vplane.size(); p++ ){
      if( !segment.cut(vplane[p]) ){
	cutted = true;
	break;
      }
    }
    if( cutted ){
      vsymmL.erase( vsymmL.begin()+s );
      s--;
    }
    else{
      vsymmL[s].positions = Position
	( Position::inner_product(La_conv,segment.positions),
	  Position::inner_product(Lb_conv,segment.positions),
	  Position::inner_product(Lc_conv,segment.positions) );
      vsymmL[s].positione = Position
	( Position::inner_product(La_conv,segment.positione),
	  Position::inner_product(Lb_conv,segment.positione),
	  Position::inner_product(Lc_conv,segment.positione) );
    }
    // end modified 2014/11/11


  }

  // change base to conv to prim
  for( int s=0; s<(int)vsymmL.size(); s++ ){
    {
      Position k =
	+ Ka_conv * vsymmL[s].positions.a
	+ Kb_conv * vsymmL[s].positions.b 
	+ Kc_conv * vsymmL[s].positions.c;

      vsymmL[s].positions = Position
	( Position::inner_product(La,k),
	  Position::inner_product(Lb,k),
	  Position::inner_product(Lc,k) );
    }
    {
      Position k =
	+ Ka_conv * vsymmL[s].positione.a
	+ Kb_conv * vsymmL[s].positione.b 
	+ Kc_conv * vsymmL[s].positione.c;

      vsymmL[s].positione = Position
	( Position::inner_product(La,k),
	  Position::inner_product(Lb,k),
	  Position::inner_product(Lc,k) );
    }
    vsymmL[s].npoint = 10;
  }
}


void DTCell::clearKPathSelect( void )
{
  vsymmLselected.clear();
}

void DTCell::defaultKPathSelect( void )
{
  //  if( vsymmL.empty() ) return;
  vsymmLselected.clear();

  // copy connecting vsymmL segments to vsymmLselected
  for( int s=0; s<(int)vsymmL.size(); s++ ){
    if( s>0 && vsymmL[s-1].labelPe != vsymmL[s-0].labelPs ) break;
    vsymmLselected.push_back( vsymmL[s] );
  }
}




bool DTCell::selectBrillouin( const int select )
{
  if( 0<=select && select<(int)vsymmL.size() ){
    if( vsymmLselected.empty() ){
      vsymmLselected.push_back(vsymmL[select]);
      return true;
    }

    bool match = false;
    for( int i=0; i<(int)vsymmLselected.size(); i++ ){
      const BrillouinSegment& symmL = vsymmLselected[i];

      if( vsymmL[select].labelPs == symmL.labelPs &&
	  vsymmL[select].labelPe == symmL.labelPe ){
	match = true;
	break;
      }
      if( vsymmL[select].labelPe == symmL.labelPs &&
	  vsymmL[select].labelPs == symmL.labelPe ){
	match = true;
	break;
      }
    }

    if( match ){
      if( vsymmL[select].labelPe == vsymmLselected.back().labelPe &&
	  vsymmL[select].labelPs == vsymmLselected.back().labelPs ){
	vsymmLselected.pop_back();
	return true;
      }
      if( vsymmL[select].labelPs == vsymmLselected.back().labelPe &&
	  vsymmL[select].labelPe == vsymmLselected.back().labelPs ){
	vsymmLselected.pop_back();
	return true;
      }
      if( vsymmL[select].labelPe == vsymmLselected.front().labelPs &&
	  vsymmL[select].labelPs == vsymmLselected.front().labelPe ){
	vsymmLselected.pop_front();
	return true;
      }
      if( vsymmL[select].labelPs == vsymmLselected.front().labelPs &&
	  vsymmL[select].labelPe == vsymmLselected.front().labelPe ){
	vsymmLselected.pop_front();
	return true;
      }
    }
    else{
      if( vsymmL[select].labelPs == vsymmLselected.back().labelPe &&
	  vsymmL[select].labelPe != vsymmLselected.back().labelPs ){
	vsymmLselected.push_back(vsymmL[select]);
	return true;
      }
      if( vsymmL[select].labelPe == vsymmLselected.back().labelPe &&
	  vsymmL[select].labelPs != vsymmLselected.back().labelPs ){
	vsymmLselected.push_back(vsymmL[select].reverse());
	return true;
      }
      if( vsymmL[select].labelPe == vsymmLselected.front().labelPs &&
	  vsymmL[select].labelPs != vsymmLselected.front().labelPe ){
	vsymmLselected.push_front(vsymmL[select]);
	return true;
      }
      if( vsymmL[select].labelPs == vsymmLselected.front().labelPs &&
	  vsymmL[select].labelPe != vsymmLselected.front().labelPe ){
	vsymmLselected.push_front(vsymmL[select].reverse());
	return true;
      }
    }
  }


  else if( (int)vsymmL.size()<=select && select<(int)vsymmL.size()+(int)vsymmLselected.size() ){
    if( (int)vsymmL.size()==select ){
      vsymmLselected.pop_front();
      return true;
    }
    if( select==(int)vsymmL.size()+(int)vsymmLselected.size() ){
      vsymmLselected.pop_back();
      return true;
    }
  }
  else{
    return false;
  }
  return false;
}
