/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

#include <stdio.h>
#include <string.h>
#include <math.h>

#include "dtcif.h"
#include "qtmisc.h"
#include <algorithm>

template <typename T> bool find( const vector<T>& vec, const T& elem )
{
  return std::find( vec.begin(), vec.end(), elem ) != vec.end();
}


vector<QString> DTCIF::parseLine( const char* line )
{
  const char* ptr = line;

  vector<QString> varg;
  char str[64];

  int len;
  while( true ){
    if( 1 == sscanf(ptr," \'%[^\']\'%n", str, &len ) ){
      ptr += len;
      varg.push_back(QString(str));
      continue;
    }
    else if( 1 == sscanf(ptr,"%s%n", str, &len ) ){
      ptr += len;
      varg.push_back(QString(str));
      continue;
    }
    else{
      break;
    }
  }
  return varg;
}


bool DTCIF::load( const QString& fname )
{
  int     symmetry_Int_Tables_number = 0;
  QString symmetry_space_group_name = "";
  QString space_group_crystal_system = "";


  char   buf[1024];
  char   svalue[1024];
  int    ivalue;
  double dvalue;

  FILE* fptr = fopen( fname, "r" );

  if( !fptr ) return false;

  while( fgets(buf,sizeof(buf),fptr) ){
    if( false );
    else if( 1 == sscanf( buf, "_symmetry_Int_Tables_number %d",
			  &ivalue ) ){
      symmetry_Int_Tables_number = ivalue;
    }
    else if( 1 == sscanf( buf, "_space_group_crystal_system \'%[^\']",
			  svalue ) ){
      space_group_crystal_system = QString(svalue);
    }
    else if( 1 == sscanf( buf, "_space_group_crystal_system %s",
			  svalue ) ){
      space_group_crystal_system = QString(svalue);
    }

    else if( 1 == sscanf( buf, "_symmetry_space_group_name_H-M \'%[^\']",
			  svalue ) ){
      char *ptr1, *ptr2;
      for( ptr1=svalue, ptr2=svalue; *ptr2; ptr2++ ){
	if( *ptr2 != ' ' ) *ptr1++ = *ptr2;
      }
      *ptr1='\0';
      symmetry_space_group_name = QString(svalue);
    }
    else if( 1 == sscanf( buf, "_symmetry_space_group_name_H-M \"%[^\"]",
			  svalue ) ){
      symmetry_space_group_name = QString(svalue);
    }

    else if( 1 == sscanf( buf, "_cell_length_a %lf",
			  &dvalue ) ){
      cell.length_a = dvalue;
    }
    else if( 1 == sscanf( buf, "_cell_length_b %lf",
			  &dvalue ) ){
      cell.length_b = dvalue;
    }
    else if( 1 == sscanf( buf, "_cell_length_c %lf",
			  &dvalue ) ){
      cell.length_c = dvalue;
    }
    else if( 1 == sscanf( buf, "_cell_angle_alpha %lf",
			  &dvalue ) ){
      cell.angle_alpha = dvalue;
    }
    else if( 1 == sscanf( buf, "_cell_angle_beta  %lf",
			  &dvalue ) ){
      cell.angle_beta  = dvalue;
    }
    else if( 1 == sscanf( buf, "_cell_angle_gamma %lf",
			  &dvalue ) ){
      cell.angle_gamma = dvalue;
    }
    else if( 1 == sscanf(buf,"%s",svalue) &&
	     strcmp(svalue,"loop_") == 0 ){
    next_loop:
      // symmetry list or atom list
      bool loop_symmetry = false;
      bool loop_atom     = false;
      int column = 0;
      int column_xyz = -1;
      int column_x = -1;
      int column_y = -1;
      int column_z = -1;
      int column_symbol = -1;
      int column_label  = -1;

      // detect column of each variable
      // detect this list is symmetry list or atom list
      while( fgets(buf,sizeof(buf),fptr) &&
	     1 == sscanf(buf,"%s",svalue) ){
	if( svalue[0] != '_' ) break;

	if( false );
	else if( strcmp(svalue,"_symmetry_equiv_pos_as_xyz") == 0 ){
	  column_xyz=column;
	  loop_symmetry = true;
	}
	else if( strcmp(svalue,"_atom_site_fract_x") == 0 ){
	  column_x=column;
	  loop_atom = true;
	}
	else if( strcmp(svalue,"_atom_site_fract_y") == 0 ){
	  column_y=column;
	  loop_atom = true;
	}
	else if( strcmp(svalue,"_atom_site_fract_z") == 0 ){
	  column_z=column;
	  loop_atom = true;
	}
	else if( strcmp(svalue,"_atom_site_type_symbol") == 0 ){
	  column_symbol=column;
	  loop_atom = true;
	}
	else if( strcmp(svalue,"_atom_site_label") == 0 ){
	  column_label=column;
	  loop_atom = true;
	}
	else{
	  // unknown paramter, ignore it
	}
	column++;
      }

      // in case of symmetry list
      if( loop_symmetry ){
	voperation.clear();
	do{
	  vector<QString> vstr = parseLine(buf);
	  if( vstr.empty() ) break;
	  if( vstr[0] == "loop_" ) goto next_loop;

	  // read xyz
	  if( column_xyz != -1 && column_xyz<=(int)vstr.size() ){
	    // [note] IMatrix(char*) store Matrix by transposed.
	    IMatrix mat = IMatrix(qPrintable(vstr[column_xyz]));
	    IMatrix matT = mat.getTransposed();
	    voperation.push_back( IMatrix(matT.M,mat.T) );
	      //	    voperation.push_back( IMatrix(qPrintable(vstr[column_xyz])) );
	  }
	}while( fgets(buf,sizeof(buf),fptr) );
      } // loop_symmetry

      // in case of atom list
      if( loop_atom ){
	vatom.clear();
	Position p;
	QString   elem;
	do{
	  vector<QString> vstr = parseLine(buf);
	  if( vstr.empty() ) break;
	  if( vstr[0] == "loop_" ) goto next_loop;

	  // read x
	  if( column_x != -1 && column_x<=(int)vstr.size() ){
	    int nom,denom;
	    if( 2 == sscanf( qPrintable(vstr[column_x]),"%d/%d",&nom,&denom) ){
	      p.a = (double)nom/denom;
	    }
	    else if( 1 == sscanf( qPrintable(vstr[column_x]), "%lf", &p.a ) ){
	    }
	    else{
	      break;
	    }
	  }

	  // read y
	  if( column_y != -1 && column_y<=(int)vstr.size() ){
	    int nom,denom;
	    if( 2 == sscanf( qPrintable(vstr[column_y]),"%d/%d",&nom,&denom) ){
	      p.b = (double)nom/denom;
	    }
	    else if( 1 == sscanf( qPrintable(vstr[column_y]), "%lf", &p.b ) ){
	    }
	    else{
	      break;
	    }
	  }

	  // read z
	  if( column_z != -1 && column_z<=(int)vstr.size() ){
	    int nom,denom;
	    if( 2 == sscanf( qPrintable(vstr[column_z]),"%d/%d",&nom,&denom) ){
	      p.c = (double)nom/denom;
	    }
	    else if( 1 == sscanf( qPrintable(vstr[column_z]), "%lf", &p.c ) ){
	    }
	    else{
	      break;
	    }
	  }

	  // read symbol
	  if( column_symbol != -1 && column_symbol<=(int)vstr.size() ){
	    if( 1 != sscanf( qPrintable(vstr[column_symbol]), "%s", svalue ) ) break;
	    elem = QString(svalue);
	  }
	  else if( column_label != -1 && column_label<=(int)vstr.size() ){
	    if( 1 != sscanf( qPrintable(vstr[column_label]), "%s", svalue ) ) break;
	    elem = QString(svalue);
	  }
	  vatom.push_back( Atom(elem,p) );
	}while( fgets(buf,sizeof(buf),fptr) );
      } // loop_atom
    }
  }

  fclose(fptr);

  //------ check CIF
  SpaceGroupTable spacegrouptable;
  if( !spacegrouptable.find( spacegroup, symmetry_space_group_name ) ){
    fprintf(stderr,"Error in CIF: unknown symmetry_space_group_name %s.\n", qPrintable(symmetry_space_group_name) );
    return false;
  }

  if( space_group_crystal_system == "" ){
    space_group_crystal_system = spacegroup.shape;
  }
  if( space_group_crystal_system != spacegroup.shape ){
    fprintf(stderr,"Error in CIF: incorrect space_group_crystal_system %s.\n", qPrintable(space_group_crystal_system) );
    return false;
  }
  /*
  if( symmetry_Int_Tables_number != spacegroup.number ){
    fprintf(stderr,"Warning in CIF: incorrect symmetry_Int_Tables_number %d.\n", symmetry_Int_Tables_number );
    return false;
  }
  */

  constructVectors();
  constructAtoms();
  constructSymmetrys();

  return true;
}








void DTCIF::constructVectors( void )
{
  const double cosa = cos(cell.angle_alpha*M_PI/180.0);
  //  const double sina = sin(cell.angle_alpha*M_PI/180.0);
  const double cosb = cos(cell.angle_beta *M_PI/180.0);
  const double sinb = sin(cell.angle_beta *M_PI/180.0);
  const double cosc = cos(cell.angle_gamma*M_PI/180.0);
  const double sinc = sin(cell.angle_gamma*M_PI/180.0);

  if( false );
  else if( spacegroup.shape == "cubic" ){
    // Cubic
    unit_conv[0] = Position(cell.length_a,0.0,0.0);
    unit_conv[1] = Position(0.0,cell.length_b,0.0);
    unit_conv[2] = Position(0.0,0.0,cell.length_c);

    if( false );
    else if( spacegroup.center == "simple" ){
      unit_prim[0] = unit_conv[0];
      unit_prim[1] = unit_conv[1];
      unit_prim[2] = unit_conv[2];

      baseP2C[0] = Position(1.0,0.0,0.0);
      baseP2C[1] = Position(0.0,1.0,0.0);
      baseP2C[2] = Position(0.0,0.0,1.0);

      baseC2P[0] = Position(1.0,0.0,0.0);
      baseC2P[1] = Position(0.0,1.0,0.0);
      baseC2P[2] = Position(0.0,0.0,1.0);
    }
    else if( spacegroup.center == "face" ){
      unit_prim[0] = (unit_conv[1]+unit_conv[2])*0.5;
      unit_prim[1] = (unit_conv[2]+unit_conv[0])*0.5;
      unit_prim[2] = (unit_conv[0]+unit_conv[1])*0.5;

      baseP2C[0] = Position(0.0,0.5,0.5);
      baseP2C[1] = Position(0.5,0.0,0.5);
      baseP2C[2] = Position(0.5,0.5,0.0);

      baseC2P[0] = Position(-1.0,1.0,1.0);
      baseC2P[1] = Position(1.0,-1.0,1.0);
      baseC2P[2] = Position(1.0,1.0,-1.0);
    }
    else if( spacegroup.center == "body" ){
      unit_prim[0] = (-unit_conv[0]+unit_conv[1]+unit_conv[2])*0.5;
      unit_prim[1] = (+unit_conv[0]-unit_conv[1]+unit_conv[2])*0.5;
      unit_prim[2] = (+unit_conv[0]+unit_conv[1]-unit_conv[2])*0.5;

      baseP2C[0] = Position(-0.5,0.5,0.5);
      baseP2C[1] = Position(0.5,-0.5,0.5);
      baseP2C[2] = Position(0.5,0.5,-0.5);

      baseC2P[0] = Position(0.0,1.0,1.0);
      baseC2P[1] = Position(1.0,0.0,1.0);
      baseC2P[2] = Position(1.0,1.0,0.0);
    }
    else{
      fprintf(stderr,"Error: unknown cell type. %s %s\n",
	      qPrintable(spacegroup.shape), qPrintable(spacegroup.center) );
    }
  }
  else if( spacegroup.shape == "hexagonal" ){
    // Hexagonal
    // defulat spacegroup.paxis == "C"
    unit_conv[0] = Position(cell.length_a,0.00,0.00);
    unit_conv[1] = Position(-0.5*cell.length_a, sqrt(3)*0.5*cell.length_a,0.0);
    unit_conv[2] = Position(0.00,0.00,cell.length_c);

    if( spacegroup.center == "simple" ){
      unit_prim[0] = unit_conv[0];
      unit_prim[1] = unit_conv[1];
      unit_prim[2] = unit_conv[2];

      baseP2C[0] = Position(1.0,0.0,0.0);
      baseP2C[1] = Position(0.0,1.0,0.0);
      baseP2C[2] = Position(0.0,0.0,1.0);

      baseC2P[0] = Position(1.0,0.0,0.0);
      baseC2P[1] = Position(0.0,1.0,0.0);
      baseC2P[2] = Position(0.0,0.0,1.0);
    }
    else{
      fprintf(stderr,"Error: unknown cell type. %s %s\n",
	      qPrintable(spacegroup.shape), qPrintable(spacegroup.center) );
    }
  }
  else if( spacegroup.shape == "trigonal" ){
    // Trigonal
    if( cell.angle_gamma == 120.0 ){ //as hexagonal conventional cell
      unit_conv[0] = Position(cell.length_a, 0.00, 0.00 );
      unit_conv[1] = Position(-0.5*cell.length_a, sqrt(3)*0.5*cell.length_a, 0.0 );
      unit_conv[2] = Position(0.00, 0.00, cell.length_c );

      if( spacegroup.center == "simple" ){
	// make hexagonal primitive cell
	unit_prim[0] = unit_conv[0];
	unit_prim[1] = unit_conv[1];
	unit_prim[2] = unit_conv[2];

	baseP2C[0] = Position(1.0,0.0,0.0);
	baseP2C[1] = Position(0.0,1.0,0.0);
	baseP2C[2] = Position(0.0,0.0,1.0);

	baseC2P[0] = Position(1.0,0.0,0.0);
	baseC2P[1] = Position(0.0,1.0,0.0);
	baseC2P[2] = Position(0.0,0.0,1.0);

	/*
	unit_prim[0] =  2.0/3*unit_conv[0] + 1.0/3*unit_conv[1] + 1.0/3*unit_conv[2];
	unit_prim[1] = -1.0/3*unit_conv[0] + 1.0/3*unit_conv[1] + 1.0/3*unit_conv[2];
	unit_prim[2] = -1.0/3*unit_conv[0] - 2.0/3*unit_conv[1] + 1.0/3*unit_conv[2];

	baseP2C[0] = Position(2.0/3,1.0/3,1.0/3);
	baseP2C[1] = Position(-1.0/3,1.0/3,1.0/3);
	baseP2C[2] = Position(-1.0/3,-2.0/3,1.0/3);

	baseC2P[0] = Position(1.0,-1.0,0.0);
	baseC2P[1] = Position(0.0,1.0,-1.0);
	baseC2P[2] = Position(1.0,1.0,1.0);
	*/
      }
      else{
	fprintf(stderr,"Error: unknown cell type. %s %s\n",
		qPrintable(spacegroup.shape), qPrintable(spacegroup.center) );
      }
    }
    else{ //as trigonal conventional cell
      // vec{A}*vec{A} =      a*a + c*c = cell.length_a*cell.length_a;
      // vec{A}*vec{C} = -0.5*a*a + c*c = cell.length_a*cell.length_a*cosa
      // then 
      //  -a*a + 2*c*c = cell.length_a*cell.length_a*cosa*2

      const double a = sqrt(2.0/3.0*(1.0-cosa))*cell.length_a;
      const double c = sqrt(1.0/3.0*(1.0+2.0*cosa))*cell.length_a;

      unit_prim[0] = Position(a,0.0,c);
      unit_prim[1] = Position(-a*0.5,+sqrt(3)*a*0.5,c);
      unit_prim[2] = Position(-a*0.5,-sqrt(3)*a*0.5,c);

      if( spacegroup.center == "simple" ){
	unit_conv[0] = unit_prim[1] - unit_prim[2];
	unit_conv[1] = unit_prim[2] - unit_prim[0];
	unit_conv[2] = unit_prim[0] + unit_prim[1] + unit_prim[2];

	baseP2C[0]  = Position(2.0/3,  1.0/3, 1.0/3 );
	baseP2C[1]  = Position(-1.0/3,  1.0/3, 1.0/3 );
	baseP2C[2]  = Position(-1.0/3, -2.0/3, 1.0/3 );

	baseC2P[0]  = Position( 1.0,-1.0, 0.0);
	baseC2P[1]  = Position( 0.0, 1.0,-1.0);
	baseC2P[2]  = Position( 1.0, 1.0, 1.0);
      }
      else{
	fprintf(stderr,"Error: unknown cell type. %s %s\n",
		qPrintable(spacegroup.shape), qPrintable(spacegroup.center) );
      }
    }
  }
  else if( spacegroup.shape == "tetragonal" ){
    // Tetragonal
    unit_conv[0] = Position(cell.length_a,0.0,0.0);
    unit_conv[1] = Position(0.0,cell.length_b,0.0);
    unit_conv[2] = Position(0.0,0.0,cell.length_c);

    if( false );
    else if( spacegroup.center == "simple" ){
      unit_prim[0] = unit_conv[0];
      unit_prim[1] = unit_conv[1];
      unit_prim[2] = unit_conv[2];

      baseP2C[0] = Position(1.0,0.0,0.0);
      baseP2C[1] = Position(0.0,1.0,0.0);
      baseP2C[2] = Position(0.0,0.0,1.0);

      baseC2P[0] = Position(1.0,0.0,0.0);
      baseC2P[1] = Position(0.0,1.0,0.0);
      baseC2P[2] = Position(0.0,0.0,1.0);
    }
    else if( spacegroup.center == "body" ){
      unit_prim[0] = (-unit_conv[0]+unit_conv[1]+unit_conv[2])*0.5;
      unit_prim[1] = (+unit_conv[0]-unit_conv[1]+unit_conv[2])*0.5;
      unit_prim[2] = (+unit_conv[0]+unit_conv[1]-unit_conv[2])*0.5;

      baseP2C[0] = Position(-0.5,0.5,0.5);
      baseP2C[1] = Position(0.5,-0.5,0.5);
      baseP2C[2] = Position(0.5,0.5,-0.5);

      baseC2P[0] = Position(0.0,1.0,1.0);
      baseC2P[1] = Position(1.0,0.0,1.0);
      baseC2P[2] = Position(1.0,1.0,0.0);
    }
    else{
      fprintf(stderr,"Error: unknown cell type. %s %s\n",
	      qPrintable(spacegroup.shape), qPrintable(spacegroup.center) );
    }
  }
  else if( spacegroup.shape == "orthorhombic" ){
    // Orthorhombic
    // default spacegroup.paxis == "C"
    unit_conv[0] = Position(+cell.length_a,0.0,0.0);
    unit_conv[1] = Position(0.0,+cell.length_b,0.0);
    unit_conv[2] = Position(0.0,0.0,+cell.length_c);

    if( false );
    else if( spacegroup.center == "simple" ){
      unit_prim[0] = unit_conv[0];
      unit_prim[1] = unit_conv[1];
      unit_prim[2] = unit_conv[2];

      baseP2C[0] = Position(1.0,0.0,0.0);
      baseP2C[1] = Position(0.0,1.0,0.0);
      baseP2C[2] = Position(0.0,0.0,1.0);

      baseC2P[0] = Position(1.0,0.0,0.0);
      baseC2P[1] = Position(0.0,1.0,0.0);
      baseC2P[2] = Position(0.0,0.0,1.0);
    }
    else if( spacegroup.center == "face" ){
      unit_prim[0] = (unit_conv[1]+unit_conv[2])*0.5;
      unit_prim[1] = (unit_conv[2]+unit_conv[0])*0.5;
      unit_prim[2] = (unit_conv[0]+unit_conv[1])*0.5;

      baseP2C[0] = Position(0.0,0.5,0.5);
      baseP2C[1] = Position(0.5,0.0,0.5);
      baseP2C[2] = Position(0.5,0.5,0.0);

      baseC2P[0] = Position(-1.0,1.0,1.0);
      baseC2P[1] = Position(1.0,-1.0,1.0);
      baseC2P[2] = Position(1.0,1.0,-1.0);
    }
    else if( spacegroup.center == "body" ){
      unit_prim[0] = (-unit_conv[0]+unit_conv[1]+unit_conv[2])*0.5;
      unit_prim[1] = (+unit_conv[0]-unit_conv[1]+unit_conv[2])*0.5;
      unit_prim[2] = (+unit_conv[0]+unit_conv[1]-unit_conv[2])*0.5;

      baseP2C[0] = Position(-0.5,0.5,0.5);
      baseP2C[1] = Position(0.5,-0.5,0.5);
      baseP2C[2] = Position(0.5,0.5,-0.5);

      baseC2P[0] = Position(0.0,1.0,1.0);
      baseC2P[1] = Position(1.0,0.0,1.0);
      baseC2P[2] = Position(1.0,1.0,0.0);
    }
    else if( spacegroup.center == "base" ){
      unit_prim[0] = (unit_conv[1]+unit_conv[0])*0.5;
      unit_prim[1] = (unit_conv[1]-unit_conv[0])*0.5;
      unit_prim[2] = (unit_conv[2]);

      baseP2C[0] = Position(0.5,0.5,0.0);
      baseP2C[1] = Position(-0.5,0.5,0.0);
      baseP2C[2] = Position(0.0,0.0,1.0);

      baseC2P[0] = Position(1.0,-1.0,0.0);
      baseC2P[1] = Position(1.0, 1.0,0.0);
      baseC2P[2] = Position(0.0, 0.0,1.0);
    }
    else{
      fprintf(stderr,"Error: unknown cell type. %s %s\n",
	      qPrintable(spacegroup.shape), qPrintable(spacegroup.center) );
    }
  }
  else if( spacegroup.shape == "monoclinic" ){
    // Monoclinic
    // default spacegroup.paxis == "C"
    unit_conv[0] = Position(cell.length_a,0.0,0.0);
    unit_conv[1] = Position(0.0,cell.length_b,0.0);
    unit_conv[2] = Position(cell.length_c*cosb,0.0,cell.length_c*sinb);

    if( false );
    else if( spacegroup.center == "simple" ){
      unit_prim[0] = unit_conv[0];
      unit_prim[1] = unit_conv[1];
      unit_prim[2] = unit_conv[2];

      baseP2C[0] = Position(1.0,0.0,0.0);
      baseP2C[1] = Position(0.0,1.0,0.0);
      baseP2C[2] = Position(0.0,0.0,1.0);

      baseC2P[0] = Position(1.0,0.0,0.0);
      baseC2P[1] = Position(0.0,1.0,0.0);
      baseC2P[2] = Position(0.0,0.0,1.0);
    }
    else if( spacegroup.center == "base" ){
      unit_prim[0] = (unit_conv[1]+unit_conv[0])*0.5;
      unit_prim[1] = (unit_conv[1]-unit_conv[0])*0.5;
      unit_prim[2] = (unit_conv[2]);

      baseP2C[0] = Position(0.5,0.5,0.0);
      baseP2C[1] = Position(-0.5,0.5,0.0);
      baseP2C[2] = Position(0.0,0.0,1.0);

      baseC2P[0] = Position(1.0,-1.0,0.0);
      baseC2P[1] = Position(1.0, 1.0,0.0);
      baseC2P[2] = Position(0.0, 0.0,1.0);
    }
    else{
      fprintf(stderr,"Error: unknown cell type. %s %s\n",
	      qPrintable(spacegroup.shape), qPrintable(spacegroup.center) );
    }
  }
  else if( spacegroup.shape == "triclinic" ){
    // Triclinic
    unit_conv[0][0] = cell.length_a;       // A_x
    unit_conv[0][1] = 0.0;                 // A_y
    unit_conv[0][2] = 0.0;                 // A_z

    unit_conv[1][0] = cell.length_b*cosc;  // B_x
    unit_conv[1][1] = cell.length_b*sinc;  // B_y
    unit_conv[1][2] = 0.0;                 // B_z

    unit_conv[2][0] = cell.length_c*cosb;    // C_x
    unit_conv[2][1] = cell.length_c*(cosa-cosb*cosc)/sinc;  // C_y
    unit_conv[2][2] = sqrt(cell.length_c*cell.length_c-unit_conv[2][0]*unit_conv[2][0]-unit_conv[2][1]*unit_conv[2][1]); // C_z

    if( spacegroup.center == "simple" ){
      unit_prim[0] = unit_conv[0];
      unit_prim[1] = unit_conv[1];
      unit_prim[2] = unit_conv[2];

      baseP2C[0] = Position(1.0,0.0,0.0);
      baseP2C[1] = Position(0.0,1.0,0.0);
      baseP2C[2] = Position(0.0,0.0,1.0);

      baseC2P[0] = Position(1.0,0.0,0.0);
      baseC2P[1] = Position(0.0,1.0,0.0);
      baseC2P[2] = Position(0.0,0.0,1.0);
    }
    else{
      fprintf(stderr,"Error: unknown cell type. %s %s\n",
	      qPrintable(spacegroup.shape), qPrintable(spacegroup.center) );
    }
  }
  else{
    fprintf(stderr,"Error: unknown cell type. %s %s\n",
	    qPrintable(spacegroup.shape), qPrintable(spacegroup.center) );
  }
} 


void DTCIF::constructAtoms( void )
{
  vector<Atom>    vatom_conv;
  // calc position of atoms in conventional cell
  vatom_conv.clear();

  for( int a=0; a<(int)vatom.size(); a++ ){
    for( int n=0; n<(int)voperation.size(); n++ ){
      // [note] IMatrix*pos operate Matrix as transposed.
      Position position_translated;
      IMatrix mat  = voperation[n];
      IMatrix matT = IMatrix(mat.getTransposed().M,mat.T);
      position_translated = matT * vatom[a].coords;
      //      position_translated = voperation[n] * vatom[a].coords;
      Position::fold( position_translated );

      const Atom atom = Atom( vatom[a].element, position_translated );
      if( !find( vatom_conv, atom ) ){
	vatom_conv.push_back(atom);
      }
    }
  }

  // calc position of atoms in primitive cell
  vatom_prim.clear();
  for( int a=0; a<(int)vatom_conv.size(); a++ ){
    Position position_conv;
    Position position_prim;

    position_prim = Position::translate( baseC2P, vatom_conv[a].coords );
    Position::fold( position_prim );

    const Atom atom = Atom( vatom_conv[a].element, position_prim );
    if( !find( vatom_prim, atom ) ){
      vatom_prim.push_back(atom);
    }
  }
}

void DTCIF::constructSymmetrys( void )
{
  for( int n=0; n<(int)voperation.size(); n++ ){
    Position MB[3], BMB[3], BV;

    /*
    {//DEBUG
      printf("operation %3d:", n);
      for( int i=0; i<3; i++ ){
	for( int j=0; j<3; j++ ){
	  printf(" %+d", voperation[n].M[i][j] );
	}
      }
      printf(" ");
      for( int i=0; i<3; i++ ){
	  printf(" %4.2f", voperation[n].T[i].toDouble() );
      }
      printf("\n");
    }
    */

    double M[3][3],T[3];
    for( int i=0; i<3; i++ ){
      for( int j=0; j<3; j++ ){
	M[i][j] = (double)voperation[n].M[i][j];
      }
      T[i] = voperation[n].T[i].toDouble();
    }

    MB[0] = Position::translate( M, baseP2C[0] );
    MB[1] = Position::translate( M, baseP2C[1] );
    MB[2] = Position::translate( M, baseP2C[2] );

    BMB[0] = Position::translate( baseC2P, MB[0] );
    BMB[1] = Position::translate( baseC2P, MB[1] );
    BMB[2] = Position::translate( baseC2P, MB[2] );
    BV     = Position::translate( baseC2P, T );
    Position::fold( BV );

    int iM[3][3];
    fraction fT[3];
    for( int i=0; i<3; i++ ){
      for( int j=0; j<3; j++ ){
	iM[i][j] = (int)BMB[j][i]; // transpose
	//	iM[i][j] = (int)BMB[i][j];
      }
      fT[i] = fraction(BV[i]);
    }

    IMatrix op = IMatrix(iM,fT);

    if( !find( voperation_prim, op ) ){
      voperation_prim.push_back(op);
    }
  }

  /*
  //DEBUG
  for( int n=0; n<(int)voperation_prim.size(); n++ ){
    printf("operation %3d:", n);
    for( int i=0; i<3; i++ ){
      for( int j=0; j<3; j++ ){
	printf(" %+d", voperation_prim[n].M[i][j] );
      }
    }
    printf(" ");
    for( int i=0; i<3; i++ ){
      printf(" %4.2f", voperation_prim[n].T[i].toDouble() );
    }
    printf("\n");
  }
  */
  /*
    (q.x,q.y,q.z) = Mxyz*(p.x,p.y,p.z) + (T.x,T.y,T.z)

    (p.x,p.y,p.z) = baseP2C * (p.a,p.b,p.c)
    (q.x,q.y,q.z) = baseP2C * (q.a,q.b,q.c)
    (T.x,T.y,T.z) = baseP2C * (T.a,T.b,T.c)

    (p.a,p.b,p.c) = baseC2P * (p.x,p.y,p.z)

    baseP2C * baseC2P = 1

    baseP2C * (q.a,q.b,q.c)
    = Mxyz*baseP2C*(q.a,q.b,q.c) + baseP2C * (T.a,T.b,T.c)

    (q.a,q.b,q.c)
    = baseC2P*Mxyz*baseP2C*(q.a,q.b,q.c) + (T.a,T.b,T.c)
    (q.a,q.b,q.c) = Mabc*(q.a,q.b,q.c) + (T.a,T.b,T.c)

    Mabc = (baseC2P*Mxyz) * baseP2C
    Tabc = (baseC2P*(T.x,T.y,T.z)) + 0
   */

  // check -I operation is involved
  {
    IMatrix op_minus = IMatrix( -1,0,0, 0,-1,0, 0,0,-1 );
    has_inv = false;
    for( int n=0; n<(int)voperation_prim.size(); n++ ){
      if( voperation_prim[n] == op_minus ){
	has_inv = true;
	break;
      }
    }
  }
}

bool DTCIF::output( void ) const
{
  // data AA: const double AA_TO_AU = 1.0/0.529177210818;

  printf("# lattice\n");
  printf("  %f %f %f [AA]\n",
	  unit_prim[0][0],
	  unit_prim[0][1],
	  unit_prim[0][2] );
  printf("  %f %f %f [AA]\n",
	  unit_prim[1][0],
	  unit_prim[1][1],
	  unit_prim[1][2] );
  printf("  %f %f %f [AA]\n",
	  unit_prim[2][0],
	  unit_prim[2][1],
	  unit_prim[2][2] );

  printf("# symmetry\n");
  printf("    has_inversion = %d\n", (int)has_inv );
  for( int n=0; n<(int)voperation_prim.size(); n++ ){
    for( int i=0; i<3; i++ ){
      for( int j=0; j<3; j++ ){
 	printf(" %2d", (int)voperation_prim[n].M[j][i] );
      }
      printf("  ");
    }
    printf(" ");
    for( int i=0; i<3; i++ ){
      printf(" %f", voperation_prim[n].T[i].toDouble() );
    }
    printf("\n");
  }
  printf("\n");

  printf("# atoms\n");
  for( int a=0; a<(int)vatom_prim.size(); a++ ){
    printf("%s %f %f %f\n",
	   qPrintable(vatom_prim[a].element),
	    vatom_prim[a].coords[0],
	    vatom_prim[a].coords[1],
	    vatom_prim[a].coords[2] );
  }

  return true;
}

/*
int main( int argc, char* argv[] )
{
  DTCIF cif;

  if( argc == 2 ){
    if( !cif.load(argv[1]) ) return 1;

    cift.output();
  }

  return 0;
}
*/
