/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

/*!
 \file dtdos.cc
 \brief $B>uBVL)EY%G!<%?$N%/%i%9(B
*/

#include <stdio.h>
#include <math.h>
#include "dtdos.h"
#include "qtexception.h"


DTDos::DTDos( void )
{
  clear();
}

void DTDos::clear( void )
{
  vdata[0].clear();
  vdata[1].clear();
}

static double round( double& a, double& b )
{
  double d = a-b;
  double u = pow( 10.0, rint(log10(d))-1.0 );

  while( d/u >= 8.0 ){
    u *= 2.0;
  }

  b = floor(b/u)*u;
  a = ceil (a/u)*u;

  return u;
}

bool DTDos::load( const QString& fname )
{
  FILE* fptr = fopen( fname, "r" );
  if( fptr == NULL ){
    throw MyException("can not open a dos file.",fname);
  }

  const char* error_msg = NULL;
  char  buf[1024];
  int   nspin, wavil, npdos;
  char  Lstr;
  vector<Data> vdata_add[2];

  fgets(buf,sizeof(buf),fptr); // skip # epsend
  fgets(buf,sizeof(buf),fptr); // skip # ifbunp
  fgets(buf,sizeof(buf),fptr); // # nspin
  if( 1 != sscanf(buf," # nspin = %d", &nspin ) ){
    error_msg = "Dos file is broken in header section.";
    goto error_block;
  }
  fgets(buf,sizeof(buf),fptr); // # itwav, wavil
  if( 1 != sscanf(buf," # itwav, wavil = %*d %d", &wavil ) ){
    error_msg = "Dos file is broken in header section.";
    goto error_block;
  }
  else{
    switch(wavil){
    default: npdos = 0; Lstr=' '; break; // no PDOS
    case 1 : npdos = 1; Lstr='S'; break; // S type PDOS
    case 2 : npdos = 3; Lstr='P'; break; // P type PDOS
    case 3 : npdos = 5; Lstr='D'; break; // D type PDOS
    }
  }

  if( nspin==1 ){
    vdata_add[0].resize(1+npdos);
  }else{
    vdata_add[0].resize(1+npdos);
    vdata_add[1].resize(1+npdos);
  }

  for( int i=0; i<1+npdos; i++ ){
    if( i == 0 ){
      {
	vdata_add[0][i].title = "TDOS";
      }
      if( nspin == 2 ){
	vdata_add[1][i].title = "TDOS";
      }
    }
    else{
      char title[8];
      sprintf( title, "PDOS %c%d", Lstr, i);
      {
	vdata_add[0][i].title = QString(title);
      }
      if( nspin == 2 ){
	vdata_add[1][i].title = QString(title);
      }
    }
  }

  while( fgets(buf,sizeof(buf),fptr) ){
    char* ptr = buf;
    int   len;
    double energy, dos;
    if( 1 != sscanf(ptr,"%lf%n", &energy, &len ) ) break;
    ptr += len;

    {
      for( int i=0; i<1+npdos; i++ ){
	sscanf(ptr,"%lf%n", &dos, &len );
	ptr += len;
	vdata_add[0][i].energy.push_back(energy);
	vdata_add[0][i].dos.push_back(dos);
      }
    }
    if( nspin==2 ){
      for( int i=0; i<1+npdos; i++ ){
	sscanf(ptr,"%lf%n", &dos, &len );
	ptr += len;
	vdata_add[1][i].energy.push_back(energy);
	vdata_add[1][i].dos.push_back(-dos);
      }
    }
  }

  fclose(fptr);

  for( int i=0; i<1+npdos; i++ ){
    {
      if( !vdata[0].empty() ){
	if( vdata[0][0].energy != vdata_add[0][i].energy ){
	  error_msg = "Dos file is broken. energy mismatch.";
	  goto error_block;
	}
      }
    }
    if( nspin==2 ){
      if( !vdata[1].empty() ){
	if( vdata[1][0].energy != vdata_add[1][i].energy ){
	  error_msg = "Dos file is broken. energy mismatch.";
	  goto error_block;
	}
      }
    }

    {
      bool match=false;
      for( int j=0; j<(int)vdata[0].size(); j++ ){
	if( vdata[0][j].title == vdata_add[0][i].title ){
	  match = true;
	  break;
	}
      }
      if( !match ){
	vdata[0].push_back( vdata_add[0][i] );
      }
    }
    if( nspin==2 ){
      bool match=false;
      for( int j=0; j<(int)vdata[1].size(); j++ ){
	if( vdata[1][j].title == vdata_add[1][i].title ){
	  match = true;
	  break;
	}
      }
      if( !match ){
	vdata[1].push_back( vdata_add[1][i] );
      }
    }
  }

  emin=vdata[0][0].energy.front();
  emax=vdata[0][0].energy.back();

  dmin=dmax=vdata[0][0].dos[0];

  {
    for( int i=0; i<(int)vdata[0].size(); i++ ){
      for( int j=0; j<(int)vdata[0][i].dos.size(); j++ ){
	if( dmin > vdata[0][i].dos[j] ){
	  dmin = vdata[0][i].dos[j];
	}
	if( dmax < vdata[0][i].dos[j] ){
	  dmax = vdata[0][i].dos[j];
	}
      }
    }
  }
  if( nspin==2 ){
    for( int i=0; i<(int)vdata[1].size(); i++ ){
      for( int j=0; j<(int)vdata[1][i].dos.size(); j++ ){
	if( dmin > vdata[1][i].dos[j] ){
	  dmin = vdata[1][i].dos[j];
	}
	if( dmax < vdata[1][i].dos[j] ){
	  dmax = vdata[1][i].dos[j];
	}
      }
    }
  }

  if( dmin<0.0 ){ // nspin=2
    dmax = std::max(fabs(dmin),fabs(dmax));
    dmin = -dmax;
  }
  else{
    dmin=0.0;
  }

  round(emax,emin);
  round(dmax,dmin);

  update();

  return true;

 error_block:
  fclose(fptr);
  clear();
  throw MyException(error_msg,fname);
  return false; // dummy statement
}

bool DTDos::update( void )
{
  if( !isset() ) return false;

  return true;
}

