/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

/*!
 \file dtfield.cc
 \brief OtB[hf[^̃NX
*/

#include <stdio.h>
#include "dtlattice.h"
#include "dtfield.h"
#include "qtexception.h"
#include "dtcell.h"
#include <algorithm>

inline void swapdouble( double& datum )
{
  const int N = 8;
  union {
    char   c[N];
    double d;
  } a, b;

  a.d = datum;
  for( int i=0; i<N; i++ ){
    b.c[i] = a.c[N-1-i];
  }
  datum = b.d;
}

inline void swapfloat( float& datum )
{
  const int N = 4;
  union {
    char   c[N];
    double d;
  } a, b;

  a.d = datum;
  for( int i=0; i<N; i++ ){
    b.c[i] = a.c[N-1-i];
  }
  datum = b.d;
}

DTField::DTField( void )
{
  clear();
}

void DTField::clear( void )
{
  vdata.clear();
  idata = 1;

  multi.mode = 0;
  multi.data.clear();
  multi.vcoord.clear();
  multi.coord_shown = Coordinates(0.0,0.0,0.0);
  multi.index_shown = 1;
  multi.coord_manual = false;
}

bool DTField::nextfield( const bool init )
{
  switch( multi.mode ){
  case 0 : return init;
  case 1 :
    if( init ){
      multi.istart = idata-1;
      return true;
    }
    else{
      idata = idata % getSize() + 1;

      return idata-1 != multi.istart;
    }
  case 2 : return init;
  }
  return false;
}

void DTField::Multi::change_shown( void )
{
  if( vcoord.empty() ) return;
  if( index_shown > (int)vcoord.size() ) index_shown = 1;

  vcoord[index_shown-1] = coord_shown;
  vcoord[index_shown-1] = coord_shown;
}

void DTField::Multi::update_shown( void )
{
  if( vcoord.empty() ){
    coord_shown = Coordinates(0.0,0.0,0.0);
    return;
  }
  if( index_shown > (int)vcoord.size() ) index_shown = 1;
    
  coord_shown  = vcoord[index_shown-1];
  coord_shown  = vcoord[index_shown-1];
}


void DTField::update( void )
{
}

bool DTField::load( const QString& fname )
{
  if( false );
  else if( fname.endsWith( ".dx" ) ){
    return loadDX(fname);
  }
  else if( fname.endsWith( ".cube" ) ){
    return loadCube(fname);
  }
  else if( fname.endsWith( ".56" ) ){
    return loadTAPP(fname);
  }
  else if( fname.endsWith( ".57" ) ){
    return loadTAPP(fname);
  }
  else{}

  return false;
}

static bool checkPeriodic( const vector3d<double>& data1 )
{
  const int ixs=0;
  const int ixe=data1.sizeX()-1;
  const int iys=0;
  const int iye=data1.sizeY()-1;
  const int izs=0;
  const int ize=data1.sizeZ()-1;

  double max=0.0;

  for( int iy=0; iy<data1.sizeY(); iy++ ){
    for( int iz=0; iz<data1.sizeZ(); iz++ ){
      double diff = fabs(data1(ixs,iy,iz) - data1(ixe,iy,iz));
      if( max<diff) max=diff;
    }
  }

  for( int ix=0; ix<data1.sizeX(); ix++ ){
    for( int iy=0; iy<data1.sizeY(); iy++ ){
      double diff = fabs(data1(ix,iy,izs) - data1(ix,iy,ize));
      if( max<diff) max=diff;
    }
  }

  for( int ix=0; ix<data1.sizeX(); ix++ ){
    for( int iz=0; iz<data1.sizeZ(); iz++ ){
      double diff = fabs(data1(ix,iys,iz) - data1(ix,iye,iz));
      if( max<diff) max=diff;
    }
  }

  return max<1.0e-8;
}

bool DTField::loadDX( const QString& fname )
{
  int no, rank, shape, size, floatsize, floatendian;
  char buf[128];

  FILE* fptr = fopen( fname, "rb" );
  if( fptr == NULL ){
    throw MyException("can not open a field file.",fname);
  }

  //  clear();

  vdata.push_back(Datum());
  Datum& data = vdata.back();

  data.filename = fname;

  fgets( buf, sizeof(buf), fptr );
  if( 3 != sscanf( buf, "object 1 class gridpositions counts %d %d %d",
		   &Na, &Nb, &Nc ) ){
    fclose(fptr);
    throw MyException("broken line, object1.",fname);
  }
    
  fgets( buf, sizeof(buf), fptr ); // skip Lo
    
  fgets( buf, sizeof(buf), fptr );
  if( 3 != sscanf( buf, " delta %lf %lf %lf", &dLa.x, &dLa.y, &dLa.z ) ){
    fclose(fptr);
    throw MyException("broken line, delta.",fname);
  }
  La = dLa*Na;
    
  fgets( buf, sizeof(buf), fptr );
  if( 3 != sscanf( buf, " delta %lf %lf %lf", &dLb.x, &dLb.y, &dLb.z ) ){
    fclose(fptr);
    throw MyException("broken line, delta.",fname);
  }
  Lb = dLb*Nb;

  fgets( buf, sizeof(buf), fptr );
  if( 3 != sscanf( buf, " delta %lf %lf %lf", &dLc.x, &dLc.y, &dLc.z ) ){
    fclose(fptr);
    throw MyException("broken line, delta.",fname);
  }
  Lc = dLc*Nc;

  // set Lo so that Lo+(La+Lb+Lc)*(0.50) locates at the origin;
  Lo = (La+Lb+Lc)*(-0.50);
    
  fgets( buf, sizeof(buf), fptr ); // skip object 2
  if( 3 != sscanf( buf, "object 2 class gridconnections counts %d %d %d",
		   &Na, &Nb, &Nc ) ){
    fclose(fptr);
    throw MyException("broken line, object2.",fname);
  }

  while( fgets( buf, sizeof(buf), fptr ) ){ // object ...

    if( false );
    else if( 3 == sscanf( buf, "object %d class array type double rank %d shape %d",
			  &no, &rank, &shape ) ){
      continue;
    }
    else if( 2 == sscanf( buf, "object %d class array type float rank %d",
			  &no, &rank ) ){
      floatsize = sizeof(float);
      data.fieldtype = FIELD_REAL;
    }
    else if( 2 == sscanf( buf, "object %d class array type double rank %d", &no, &rank ) ){
      floatsize = sizeof(double);
      data.fieldtype = FIELD_REAL;
    }
    else if( 2 == sscanf( buf, "object %d class array type float category complex rank %d", &no, &rank ) ){
      floatsize = sizeof(float);
      data.fieldtype = FIELD_COMPLEX;
    }
    else if( 2 == sscanf( buf, "object %d class array type double category complex rank %d", &no, &rank ) ){
      floatsize = sizeof(double);
      data.fieldtype = FIELD_COMPLEX;
    }
    else{
      continue;
    }

    fgets( buf, sizeof(buf), fptr ); // items ...

    char str;
    if( false );
    else if( 2 == sscanf( buf, "items %d data follows%c", &size, &str ) ){
      floatendian = 0; // ascii
    }
    else if( 2 == sscanf( buf, "items %d lsb ieee data follows%c", &size, &str ) ){
      floatendian = 1; // binary lsb, little endian
    }
    else if( 2 == sscanf( buf, "items %d msb ieee data follows%c", &size, &str ) ){
      floatendian = 2; // binary, msb, big endian
    }
    else{
      continue;
    }

    switch(data.fieldtype){
    case FIELD_REAL : {
      data.data1.resize(Na,Nb,Nc);
      data.data2.clear();

      if( false );
      else if( floatsize == sizeof(double) ){
	if( false );
	else if( floatendian == 0 ){
	  for( int i=0; i<Na*Nb*Nc; i++ ){
	    fscanf( fptr, "%lf", &data.data1(i) );
	  }
	}
	else if( floatendian == 1 ){
	  fread( (double*) data.data1, sizeof(double), Na*Nb*Nc, fptr );
	  fgets( buf, sizeof(buf), fptr ); // skip end of this line
	}
	else if( floatendian == 2 ){
	  double datum;
	  for( int i=0; i<Na*Nb*Nc; i++ ){
	    fread( &datum, sizeof(double), 1, fptr );
	    swapdouble(datum);
	    data.data1(i) = datum;
	  }
	  fgets( buf, sizeof(buf), fptr ); // skip end of this line
	}
      }
      else if( floatsize == sizeof(float) ){
	if( false );
	else if( floatendian == 0 ){
	  for( int i=0; i<Na*Nb*Nc; i++ ){
	    fscanf( fptr, "%lf", &data.data1(i) );
	  }
	}
	else if( floatendian == 1 ){
	  float datum;
	  for( int i=0; i<Na*Nb*Nc; i++ ){
	    fread( &datum, sizeof(float), 1, fptr );
	    data.data1(i) = double(datum);
	  }
	  fgets( buf, sizeof(buf), fptr ); // skip end of this line
	}
	else if( floatendian == 2 ){
	  float datum;
	  for( int i=0; i<Na*Nb*Nc; i++ ){
	    fread( &datum, sizeof(float), 1, fptr );
	    swapfloat(datum);
	    data.data1(i) = double(datum);
	  }
	  fgets( buf, sizeof(buf), fptr ); // skip end of this line
	}
      }

      data.data1.range();
      bool periodic = checkPeriodic(data.data1);
      data.data1.setPeriodic(periodic);
      if( periodic ){
	La *= double(Na-1)/Na;
	Lb *= double(Nb-1)/Nb;
	Lc *= double(Nc-1)/Nc;
	Lo = (La+Lb+Lc)*(-0.50);
      }
    } break;

    case FIELD_COMPLEX : {
      data.data1.resize(Na,Nb,Nc);
      data.data2.resize(Na,Nb,Nc);

      if( false );
      else if( floatsize == sizeof(double) ){
	if( false );
	else if( floatendian == 0 ){
	  for( int i=0; i<Na*Nb*Nc; i++ ){
	    fscanf( fptr, "%lf %lf", &data.data1(i), &data.data2(i) );
	  }
	}
	else if( floatendian == 1 ){
	  double datum_r, datum_i;
	  for( int i=0; i<Na*Nb*Nc; i++ ){
	    fread( &datum_r, sizeof(double), 1, fptr );
	    fread( &datum_i, sizeof(double), 1, fptr );

	    data.data1(i) = hypot(datum_i,datum_r);
	    data.data2(i) = (atan2(datum_i,datum_r)+M_PI)*M_1_PI*0.5;
	  }
	  fgets( buf, sizeof(buf), fptr ); // skip end of this line
	}
	else if( floatendian == 2 ){
	  double datum_r, datum_i;
	  for( int i=0; i<Na*Nb*Nc; i++ ){
	    fread( &datum_r, sizeof(double), 1, fptr );
	    fread( &datum_i, sizeof(double), 1, fptr );
	    swapdouble(datum_r);
	    swapdouble(datum_i);
	    data.data1(i) = hypot(datum_i,datum_r);
	    data.data2(i) = (atan2(datum_i,datum_r)+M_PI)*M_1_PI*0.5;
	  }
	  fgets( buf, sizeof(buf), fptr ); // skip end of this line
	}
      }
      else if( floatsize == sizeof(float) ){
	if( false );
	else if( floatendian == 0 ){
	  for( int i=0; i<Na*Nb*Nc; i++ ){
	    fscanf( fptr, "%lf %lf", &data.data1(i), &data.data2(i) );
	  }
	}
	else if( floatendian == 1 ){
	  float datum_r, datum_i;
	  for( int i=0; i<Na*Nb*Nc; i++ ){
	    fread( &datum_r, sizeof(float), 1, fptr );
	    fread( &datum_i, sizeof(float), 1, fptr );

	    data.data1(i) = hypot(double(datum_i),double(datum_r));
	    data.data2(i) = (atan2(double(datum_i),double(datum_r))+M_PI)*M_1_PI*0.5;
	  }
	  fgets( buf, sizeof(buf), fptr ); // skip end of this line
	}
	else if( floatendian == 2 ){
	  float datum_r, datum_i;
	  for( int i=0; i<Na*Nb*Nc; i++ ){
	    fread( &datum_r, sizeof(float), 1, fptr );
	    fread( &datum_i, sizeof(float), 1, fptr );
	    swapfloat(datum_r);
	    swapfloat(datum_i);
	    data.data1(i) = hypot(double(datum_i),double(datum_r));
	    data.data2(i) = (atan2(double(datum_i),double(datum_r))+M_PI)*M_1_PI*0.5;
	  }
	  fgets( buf, sizeof(buf), fptr ); // skip end of this line
	}
      }

      data.data1.range();
      data.data2.range();
      bool periodic = checkPeriodic(data.data1); // abs part
      data.data1.setPeriodic(periodic); // set abs part
      data.data2.setPeriodic(periodic); // set arg part
      if( periodic ){
	La *= double(Na-1)/Na;
	Lb *= double(Nb-1)/Nb;
	Lc *= double(Nc-1)/Nc;
	Lo = (La+Lb+Lc)*(-0.50);
      }
    } break;
    }

  } // end while

  fclose(fptr);

  switch(data.fieldtype){
  case FIELD_REAL : {
    multi.min = vdata.front().data1.min();
    multi.max = vdata.front().data1.max();
    for( int n=1; n<(int)vdata.size(); n++ ){
      if( multi.min > vdata[n].data1.min() ){
	multi.max = vdata[n].data1.max();
      }
      if( multi.max < vdata[n].data1.max() ){
	multi.max = vdata[n].data1.max();
      }
    }
  } break;
  case FIELD_COMPLEX : {
    multi.min = vdata.front().data1.min();
    multi.max = vdata.front().data1.max();
    for( int n=1; n<(int)vdata.size(); n++ ){
      if( multi.min > vdata[n].data1.min() ){
	multi.max = vdata[n].data1.max();
      }
      if( multi.max < vdata[n].data1.max() ){
	multi.max = vdata[n].data1.max();
      }
    }
  } break;
  }

  return true;
}

bool DTField::loadCube( const QString& fname )
{
  char buf[128];
  FILE* fptr = fopen( fname, "rb" );
  if( fptr == NULL ){
    throw MyException("can not open a field file.",fname);
  }

  //  clear();

  vdata.push_back(Datum());
  Datum& data = vdata.back();

  data.filename = fname;

  fgets( buf, sizeof(buf), fptr ); // SKIP
  fgets( buf, sizeof(buf), fptr ); // SKIP

  int Natom=0;
  fgets( buf, sizeof(buf), fptr );
  if( 1 != sscanf( buf, "%d", &Natom ) ){
    fclose(fptr);
    throw MyException("broken header.",fname);
  }
  if( Natom<0 ){
    Natom=-Natom;
  }

  fgets( buf, sizeof(buf), fptr );
  if( 4 != sscanf( buf, " %d %lf %lf %lf", &Na, &dLa.x, &dLa.y, &dLa.z ) ){
    fclose(fptr);
    throw MyException("broken header.",fname);
  }
  if( Na<0 ){
    Na = -Na;
    dLa *= AU_TO_AA;
  }
  La = dLa*Na;

  fgets( buf, sizeof(buf), fptr );
  if( 4 != sscanf( buf, " %d %lf %lf %lf", &Nb, &dLb.x, &dLb.y, &dLb.z ) ){
    fclose(fptr);
    throw MyException("broken header.",fname);
  }
  if( Nb<0 ){
    Nb = -Nb;
    dLb *= AU_TO_AA;
  }
  Lb = dLb*Nb;

  fgets( buf, sizeof(buf), fptr );
  if( 4 != sscanf( buf, " %d %lf %lf %lf", &Nc, &dLc.x, &dLc.y, &dLc.z ) ){
    fclose(fptr);
    throw MyException("broken header.",fname);
  }
  if( Nc<0 ){
    Nc = -Nc;
    dLc *= AU_TO_AA;
  }
  Lc = dLc*Nc;
  Lo = (La+Lb+Lc)*(-0.50);

  for( int n=0; n<Natom; n++ ){
    fgets( buf, sizeof(buf), fptr ); // SKIP Atomic data
  }

  data.data1.resize(Na,Nb,Nc);
  data.data2.clear();
  data.fieldtype = FIELD_REAL;

  double v;
  for( int ix=0; ix<data.data1.sizeX(); ix++ ){
    for( int iy=0; iy<data.data1.sizeY(); iy++ ){
      for( int iz=0; iz<data.data1.sizeZ(); iz++ ){
	if( 1 != fscanf( fptr, "%le", &v ) ){
	  fclose(fptr);
	  throw MyException("invalid data.",fname);
	}
	data.data1(ix,iy,iz) = v;
      }
    }
  }
  fclose(fptr);

  data.data1.range();

  multi.min = data.data1.min();
  multi.max = data.data1.max();

  return true;
}



bool DTField::loadTAPP( const QString& fname )
{
  char buf[128];
  FILE* fptr = fopen( fname, "rb" );
  if( fptr == NULL ){
    throw MyException("can not open a field file.",fname);
  }

  clear();

  vdata.push_back(Datum());
  Datum& data = vdata.back();

  data.filename = fname;

  multi.vcoord.clear();

  while(true){
    while(true){
      if( fgets( buf, sizeof(buf), fptr ) == NULL ){
	goto end_block;
      }
      if( !strncmp( buf, "BEGIN-FIELD", 11 ) ) break;
    }

    fgets( buf, sizeof(buf), fptr ); // SKIP dimension

    fgets( buf, sizeof(buf), fptr );
    if( 3 != sscanf( buf, " %d %d %d", &Na, &Nb, &Nc ) ){
      fclose(fptr);
      throw MyException("broken header.",fname);
    }

    fgets( buf, sizeof(buf), fptr );
    if( 3 != sscanf( buf, " %lf %lf %lf", &La.x, &La.y, &La.z ) ){
      fclose(fptr);
      throw MyException("broken header.",fname);
    }

    fgets( buf, sizeof(buf), fptr );
    if( 3 != sscanf( buf, " %lf %lf %lf", &Lb.x, &Lb.y, &Lb.z ) ){
      fclose(fptr);
      throw MyException("broken header.",fname);
    }

    fgets( buf, sizeof(buf), fptr );
    if( 3 != sscanf( buf, " %lf %lf %lf", &Lc.x, &Lc.y, &Lc.z ) ){
      fclose(fptr);
      throw MyException("broken header.",fname);
    }

    Lo = (La+Lb+Lc)*(-0.50);

    data.data1.resize(Na,Nb,Nc);
    data.data2.clear();
    data.fieldtype = FIELD_REAL;

    double v;
    for( int iz=0; iz<data.data1.sizeZ(); iz++ ){
      for( int iy=0; iy<data.data1.sizeY(); iy++ ){
	for( int ix=0; ix<data.data1.sizeX(); ix++ ){
	  if( 1 != fscanf( fptr, "%le", &v ) ){
	    fclose(fptr);
	    throw MyException("invalid data.",fname);
	  }
	  data.data1(ix,iy,iz) = v;
	}
      }
    }
    fgets( buf, sizeof(buf), fptr ); // SKIP
    fgets( buf, sizeof(buf), fptr );
    if( strncmp( buf, "END-FIELD", 9 ) ){
      fclose(fptr);
      throw MyException("broken footer.",fname);
    }

    data.data1.range();
    if( vdata.size() == 1 || multi.min > data.data1.min() ){
      multi.min = data.data1.min();
    }
    if( vdata.size() == 1 || multi.max < data.data1.max() ){
      multi.max = data.data1.max();
    }
  }

 end_block:
  fclose(fptr);

  return true;
}

bool DTField::saveDX( const QString& fname )
{
  FILE* fptr = fopen( fname, "wb" );
  if( fptr == NULL ){
    throw MyException("can not create a field file.",fname);
  }

  fprintf(fptr,"object 1 class gridpositions counts      %d %d %d\n", Na, Nb, Nc );
  fprintf(fptr,"origin   %+20.15e  %+20.15e  %+20.15e\n",
	  Lo.x*AU_TO_AA, Lo.y*AU_TO_AA, Lo.z*AU_TO_AA );
  fprintf(fptr,"delta    %+20.15e  %+20.15e  %+20.15e\n",
	  La.x*AU_TO_AA, La.y*AU_TO_AA, La.z*AU_TO_AA );
  fprintf(fptr,"delta    %+20.15e  %+20.15e  %+20.15e\n",
	  Lb.x*AU_TO_AA, Lb.y*AU_TO_AA, Lb.z*AU_TO_AA );
  fprintf(fptr,"delta    %+20.15e  %+20.15e  %+20.15e\n",
	  Lc.x*AU_TO_AA, Lc.y*AU_TO_AA, Lc.z*AU_TO_AA );
  fprintf(fptr,"object 2 class gridconnections counts    %d %d %d\n", Na, Nb, Nc );
  fprintf(fptr,"object 3 array type double rank 0\nitems %d data follows\n", Na*Nb*Nc );

  const vector3d<double>& data1 = vdata.front().data1;
  for( int ix=0; ix<data1.sizeX(); ix++ ){
    for( int iy=0; iy<data1.sizeY(); iy++ ){
      for( int iz=0; iz<data1.sizeZ(); iz++ ){
	fprintf(fptr,"%+20.15e\n", data1(ix,iy,iz) );
      }
    }
  }
  fclose(fptr);

  return true;
}

bool DTField::saveCube( const QString& fname )
{
  FILE* fptr = fopen( fname, "wb" );
  if( fptr == NULL ){
    throw MyException("can not create a field file.",fname);
  }

  fprintf(fptr, "# Gaussian Cube file\n");
  fprintf(fptr, "# %s\n", qPrintable(fname) );
  fprintf(fptr, "%3d %+20.15e %+20.15e %+20.15e\n",  0, 0.0,  0.0,  0.0  );
  fprintf(fptr, "%3d %+20.15e %+20.15e %+20.15e\n", Na, La.x, La.y, La.z );
  fprintf(fptr, "%3d %+20.15e %+20.15e %+20.15e\n", Nb, Lb.x, Lb.y, Lb.z );
  fprintf(fptr, "%3d %+20.15e %+20.15e %+20.15e\n", Nc, Lc.x, Lc.y, Lc.z );

  const vector3d<double>& data1 = vdata.front().data1;
  for( int ix=0; ix<data1.sizeX(); ix++ ){
    for( int iy=0; iy<data1.sizeY(); iy++ ){
      for( int iz=0; iz<data1.sizeZ(); iz++ ){
	fprintf(fptr,"%+20.15e\n", data1(ix,iy,iz) );
      }
    }
  }
  fclose(fptr);

  return true;
}


bool DTField::saveTAPP( const QString& fname )
{
  FILE* fptr = fopen( fname, "wb" );
  if( fptr == NULL ){
    throw MyException("can not create a field file.",fname);
  }

  fprintf(fptr, "#TAPP-FIELD-FILE 1.0\n");

  for( int n=0; n<(int)vdata.size(); n++ ){  
    fprintf(fptr, "BEGIN-FIELD\n");
    fprintf(fptr, " %d                               # ndim\n", 3 );
    fprintf(fptr, " %d %d %d                         # nrx nry nrz\n", Na, Nb, Nc );
    
    fprintf(fptr, " %+20.15e %+20.15e %+20.15e     # AA\n", La.x, La.y, La.z );
    fprintf(fptr, " %+20.15e %+20.15e %+20.15e     # AA\n", Lb.x, Lb.y, Lb.z );
    fprintf(fptr, " %+20.15e %+20.15e %+20.15e     # AA\n", Lc.x, Lc.y, Lc.z );
    
    const vector3d<double>& data1 = vdata[n].data1;
    for( int iz=0; iz<data1.sizeZ(); iz++ ){
      for( int iy=0; iy<data1.sizeY(); iy++ ){
	for( int ix=0; ix<data1.sizeX(); ix++ ){
	  fprintf(fptr,"%+20.15e\n", data1(ix,iy,iz) );
	}
      }
    }
    fprintf(fptr, "END-FIELD\n");
  }
  fclose(fptr);

  return true;
}


extern "C" {
  void dgetrf_ ( const int& M, const int& N,
		 double* A, const int& LDA,
		 int* IPIV, int& INFO );
}

double DTField::calcDeterminant( const vector2d<double>& matrix )
{
  static vector<int>      ipiv;
  static vector2d<double> A;
  int info;

  ipiv.resize( matrix.sizeX() );
  A.resize( matrix.sizeX(), matrix.sizeX() );

  A = matrix;

  dgetrf_( A.sizeX(), A.sizeY(),
	   (double*)A, A.sizeX(), &ipiv[0], info );

  double det=1.0;
  for( int i=0; i<A.sizeX(); i++ ){
    if( i+1 != ipiv[i] ){
      det *= -A(i,i);
    }else{
      det *= +A(i,i);
    }
  }

  return det;
}

void DTField::updateSlaterCoords( const DTLattice& lattice )
{
  if( multi.coord_manual || multi.mode != 2 ) return;

  multi.vcoord.clear();

  for( int i=0; i<(int)lattice.getData().viatom.size(); i++ ){
    const Coordinates coord =
      lattice.getData().vatom
      [ lattice.getData().viatom[i] ].coords;

    if( std::find( multi.vcoord.begin(), multi.vcoord.end(), coord )
	== multi.vcoord.end() ){
      multi.vcoord.push_back(coord);
    }

    if( multi.vcoord.size()+1 == vdata.size() ) return;
  }

  for( int i=0; i<(int)lattice.getData().vatom.size(); i++ ){
    if( lattice.getData().vatom[i].isCopy() ) continue; // Rs[q

    const Coordinates coord = lattice.getData().vatom[i].coords;
    if( std::find( multi.vcoord.begin(), multi.vcoord.end(), coord )
	== multi.vcoord.end() ){
      multi.vcoord.push_back(coord);
    }

    if( multi.vcoord.size()+1 == vdata.size() ) return;
  }


  for( int i=0; i<(int)lattice.getData().vatom.size(); i++ ){
    if( lattice.getData().vatom[i].isCopy() ) continue; // Rs[q

    for( int j=i+1; i<(int)lattice.getData().vatom.size(); i++ ){
      if( lattice.getData().vatom[j].isCopy() ) continue; // Rs[q

      const Coordinates coord1 = lattice.getData().vatom[i].coords;
      const Coordinates coord2 = lattice.getData().vatom[j].coords;
      const Coordinates coord  = (coord1+coord2)*0.5;

      if( std::find( multi.vcoord.begin(), multi.vcoord.end(), coord )
	  == multi.vcoord.end() ){
	multi.vcoord.push_back(coord);
      }

      if( multi.vcoord.size()+1 == vdata.size() ) return;
    }
  }
    
  const Coordinates extra_coord[16] = {
    Coordinates(0.00,0.00,0.00),
    Coordinates(0.50,0.00,0.00),
    Coordinates(0.00,0.50,0.00),
    Coordinates(0.00,0.00,0.50),
    Coordinates(0.00,0.50,0.50),
    Coordinates(0.50,0.00,0.50),
    Coordinates(0.50,0.50,0.00),
    Coordinates(0.50,0.50,0.50),
    Coordinates(0.25,0.25,0.25),
    Coordinates(0.75,0.25,0.25),
    Coordinates(0.25,0.75,0.25),
    Coordinates(0.25,0.25,0.75),
    Coordinates(0.25,0.75,0.75),
    Coordinates(0.75,0.25,0.75),
    Coordinates(0.75,0.75,0.25),
    Coordinates(0.75,0.75,0.75),
  };

  for( int i=0; i<16 &&
	 multi.vcoord.size() < vdata.size(); i++ ){
    const Coordinates coord = extra_coord[i];
    if( std::find( multi.vcoord.begin(), multi.vcoord.end(), coord )
	== multi.vcoord.end() ){
      multi.vcoord.push_back(coord);
    }
    if( multi.vcoord.size()+1 == vdata.size() ) return;
  }

  while( multi.vcoord.size()+1 < vdata.size() ){
    const Coordinates coord = extra_coord[0];
    multi.vcoord.push_back(coord);
  }

}

void DTField::calcSlater( void )
{
  if( multi.mode != 2 ) return;

  multi.data.resize(Na,Nb,Nc);

  vector2d<double> matrix;
  matrix.resize( vdata.size(), vdata.size() );

  double factor = 1.0;
  for( int n=1; n<=(int)vdata.size(); n++ ){
    factor *= (double)n;
  }
  factor = 1.0/sqrt(factor);

  for( int m1=0; m1+1<(int)vdata.size(); m1++ ){
    for( int m2=0; m2<(int)vdata.size(); m2++ ){
      const double na = multi.vcoord[m1].a*Na;
      const double nb = multi.vcoord[m1].b*Nb;
      const double nc = multi.vcoord[m1].c*Nc;
      const int ia = (int)floor(na);
      const int ib = (int)floor(nb);
      const int ic = (int)floor(nc);
      const double fa = na - (double)ia;
      const double fb = nb - (double)ib;
      const double fc = nc - (double)ic;

      const double F0 =
	+(1.0-fa)*vdata[m2].data1(ia+0,ib  ,ic  )
	+(fa)    *vdata[m2].data1(ia+1,ib  ,ic  );
      const double F1 =
	+(1.0-fa)*vdata[m2].data1(ia+0,ib+1,ic  )
	+(fa)    *vdata[m2].data1(ia+1,ib+1,ic  );
      const double F2 =
	+(1.0-fa)*vdata[m2].data1(ia+0,ib  ,ic+1)
	+(fa)    *vdata[m2].data1(ia+1,ib  ,ic+1);
      const double F3 =
	+(1.0-fa)*vdata[m2].data1(ia+0,ib+1,ic+1)
	+(fa)    *vdata[m2].data1(ia+1,ib+1,ic+1);

      matrix(m1,m2) = (1.0-fc)*( (1.0-fb)*F0 + (fb)*F1 ) + (fc)*( (1.0-fb)*F2 + (fb)*F3 );
    } // m2
  } // m1

  vector<double> vcoef;
  vcoef.resize(vdata.size());

  for( int m1=0; m1<(int)vdata.size(); m1++ ){
    for( int m2=0; m2<(int)vdata.size(); m2++ ){
      matrix((int)vdata.size()-1,m2) = m1 == m2 ? 1.0 : 0.0;
    } // m2
    vcoef[m1] = factor*DTField::calcDeterminant(matrix);
  } // m1

  for( int ia=0; ia<Na; ia++ ){
    for( int ib=0; ib<Nb; ib++ ){
      for( int ic=0; ic<Nc; ic++ ){
	multi.data(ia,ib,ic) = 0.0;
	for( int m=0; m<(int)vdata.size(); m++ ){
	  multi.data(ia,ib,ic) += vcoef[m] * vdata[m].data1(ia,ib,ic);
	} // m
      } // ic
    } // ib
  } // ia

  multi.data.range();
}

double DTField::getScale( void ) const
{
  if( !isset() ) return 0.0;

  return Position::length(La+Lb+Lc);
}


static void foldindeces( int& i0, int&i1, double& w0, double& w1,
		  int& l0, int& l1, const double i, const int N )
{
  l0 = (int)floor((i+0.0)/N);
  l1 = (int)floor((i+1.0)/N);
  double il0 = i+0.0 - (double)(l0*N);
  double il1 = i+1.0 - (double)(l1*N);

  i0 = (int)floor(il0);
  i1 = (int)floor(il1);
  w1 = il0 - (double)i0;
  w0 = 1.0-w1;
}

// fold into [0:1]
static double foldphase( double phase )
{
  return fmod(1.0+fmod(phase,1.0),1.0);
}

static double phase( const vector3d<double>& v, const int la, const int lb, const int lc )
{
  if( !v.periodic ) return 0.0;
  return v.kx*la + v.ky*lb + v.kz*lc;
}

void DTField::exec_clustering( const DTCell& cell )
{
  if( !isset() ) return;

  // find dL of original field
  double dL = 0.0;
  dL = Position::length(dLa);
  if( dL>Position::length(dLb) ){
    dL = Position::length(dLb);
  }
  if( dL>Position::length(dLc) ){
    dL = Position::length(dLc);
  }

  const Position La2 = cell.La;
  const Position Lb2 = cell.Lb;
  const Position Lc2 = cell.Lc;

  const int Na2 = (int)ceil(Position::length(La2)/dL);
  const int Nb2 = (int)ceil(Position::length(Lb2)/dL);
  const int Nc2 = (int)ceil(Position::length(Lc2)/dL);

  
  const Position dLa2 = La2*(1.0/Na2);
  const Position dLb2 = Lb2*(1.0/Nb2);
  const Position dLc2 = Lc2*(1.0/Nc2);

  const double   V   = ( La * ( Lb % Lc) );
  const Position dKa = (Lb % Lc) * (1.0/V);
  const Position dKb = (Lc % La) * (1.0/V);
  const Position dKc = (La % Lb) * (1.0/V);


  vector3d<double> data_new;
  data_new.resize(Na2,Nb2,Nc2);

  int ia0,ia1,la0,la1;
  double wa0, wa1;
  int ib0,ib1,lb0,lb1;
  double wb0, wb1;
  int ic0,ic1,lc0,lc1;
  double wc0, wc1;

  for( int n=0; n<(int)vdata.size(); n++ ){
    vector3d<double>& data_org = vdata[n].data1;

    for( int ia2=0; ia2<Na2; ia2++ ){
      for( int ib2=0; ib2<Nb2; ib2++ ){
	for( int ic2=0; ic2<Nc2; ic2++ ){
	  Position r = dLa2*ia2 + dLb2*ib2 + dLc2*ic2;

	  const double ia = Na*(r*dKa);
	  const double ib = Nb*(r*dKb);
	  const double ic = Nc*(r*dKc);
	  foldindeces(ia0,ia1,wa0,wa1,la0,la1,ia,Na);
	  foldindeces(ib0,ib1,wb0,wb1,lb0,lb1,ib,Nb);
	  foldindeces(ic0,ic1,wc0,wc1,lc0,lc1,ic,Nc);

	  data_new(ia2,ib2,ic2) =
	    wc0*( wb0*(wa0*(data_org(ia0,ib0,ic0)) +
		       wa1*(data_org(ia1,ib0,ic0)) ) +
		  wb1*(wa0*(data_org(ia0,ib1,ic0)) +
		       wa1*(data_org(ia1,ib1,ic0)) ) ) +
	    wc1*( wb0*(wa0*(data_org(ia0,ib0,ic1)) +
		       wa1*(data_org(ia1,ib0,ic1)) ) +
		  wb1*(wa0*(data_org(ia0,ib1,ic1)) +
		       wa1*(data_org(ia1,ib1,ic1)) ) );
	}
      }
    }
    data_org = data_new;
    data_org.range();
  }


  for( int n=0; n<(int)vdata.size(); n++ ){
    vector3d<double>& data_org = vdata[n].data2;

    for( int ia2=0; ia2<Na2; ia2++ ){
      for( int ib2=0; ib2<Nb2; ib2++ ){
	for( int ic2=0; ic2<Nc2; ic2++ ){
	  Position r = dLa2*ia2 + dLb2*ib2 + dLc2*ic2;

	  const double ia = Na*(r*dKa);
	  const double ib = Nb*(r*dKb);
	  const double ic = Nc*(r*dKc);
	  foldindeces(ia0,ia1,wa0,wa1,la0,la1,ia,Na);
	  foldindeces(ib0,ib1,wb0,wb1,lb0,lb1,ib,Nb);
	  foldindeces(ic0,ic1,wc0,wc1,lc0,lc1,ic,Nc);

	  const double d =
	    wc0*( wb0*(wa0*(data_org(ia0,ib0,ic0)+phase(data_org,la0,lb0,lc0)) +
		       wa1*(data_org(ia1,ib0,ic0)+phase(data_org,la1,lb0,lc0)) ) +
		  wb1*(wa0*(data_org(ia0,ib1,ic0)+phase(data_org,la0,lb1,lc0)) +
		       wa1*(data_org(ia1,ib1,ic0)+phase(data_org,la1,lb1,lc0)) ) ) +
	    wc1*( wb0*(wa0*(data_org(ia0,ib0,ic1)+phase(data_org,la0,lb0,lc1)) +
		       wa1*(data_org(ia1,ib0,ic1)+phase(data_org,la1,lb0,lc1)) ) +
		  wb1*(wa0*(data_org(ia0,ib1,ic1)+phase(data_org,la0,lb1,lc1)) +
		       wa1*(data_org(ia1,ib1,ic1)+phase(data_org,la1,lb1,lc1)) ) );

	  data_new(ia2,ib2,ic2) = foldphase(d);
	}
      }
    }
    data_org = data_new;
    data_org.range();
  }

  La = La2;
  Lb = Lb2;
  Lc = Lc2;
  Lo = (La+Lb+Lc)*(-0.50);

  Na = Na2;
  Nb = Nb2;
  Nc = Nc2;
  
  dLa = dLa2;
  dLb = dLb2;
  dLc = dLc2;
}

