/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

/*!
 \file dtlattice.cc
 \brief iqf[^̓NX
*/

#include "dtlattice.h"
#include <stdio.h>
#include <QtGui/QtGui>
#include "qtexception.h"

DTLattice::DTLattice( void )
{
  clear();
}

void DTLattice::clear( void )
{
  cell.clear();
  symmetry.clear();

  cluster.clear();

  vdata.clear();
  idata=1;
  vdata.push_back(DTAtoms(cell));
}

void DTLattice::update( void )
{
  cell.update();
  symmetry.update();
  DTAtoms& atoms = getData();
  atoms.update();

  findSymmetry();
  updateAtoms();
}

DTLattice::Cluster::Cluster( void )
{
  clear();
}
void DTLattice::Cluster::clear( void )
{
  method = 0;
  for( int i=0; i<3; i++ ){
    for( int j=0; j<3; j++ ){
      M[i][j] = i==j ? 1 : 0;
    }
  }

  Ex = Ey = Ez = 1;
  La = Position(double(Ex),0.0,0.0);
  Lb = Position(0.0,double(Ey),0.0);
  Lc = Position(0.0,0.0,double(Ez));

  vatom.clear();
}

double round(double x)
{
  if ( x > 0.0 ) {
    return floor(x+0.5);
  }
  else {
    return -floor(-x+0.5);
  }
}

void DTLattice::clustering( void )
{
  cluster.vatom.clear();

  /*
  Position cell_VKa = cell.Eb%cell.Ec;
  Position cell_VKb = cell.Ec%cell.Ea;
  Position cell_VKc = cell.Ea%cell.Eb;
  double   cell_V   = cell.Ea*cell_VKa;
  */
  Position cell_VKa = cell.Lb%cell.Lc;
  Position cell_VKb = cell.Lc%cell.La;
  Position cell_VKc = cell.La%cell.Lb;
  double   cell_V   = cell.La*cell_VKa;

  if( cluster.method == 0 ){
    return;
  }
  if( cluster.method == 1 ){
    Position L = cell.La + cell.Lb + cell.Lc;
    cluster.La = Position((double)cluster.Ex,0.0,0.0)*L.x;
    cluster.Lb = Position(0.0,(double)cluster.Ey,0.0)*L.y;
    cluster.Lc = Position(0.0,0.0,(double)cluster.Ez)*L.z;

    cluster.M[0][0] = (int)round(cluster.La*cell_VKa/cell_V);
    cluster.M[0][1] = (int)round(cluster.La*cell_VKb/cell_V);
    cluster.M[0][2] = (int)round(cluster.La*cell_VKc/cell_V);
    cluster.M[1][0] = (int)round(cluster.Lb*cell_VKa/cell_V);
    cluster.M[1][1] = (int)round(cluster.Lb*cell_VKb/cell_V);
    cluster.M[1][2] = (int)round(cluster.Lb*cell_VKc/cell_V);
    cluster.M[2][0] = (int)round(cluster.Lc*cell_VKa/cell_V);
    cluster.M[2][1] = (int)round(cluster.Lc*cell_VKb/cell_V);
    cluster.M[2][2] = (int)round(cluster.Lc*cell_VKc/cell_V);
  }

  /*
  cluster.Ea =
    + cell.Ea*double(cluster.M[0][0])
    + cell.Eb*double(cluster.M[0][1])
    + cell.Ec*double(cluster.M[0][2]);
  cluster.Eb =
    + cell.Ea*double(cluster.M[1][0])
    + cell.Eb*double(cluster.M[1][1])
    + cell.Ec*double(cluster.M[1][2]);
  cluster.Ec =
    + cell.Ea*double(cluster.M[2][0])
    + cell.Eb*double(cluster.M[2][1])
    + cell.Ec*double(cluster.M[2][2]);
  */
  cluster.La =
    + cell.La*double(cluster.M[0][0])
    + cell.Lb*double(cluster.M[0][1])
    + cell.Lc*double(cluster.M[0][2]);
  cluster.Lb =
    + cell.La*double(cluster.M[1][0])
    + cell.Lb*double(cluster.M[1][1])
    + cell.Lc*double(cluster.M[1][2]);
  cluster.Lc =
    + cell.La*double(cluster.M[2][0])
    + cell.Lb*double(cluster.M[2][1])
    + cell.Lc*double(cluster.M[2][2]);

  Position cluster_VKa = cluster.Lb%cluster.Lc;
  Position cluster_VKb = cluster.Lc%cluster.La;
  Position cluster_VKc = cluster.La%cluster.Lb;
  double   cluster_V   = cluster.La*cluster_VKa;
  if( cluster_V < 0.0 ){
    cluster_VKa *= -1.0;
    cluster_VKb *= -1.0;
    cluster_VKc *= -1.0;
    cluster_V   *= -1.0;
  }

  int Mamax =
    + (cluster.M[0][0] >= 0 ? cluster.M[0][0] : 0)
    + (cluster.M[1][0] >= 0 ? cluster.M[1][0] : 0)
    + (cluster.M[2][0] >= 0 ? cluster.M[2][0] : 0);
  int Mbmax =
    + (cluster.M[0][1] >= 0 ? cluster.M[0][1] : 0)
    + (cluster.M[1][1] >= 0 ? cluster.M[1][1] : 0)
    + (cluster.M[2][1] >= 0 ? cluster.M[2][1] : 0);
  int Mcmax =
    + (cluster.M[0][2] >= 0 ? cluster.M[0][2] : 0)
    + (cluster.M[1][2] >= 0 ? cluster.M[1][2] : 0)
    + (cluster.M[2][2] >= 0 ? cluster.M[2][2] : 0);
  int Mamin =
    + (cluster.M[0][0] <= 0 ? cluster.M[0][0] : 0)
    + (cluster.M[1][0] <= 0 ? cluster.M[1][0] : 0)
    + (cluster.M[2][0] <= 0 ? cluster.M[2][0] : 0);
  int Mbmin =
    + (cluster.M[0][1] <= 0 ? cluster.M[0][1] : 0)
    + (cluster.M[1][1] <= 0 ? cluster.M[1][1] : 0)
    + (cluster.M[2][1] <= 0 ? cluster.M[2][1] : 0);
  int Mcmin =
    + (cluster.M[0][2] <= 0 ? cluster.M[0][2] : 0)
    + (cluster.M[1][2] <= 0 ? cluster.M[1][2] : 0)
    + (cluster.M[2][2] <= 0 ? cluster.M[2][2] : 0);

  // x̍L͈͂{
  Coordinates Ep;
  for( Ep.a =Mamin; Ep.a<=Mamax; Ep.a+=1.0 ){
    for( Ep.b=Mbmin; Ep.b<=Mbmax; Ep.b+=1.0 ){
      for( Ep.c=Mcmin; Ep.c<=Mcmax; Ep.c+=1.0 ){
	/*
  for( Ep.a =-2*cluster.Ex; Ep.a<=+2*cluster.Ex; Ep.a+=1.0 ){
    for( Ep.b =-2*cluster.Ey; Ep.b<=+2*cluster.Ey; Ep.b+=1.0 ){
      for( Ep.c =-2*cluster.Ez; Ep.c<=+2*cluster.Ez; Ep.c+=1.0 ){
	*/

	DTAtoms& atoms = getData();
	for( int i=0; i<(int)atoms.size(); i++ ){
	  if( atoms[i].isCopy() ) continue; // Rs[q
	  //	  if( atoms[i].element == 0 ) continue; // _~[q

	  DTAtom atom = atoms[i];
	  atom.coords = atoms[i].coords + Ep;

	  // XYZWn(iqxNgP)ɕϊĂ
	  /*
	  Position pos =
	    +cell.Ea*atom.coords.a
	    +cell.Eb*atom.coords.b
	    +cell.Ec*atom.coords.c;
	  */
	  Position pos =
	    +cell.La*atom.coords.a
	    +cell.Lb*atom.coords.b
	    +cell.Lc*atom.coords.c;

	  const double err=1.0e-8;
	  double cluster_Va = pos*cluster_VKa;
	  double cluster_Vb = pos*cluster_VKb;
	  double cluster_Vc = pos*cluster_VKc;
	  if( 0.0-err<=cluster_Va && cluster_Va<cluster_V-err &&
	      0.0-err<=cluster_Vb && cluster_Vb<cluster_V-err &&
	      0.0-err<=cluster_Vc && cluster_Vc<cluster_V-err ){
	    /*
	  if( 0.0 <= pos.x && pos.x < cluster.Ex &&
	      0.0 <= pos.y && pos.y < cluster.Ey &&
	      0.0 <= pos.z && pos.z < cluster.Ez ){
	    */
	    cluster.vatom.push_back(atom);
	  }
	} // i
      } // c
    } // b
  } // a
}


void DTLattice::cancel_clustering( void )
{
  cluster.clear();
}

void DTLattice::exec_clustering( void )
{
  clustering();
  if( cluster.vatom.empty() ) return;

  Position cluster_VKa = cluster.Lb%cluster.Lc;
  Position cluster_VKb = cluster.Lc%cluster.La;
  Position cluster_VKc = cluster.La%cluster.Lb;
  double   cluster_V   = cluster.La*cluster_VKa;
  if( cluster_V < 0.0 ){
    cluster_VKa *= -1.0;
    cluster_VKb *= -1.0;
    cluster_VKc *= -1.0;
    cluster_V   *= -1.0;
  }
  Position cluster_Ka = cluster_VKa*(1.0/cluster_V);
  Position cluster_Kb = cluster_VKb*(1.0/cluster_V);
  Position cluster_Kc = cluster_VKc*(1.0/cluster_V);

  for( int i=0; i<(int)cluster.vatom.size(); i++ ){
    Coordinates& coords = cluster.vatom[i].coords;
    /*
    const Position pos =
      +cell.Ea*coords.a
      +cell.Eb*coords.b
      +cell.Ec*coords.c;
    */
    const Position pos =
      +cell.La*coords.a
      +cell.Lb*coords.b
      +cell.Lc*coords.c;
    coords.a = pos*cluster_Ka;
    coords.b = pos*cluster_Kb;
    coords.c = pos*cluster_Kc;
  }

  /*
  cell.Ea = cluster.Ea;
  cell.Eb = cluster.Eb;
  cell.Ec = cluster.Ec;
  */
  cell.Ea = cluster.La*(1.0/cell.length);
  cell.Eb = cluster.Lb*(1.0/cell.length);
  cell.Ec = cluster.Lc*(1.0/cell.length);
  cell.vectors_changed = true;
  cell.update();

  for( int idata=0; idata<(int)vdata.size(); idata++ ){
    DTAtoms& atoms = vdata[idata];
    atoms.vatom.clear();
    atoms.unselectAtom();
    atoms.alignment = DTAtoms::ANY;

    for( int i=0; i<(int)cluster.vatom.size(); i++ ){
      atoms.push_back( cluster.vatom[i] );
      atoms.back().setAsOriginal(i);
    }
  }
  symmetry.clearIdentity();
  DTSymmetry::findDB(symmetry);

  cluster.clear();
}


void DTLattice::updateAtoms( void )
{
  // Rs[łȂIWǐq𒊏oB
  vector<DTAtom> vatom_orig;

  DTAtoms& atoms = getData();
  for( int i=0; i<(int)atoms.size(); i++ ){
    if( atoms[i].isCopy() ) continue;
    vatom_orig.push_back(atoms[i]);
  }

  // ̃f[^B
  atoms.vatom.clear();

  // Rs[łȂqf[^č\B
  for( int i=0; i<(int)vatom_orig.size(); i++ ){
    atoms.push_back( vatom_orig[i] );
    atoms.back().setAsOriginal(i);
  }

  // q邢͕sړΏ̑삪ꍇ͈ȍ~̏~B
  // Ώ̐󂷌qW̕ύX̐@\@\ȂȂB
  // ̒~̌ʁAq͑Ώ̑삪ꍇ́A
  // ʏ֎~Ώ̐󂷌q̈ړ\ɂȂB
  if( (int)vatom_orig.size() >= 100 ||
      (int)symmetry.vmatrix.size() >= 100 ){
    return;
  }

  // ΏەϊŃRs[qǉĂ
  for( int n=0; n<(int)symmetry.vmatrix.size(); n++ ){
    const IMatrix& imatrix = symmetry.vmatrix[n];
    if( imatrix == IMatrix::getIdentity() ) continue;

    for( int i=0; i<(int)vatom_orig.size(); i++ ){
      Coordinates coords0 = vatom_orig[i].getCoordinates();
      Coordinates coords = imatrix * coords0;
      Coordinates::normalize(coords);
      atoms.push_back( vatom_orig[i] );
      atoms.back().changeCoordinates( coords );
      atoms.back().setAsCopy(i,imatrix);

      if( Position::neareq(coords,coords0) ){
	atoms[i].setAsSpecial(imatrix);
      }
    }
  }

  for( int i=0; i<(int)atoms.size(); i++ ){
    for( int j=i+1; j<(int)atoms.size(); j++ ){
      if( Position::neareq(atoms[j].getCoordinates(),atoms[i].getCoordinates()) ){
	atoms[j].setAsDuplicate(); // 둤ɐݒ
      }
    }
  }
}


double DTLattice::getScale( void ) const
{
  double scale=0.0;
  {
    for( int ika=-1;ika<=+1; ika+=2 ){
      for( int ikb=-1;ikb<=+1; ikb+=2 ){
	for( int ikc=-1;ikc<=+1; ikc+=2 ){
	  const Position L = 
	    +cell.La*double(ika)
	    +cell.Lb*double(ikb)
	    +cell.Lc*double(ikc);
	  const double length = Position::length(L);
	  if( scale < length ){
	    scale = length;
	  }
	}
      }
    }
  }

  return scale;
}

static inline int nabs( const int n )
{
  return n<0 ? -n : n;
}

static bool match( const double matrixA[3][3], const double matrixB[3][3] )
{
  const double tolerance = 1.0e-5;
  double residue = 0.0;
  for( int i=0; i<3; i++ ){
    for( int j=0; j<3; j++ ){
      const double diff = matrixA[i][j] - matrixB[i][j];
      residue += diff*diff;
    }
  }
  return residue<tolerance*tolerance;
}

static bool match( const Coordinates& coordsA, const Coordinates& coordsB )
{
  const double tolerance = 1.0e-8;
  const double residue = Coordinates::normalize5(coordsA-coordsB).norm();
  return residue<tolerance*tolerance;
}


bool checkChanged( const vector<DTAtom>& vatom, const Position latvec[3] )
{
  static vector<DTAtom> vatom_prev;
  static Position latvec_prev[3] = {
    Position(0.0,0.0,0.0),
    Position(0.0,0.0,0.0),
    Position(0.0,0.0,0.0)
  };
  bool status = false;

  if( latvec_prev[0] != latvec[0] ||
      latvec_prev[1] != latvec[1] ||
      latvec_prev[2] != latvec[2] ){
    status = true;
    goto end_block;
  }

  for( int ia=0; ia<(int)vatom.size(); ia++ ){
    if( vatom[ia].isCopy() ) continue;
    if( ia>=(int)vatom_prev.size() ){
      status = true;
      goto end_block;
    }
    if( vatom_prev[ia].element.number != vatom[ia].element.number ||
	vatom_prev[ia].coords != vatom[ia].coords ){
      status = true;
      goto end_block;
    }
  }

end_block:
  vatom_prev = vatom;
  latvec_prev[0] = latvec[0];
  latvec_prev[1] = latvec[1];
  latvec_prev[2] = latvec[2];

  return status;
}

void DTLattice::findSymmetry( void )
{
  Position latvec[3];
  vector<DTAtom>& vatom = getData().vatom;

  if( vatom.empty() ) return;

  double metric_original[3][3];
  double metric_rotated[3][3];

  latvec[0] = cell.La;
  latvec[1] = cell.Lb;
  latvec[2] = cell.Lc;

  // check lattice is not changed
  if( !checkChanged( vatom, latvec ) ) return;

  // check this lattice has_inversion symmetry.
  // move all atoms so that the inversion center locates at the origin
  {
    symmetry.has_inversion = false;

    const int ia = 0;
    const Coordinates coords0_transformed = vatom[ia].coords*(-1.0);

    for( int ja=0; ja<(int)vatom.size(); ja++ ){
      if( vatom[ja].isCopy() ) continue;
      if( vatom[ja].element.number != vatom[ia].element.number ) continue;
      const Coordinates difference =
	Coordinates::normalize5(vatom[ja].coords - coords0_transformed);

      bool is_acceptable = true;
      for( int ka=0; ka<(int)vatom.size(); ka++ ){
	if( vatom[ka].isCopy() ) continue;
	const Coordinates coords_transformed = vatom[ka].coords*(-1.0) + difference;
	const int element_number_transformed = vatom[ka].element.number;

	bool is_match = false;
	for( int la=0; la<(int)vatom.size(); la++ ){
	  if( vatom[la].isCopy() ) continue;
	  if( vatom[la].element.number != element_number_transformed ) continue;
	  if( match( vatom[la].coords, coords_transformed ) ){
	    is_match = true;
	    break;
	  }
	} // end for la
	if( !is_match ){
	  is_acceptable = false;
	  break;
	}
      } // end for ka

      if( is_acceptable ){
	// it has inversion symmetry
	/*
	const Coordinates translation = difference*(-0.5);
	if( translation != Coordinates( 0.0, 0.0, 0.0 ) &&
	    MyException::question( "Move atoms for symmetry?" ) ){
	  // translate all atoms so that the inversion center locates on the origin
	  for( int ka=0; ka<(int)vatom.size(); ka++ ){
	    if( vatom[ka].isCopy() ) continue;

	    vatom[ka].coords += translation;
	    Coordinates::normalize(vatom[ka].coords);
	  }
	  symmetry.has_inversion = true;
	}
	*/
	const Coordinates translation = difference*(-0.5);
	if( translation == Coordinates( 0.0, 0.0, 0.0 ) ){
	  symmetry.has_inversion = true;
	}

	break; // end of check
      }
    }
  } // end of check


  for( int i=0; i<3; i++ ){
    for( int j=0; j<3; j++ ){
      metric_original[i][j] = latvec[i] * latvec[j];
    }
  }

  IMatrix matrix;

  symmetry.vmatrix.clear();
  matrix.initCanditate();
  do{
    if( nabs(matrix.determinant()) != 1 ){
      continue;
    }

    matrix.transformUnitary( metric_rotated, metric_original );

    if( !match( metric_rotated, metric_original ) ){
      continue;
    }

    const int ia = 0;
    const Coordinates coords0_transformed = matrix * vatom[ia].coords;

    for( int ja=0; ja<(int)vatom.size(); ja++ ){
      if( vatom[ja].isCopy() ) continue;
      if( vatom[ja].element.number != vatom[ia].element.number ) continue;

      const Coordinates difference =
	Coordinates::normalize(vatom[ja].coords - coords0_transformed);

      matrix.T[0] = fraction(difference.a);
      matrix.T[1] = fraction(difference.b);
      matrix.T[2] = fraction(difference.c);
      if( !matrix.T[0].isvalid() ) continue;
      if( !matrix.T[1].isvalid() ) continue;
      if( !matrix.T[2].isvalid() ) continue;


      bool is_acceptable = true;

      for( int ka=0; ka<(int)vatom.size(); ka++ ){
	if( vatom[ka].isCopy() ) continue;
	const Coordinates coords_transformed = matrix * vatom[ka].coords;
	const int element_number_transformed = vatom[ka].element.number;

	bool is_match = false;
	for( int la=0; la<(int)vatom.size(); la++ ){

	  if( vatom[la].isCopy() ) continue;
	  if( vatom[la].element.number != element_number_transformed ) continue;
	  if( match( vatom[la].coords, coords_transformed ) ){
	    is_match = true;
	    break;
	  }
	} // end for la

	if( !is_match ){
	  is_acceptable = false;
	  break;
	}
      } // end for ka

      if( is_acceptable ){
	symmetry.vmatrix.push_back(matrix);
      }
    } // end for ja
  }while( matrix.nextCanditate() );

  // Ώ̑삪ꍇɂ͑Ώ̑Pϊ݂̂ɐ
  if( symmetry.vmatrix.size()>128 ){
    symmetry.clearIdentity();
  }

  DTSymmetry::findDB(symmetry);
}
