/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

/*!
 \file dtmodel.cc
 \brief f[^̓NX
*/

#include <vector>
using namespace std;

#include "dtmodel.h"
#include "qtexception.h"
#include "dtcif.h"

QString getDirName( const QString& path )
{
  int index;
  index = path.lastIndexOf("/");
  if( index == -1 ){
    index = path.lastIndexOf("\\");
  }
  if( index == -1 ){
    return QString(".");
  }
  return path.mid(0,index);
}

QString getFileName( const QString& path )
{
  int index;
  index = path.lastIndexOf("/");
  if( index == -1 ){
    index = path.lastIndexOf("\\");
  }
  return path.mid(index+1);
}

DTModel::DTModel( void ) : 
  history_lattice(lattice),
  history_xtapp(xtapp)
{
  clear();
}

void DTModel::update( void )
{
  this->fname_saved = "";

  lattice.update();
  field.update();

  if( band.update() ){
    gloption.band.kmin = band.kmin;
    gloption.band.kmax = band.kmax;
    gloption.band.emin = band.emin;
    gloption.band.emax = band.emax;
  }

  if( dos.update() ){
    gloption.dos.emin = dos.emin;
    gloption.dos.emax = dos.emax;
    gloption.dos.dmin = dos.dmin;
    gloption.dos.dmax = dos.dmax;
  }

  tapp.update();
  xtapp.update();

  // backup current state to the history
  history_lattice.update();
  history_xtapp.update();

  emit changed();
}

void DTModel::clear( void )
{
  fname_saved = "";

  lattice.clear();
  field.clear();
  band.clear();
  dos.clear();
  tapp.clear();
  xtapp.clear();

  history_lattice.clear();
  history_xtapp.clear();

  update();
}

void DTModel::undo( void )
{
  history_lattice.undo();
  history_xtapp.undo();

  update();
}


bool DTModel::loadMolecule( const QString& fname )
{
  if( lattice.getData().load(lattice,fname) ){
    history_lattice.clear();
    history_xtapp.clear();
    this->fname_saved = fname;

    // ̃f[^ATCYقȂȂ
    if( field.isset() ){
      lattice.cell.update();
      if( !Position::neareq( lattice.cell.La, field.La ) ||
	  !Position::neareq( lattice.cell.Lb, field.Lb ) ||
	  !Position::neareq( lattice.cell.Lc, field.Lc ) ){
	field.clear();
      }
    }

    update();
    return true;
  }

  return false;
}

bool DTModel::saveMolecule( const QString& fname )
{
  if( lattice.getData().save(fname) ){
    update();
    return true;
  }

  return false;
}

bool DTModel::loadXTAPP( const QString& fname )
{
  if( (xtapp.guess(fname) && xtapp.load(lattice,fname)) ||
      (tapp.guess(fname) && tapp.load(lattice,fname) &&
       xtapp.convert(tapp)) ){
    history_lattice.clear();
    history_xtapp.clear();
    this->fname_saved = fname;

    // ̃f[^ATCYقȂȂ
    if( field.isset() ){
      lattice.cell.update();
      if( !Position::neareq( lattice.cell.La, field.La ) ||
	  !Position::neareq( lattice.cell.Lb, field.Lb ) ||
	  !Position::neareq( lattice.cell.Lc, field.Lc ) ){
	field.clear();
      }
    }

    update();

    return true;
  }

  return false;
}

bool DTModel::loadField( const QString& fname )
{
  if( field.load(fname) ){
    history_lattice.clear();
    history_xtapp.clear();
    this->fname_saved = fname;

    // lattice͏ɂB
    if( !Position::neareq( lattice.cell.La, field.La ) ||
	!Position::neareq( lattice.cell.Lb, field.Lb ) ||
	!Position::neareq( lattice.cell.Lc, field.Lc ) ){
      lattice.clear();
      lattice.cell.length = 1.0;
      lattice.cell.Ea = field.La;
      lattice.cell.Eb = field.Lb;
      lattice.cell.Ec = field.Lc;
      lattice.cell.vectors_changed = true;
    }

    update();

    return true;
  }

  return  false;
}


bool DTModel::loadBand( const QString& fname )
{
  if( band.load(fname) ){
    update();
    return true;
  }

  return false;
}

bool DTModel::loadDos( const QString& fname )
{
  if( dos.load(fname) ){
    update();
    return true;
  }

  return false;
}

bool DTModel::saveXTAPP( const QString& fname )
{
  if( xtapp.save(lattice,fname) ){
    this->fname_saved = fname;
    return true;
  }

  return false;
}

void DTModel::calcSlater( void )
{
  field.updateSlaterCoords( lattice );
  field.calcSlater();
}

double DTModel::getScale( void ) const
{
  return std::max( lattice.getScale(), field.getScale() );
}

void DTModel::exec_clustering( void )
{
  lattice.exec_clustering();
  if( field.isset() ){
    history_lattice.clear();
    field.exec_clustering(lattice.cell);
    MyException::warning( MyException("undo info has been cleared.") );
  }
}
