/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

/*!
 \file dtsymmetry.cc
 \brief iq̑Ώ̐f[^̃NX
*/

#include <stdio.h>
#include <string.h>
#include "dtsymmetry.h"
#include "qtmisc.h"

DTSymmetry::DTSymmetry( void )
{
  DTSymmetry::loadDB("spacegroup.db");
  this->idbsymname = 0;
}

DTSymmetry::DTSymmetry
( const char* symname,
  const vector<const char*> vmatname,
  const bool has_inversion, const int index )
{
  this->name = symname;
  this->has_inversion = has_inversion;

  for( int n=0; n<(int)vmatname.size(); n++ ){
    vmatrix.push_back( IMatrix(vmatname[n]) );
  }

  this->idbsymname = index;
  // suppose all components are listed in DB.
  //  expand();
}

void DTSymmetry::clear( void )
{
  clearIdentity();
  /*
  name = "";
  has_inversion = false;
  vmatrix.clear();
  */
}

void DTSymmetry::clearIdentity( void )
{
  name = "";
  has_inversion = false;
  vmatrix.clear();
  vmatrix.push_back( IMatrix("x,y,z") );
  idbsymname = 0;
  expand();
}

void DTSymmetry::update( void )
{
  if( vdbsymmetry.empty() ) return;

  *this = vdbsymmetry[idbsymname];
}

bool DTSymmetry::match( const DTSymmetry& sym ) const
{
  if( vmatrix.size() != sym.vmatrix.size() ) return false;

  if( has_inversion != sym.has_inversion ) return false;

  for( int n=0; n<(int)vmatrix.size(); n++ ){
    bool match = false;
    IMatrix M = vmatrix[n];
    IMatrix Minv = vmatrix[n];
    Minv *= -1.0;
    for( int m=0; m<(int)sym.vmatrix.size(); m++ ){
      if( M == sym.vmatrix[m] ){
	match = true;
	break;
      }
      if( has_inversion && Minv == sym.vmatrix[m] ){
	match = true;
	break;
      }
    }

    if( !match ){
      return false;
    }
  }
  return true;
}

void DTSymmetry::expand( void )
{
  for( int n=0; n<(int)vmatrix.size(); n++ ){
    expand_recursive(vmatrix[n]);
  }

  // left inversion components
  /*
  for( int n=0; n<(int)vmatrix.size(); n++ ){
    IMatrix M = vmatrix[n];
    M *= -1.0;
    for( int m=n+1; m<(int)vmatrix.size(); m++ ){
      if( M == vmatrix[m] ){
	has_inversion = true;
	vmatrix.erase( vmatrix.begin()+m );
	break;
      }
    }
  }
  */
}

void DTSymmetry::expand_recursive( const IMatrix matrix )
{
  const static IMatrix Midentity = IMatrix::getIdentity();

  if( matrix == Midentity ) return;

  for( int n=0; n<(int)vmatrix.size(); n++ ){
    if( vmatrix[n] == Midentity ) continue;

    const IMatrix new_matrix = vmatrix[n] * matrix;

    bool match=false;
    for( int m=0; m<(int)vmatrix.size(); m++ ){
      if( new_matrix == vmatrix[m] ){
	match=true;
	break;
      }
    }
    if( !match ){
      vmatrix.push_back( new_matrix );
      expand_recursive( new_matrix );
    }
  }
}

vector<DTSymmetry> DTSymmetry::vdbsymmetry;
vector<QString>     DTSymmetry::vdbsymname;


bool DTSymmetry::loadDB( const QString& fname )
{
  if( !vdbsymmetry.empty() ) return false;

  vector<const char*> vmatname;

  vdbsymmetry.clear();
  vdbsymname.clear();

  FILE* fptr = fopen(fname,"rb");
  if( fptr == NULL ){
    vmatname.clear();
    vmatname.push_back("x,y,z");

    vdbsymmetry.push_back( DTSymmetry("I",vmatname,false,vdbsymmetry.size()) );
    vdbsymname.push_back(QString("I"));

    return false;
  }

  char buf[256];
  char symname[32], matname[32];
  int  number;
  int  has_inversion;


  while( fgets(buf,sizeof(buf),fptr) ){
    if( buf[0] == '#' ) continue;

    if( false );
    else if( 1 == sscanf( buf, " spacegroup_name: %s", symname ) ){
    }
    else if( 1 == sscanf( buf, " has_inversion: %d", &has_inversion ) ){
    }
    else if( 1 == sscanf( buf, " number_operations: %d", &number ) ){
      fgets(buf,sizeof(buf),fptr);
      vmatname.resize(number);

      for( int n=0; n<number; n++ ){
	fgets(buf,sizeof(buf),fptr);
	sscanf(buf,"%*d %s",matname);
	vmatname[n] = strdup(matname);
      }
      vdbsymmetry.push_back
	( DTSymmetry(symname,vmatname,has_inversion==1,vdbsymmetry.size()) );
      vdbsymname.push_back(QString(symname));
    }
  }
  fclose(fptr);


  return true;
}

bool DTSymmetry::findDB( DTSymmetry& sym )
{
  for( int n=0; n<(int)vdbsymmetry.size(); n++ ){
    if( sym.match( vdbsymmetry[n] ) ){
      sym.idbsymname = n;
      return true;
    }
  }
  // new sysmmetry, push it.
  sym.name = "unknown new";
  sym.idbsymname = (int)vdbsymmetry.size();
  vdbsymmetry.push_back(sym);
  vdbsymname.push_back(sym.name);

  return false;
}
