/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

/*!
 \file dttapp.cc
 \brief TAPPݒf[^̃NX
*/

#include "dtlattice.h"
#include "dttapp.h"
#include "qtxml.h"
#include "qtexception.h"
#include "symbol.h"

bool  DTTapp::Option::wfn_on_disk_def = false;
bool  DTTapp::Option::qg_on_disk_def = false;
int DTTapp::Option::fsmear_def = 1;
int DTTapp::Option::hpc_def = 0;

int  DTTapp::LSDA::ixctype_def = 0;
int  DTTapp::LSDA::nspin_def  = 1;
int  DTTapp::LSDA::spinmode_def = 0;
int  DTTapp::LSDA::ipmode_def = 0;
int  DTTapp::LSDA::nfixed_def = 0, DTTapp::LSDA::nfixed_min=0, DTTapp::LSDA::nfixed_max=10;

double DTTapp::Cutoff::qf_def = 12.1,  DTTapp::Cutoff::qf_min=1.0,   DTTapp::Cutoff::qf_max=100.0;
double DTTapp::Cutoff::rf_def = 30.0,  DTTapp::Cutoff::rf_min=1.0,   DTTapp::Cutoff::rf_max=100.0;
double DTTapp::Cutoff::qm_def = 6.0,   DTTapp::Cutoff::qm_min=1.0,   DTTapp::Cutoff::qm_max=100.0;
double DTTapp::Cutoff::qc_def = 12.0,  DTTapp::Cutoff::qc_min=1.0,   DTTapp::Cutoff::qc_max=100.0;
double DTTapp::Cutoff::beta_def = 0.4, DTTapp::Cutoff::beta_min=0.0, DTTapp::Cutoff::beta_max=1.0;

int  DTTapp::Mesh::nd_def = 8, DTTapp::Mesh::nd_min=1, DTTapp::Mesh::nd_max=16;
int  DTTapp::Mesh::nk_def = 8, DTTapp::Mesh::nk_min=1, DTTapp::Mesh::nk_max=16;
int  DTTapp::Mesh::nw_def = 8, DTTapp::Mesh::nw_min=1, DTTapp::Mesh::nw_max=16;
int  DTTapp::Mesh::nb1_def = 2, DTTapp::Mesh::nb2_def = 8, DTTapp::Mesh::nb_min=1, DTTapp::Mesh::nb_max=16;
double DTTapp::Mesh::znext_def = 0.0, DTTapp::Mesh::znext_min=0.0, DTTapp::Mesh::znext_max=10.0;
double DTTapp::Mesh::dspin_def = 0.0, DTTapp::Mesh::dspin_min=0.0, DTTapp::Mesh::dspin_max=10.0;

double DTTapp::Ewald::qgb_def = 25.0,  DTTapp::Ewald::qgb_min=1.0, DTTapp::Ewald::qgb_max=50.0;
double DTTapp::Ewald::qrb_def = 150.0, DTTapp::Ewald::qrb_min=1.0, DTTapp::Ewald::qrb_max=300.0;
double DTTapp::Ewald::g_def   = 0.68,  DTTapp::Ewald::g_min=0.1,   DTTapp::Ewald::g_max=10.0;

int DTTapp::Sampling::nkrnew_def = 1, DTTapp::Sampling::nkrnew_min=0, DTTapp::Sampling::nkrnew_max=1;
int DTTapp::Sampling::ll_def = 0, DTTapp::Sampling::ll_min=0, DTTapp::Sampling::ll_max=1;

bool DTTapp::File::iopt1_def=0, DTTapp::File::iopt2_def=0, DTTapp::File::iopt3_def=0;
bool DTTapp::File::iopt4_def=1, DTTapp::File::iopt5_def=0, DTTapp::File::iopt6_def=0;

double DTTapp::Parameter::eps_def = 1.0e-16, DTTapp::Parameter::eps_min=1.0e-20, DTTapp::Parameter::eps_max=1.0e-12; 
double DTTapp::Parameter::eepsa_def = 1.0e-10, DTTapp::Parameter::eepsa_min=1.0e-14, DTTapp::Parameter::eepsa_max=1.0e-6; 
double DTTapp::Parameter::feps_def = 1.0e-3, DTTapp::Parameter::feps_min=1.0e-4, DTTapp::Parameter::feps_max=1.0e-2; 
double DTTapp::Parameter::ekbt_def = 1.0e-5, DTTapp::Parameter::ekbt_min=1.0e-7, DTTapp::Parameter::ekbt_max=1.0e-3;
double DTTapp::Parameter::decr_def = 0.1, DTTapp::Parameter::decr_min=0.01, DTTapp::Parameter::decr_max=0.1;
double DTTapp::Parameter::okatom_def = 0.5, DTTapp::Parameter::okatom_min=0.0, DTTapp::Parameter::okatom_max=1.0; 
double DTTapp::Parameter::uptime_def = 3.4e4, DTTapp::Parameter::uptime_min=0.0, DTTapp::Parameter::uptime_max=10000.0; 
int  DTTapp::Parameter::niter0_def = 50, DTTapp::Parameter::niter1_def = 40, DTTapp::Parameter::niter_min=0, DTTapp::Parameter::niter_max=100; 
int  DTTapp::Parameter::itsb_def = 3, DTTapp::Parameter::itsb_min=3, DTTapp::Parameter::itsb_max=5; 
int  DTTapp::Parameter::ndiag0_def = 1, DTTapp::Parameter::ndiag1_def = 1, DTTapp::Parameter::ndiag_min=0, DTTapp::Parameter::ndiag_max=10; 
int  DTTapp::Parameter::ncycl_def = 0, DTTapp::Parameter::ncycl_min=0, DTTapp::Parameter::ncycl_max=10; 
int  DTTapp::Parameter::mrfr_def = 5, DTTapp::Parameter::mrfr_min=3, DTTapp::Parameter::mrfr_max=5; 
int  DTTapp::Parameter::most_def = 4, DTTapp::Parameter::most_min=5, DTTapp::Parameter::most_max=8; 


int DTTapp::VBPEF::ifbunp_def = 2, DTTapp::VBPEF::ifbunp_min=0, DTTapp::VBPEF::ifbunp_max=2; 
int DTTapp::VBPEF::ifbfix_def = 0, DTTapp::VBPEF::ifbfix_min=0, DTTapp::VBPEF::ifbfix_max=4; 
int DTTapp::VBPEF::nw_def = 8, DTTapp::VBPEF::nw_min=1, DTTapp::VBPEF::nw_max=100; 
int DTTapp::VBPEF::nb2_def = 8, DTTapp::VBPEF::nb2_min=1, DTTapp::VBPEF::nb2_max=100; 
int DTTapp::VBPEF::itsb_def = 4, DTTapp::VBPEF::itsb_min=3, DTTapp::VBPEF::itsb_max=5; 
int DTTapp::VBPEF::ndiag_def = 30, DTTapp::VBPEF::ndiag_min=30, DTTapp::VBPEF::ndiag_max=50; 
double DTTapp::VBPEF::eps_def = 1.0e-5, DTTapp::VBPEF::eps_min=1.0e-5, DTTapp::VBPEF::eps_max=1.0e-4; 
int  DTTapp::VBPEF::nbba_def = 1, DTTapp::VBPEF::nbba_min=1, DTTapp::VBPEF::nbba_max=100; 
int  DTTapp::VBPEF::nsph_def = 1, DTTapp::VBPEF::nsph_min=1, DTTapp::VBPEF::nsph_max=3; 
int  DTTapp::VBPEF::iaxis_def = 1, DTTapp::VBPEF::iaxis_min=1, DTTapp::VBPEF::iaxis_max=3; 
double DTTapp::VBPEF::rzero_def = 0.00, DTTapp::VBPEF::rzero_min=0.0, DTTapp::VBPEF::rzero_max=1.0; 
double DTTapp::VBPEF::rmin_def = 0.0, DTTapp::VBPEF::rmin_min=0.0, DTTapp::VBPEF::rmin_max=10.0; 
double DTTapp::VBPEF::rmax_def = 5.0, DTTapp::VBPEF::rmax_min=0.0, DTTapp::VBPEF::rmax_max=10.0; 
int  DTTapp::VBPEF::ndiv_def = 20, DTTapp::VBPEF::ndiv_min=1, DTTapp::VBPEF::ndiv_max=100; 

// Rgsǂݔ΂ĔRgŝ݂Ԃ֐
static char* fgets_comment( char* buf, int size, FILE* fptr )
{
  do{
    if( fgets(buf,size,fptr) == NULL ) return NULL;
  }while( buf[0] == '#' || buf[0] == '!' ||
	  buf[0] == '\r' || buf[0] == '\n' );

  return buf;
}

// Fortran̎w` 1.0d2 C̎w` 1.0e2 ɕϊ֐
static bool change_exp_format( char* ptr )
{
  while( *ptr ){
    if( *ptr == 'd' ) *ptr = 'e';
    ptr++;
  }
  return true;
}


DTTapp::DTTapp( void )
{
  loadDefaults();
  clear();
}

void DTTapp::update( void )
{
}

void DTTapp::clear( void )
{
  lsda.clear();
  cutoff.clear();
  mesh.clear();
  ewald.clear();
  sampling.clear();
  file.clear();
  mass.clear();
  parameter.clear();
  polar.clear();
  vbpef.clear();
  vbwfn.clear();
  tchrpot.clear();
}

DTTapp::Option::Option( void )
{
  clear();
}

void DTTapp::Option::clear( void )
{
  wfn_on_disk = wfn_on_disk_def;
  qg_on_disk  = qg_on_disk_def;
  fsmear      = fsmear_def;
  hpc         = hpc_def;
}

bool DTTapp::Option::load( FILE* fptr )
{
  char buf[1024];

  int bvalue1, bvalue2, bvalue3, bvalue4;
  if( fgets_comment(buf,sizeof(buf),fptr) &&
      4 == sscanf(buf, "%d %d %d %d",
		  &bvalue1, &bvalue2, &bvalue3, &bvalue4 ) ){
    wfn_on_disk = bvalue1 != 0;
    qg_on_disk  = bvalue2 != 0;
    fsmear      = bvalue3;
    hpc         = bvalue4;
  }
  else{
    return false;
  }

  return true;
}

bool DTTapp::Option::save( FILE* fptr )
{
  fprintf(fptr, "!----------- section option\n");

  fprintf(fptr, "%d %d %d %d 0 0 0 "
	  "# if_wfn_on_disk if_qg_on_disk if_fsmear if_hpc\n",
	  (int)wfn_on_disk,
	  (int)qg_on_disk,
	  fsmear,
	  hpc );

  return true;
}


DTTapp::LSDA::LSDA( void )
{
  vxctype.push_back("'LDAPZ81'");
  vxctype.push_back("'LDAPW92'");
  vxctype.push_back("'LDAWIGNR'");
  vxctype.push_back("'LDAXALPH'");
  vxctype.push_back("'GGAPW91'");
  vxctype.push_back("'GGAPBE96'");
  vxctype.push_back("'ANY'");

  clear();
}

void DTTapp::LSDA::clear( void )
{
  ixctype  = ixctype_def;
  nspin    = nspin_def;
  spinmode = spinmode_def;
  ipmode   = ipmode_def;
  nfixed   = nfixed_def;
}


bool DTTapp::LSDA::load( FILE* fptr )
{
  char buf[1024];
  char svalue[32];

  if( fgets_comment(buf,sizeof(buf),fptr) &&
      5 == sscanf(buf, "%s %d %d %d %d",
		  svalue, &nspin, &spinmode, &ipmode, &nfixed ) ){
    bool found=false;

    for( int n=0; n<(int)vxctype.size(); n++ ){
      if( vxctype[n] == svalue ){
	ixctype = n;
	found=true;
	break;
      }
    }
    if(!found){
      return false;
    }
  }
  else{
    return false;
  }
  return true;
}

bool DTTapp::LSDA::save( FILE* fptr )
{
  fprintf(fptr, "!----------- section lsda\n");

  fprintf(fptr, "%s %d %d %d %d "
	  "# xctype nspin spinmode ipmode nfixed\n",
	  qPrintable(vxctype[ixctype]),
	  nspin, spinmode, ipmode, nfixed );

  return true;
}

bool DTTapp::Cell::load( DTCell& cell, FILE* fptr )
{
  char buf[1024];

  if( fgets_comment(buf,sizeof(buf),fptr) &&
      change_exp_format(buf) &&
      1 == sscanf(buf, "%le", &cell.length ) &&
      fgets_comment(buf,sizeof(buf),fptr) &&
      change_exp_format(buf) &&
      3 == sscanf(buf, "%le %le %le", &cell.Ea.x, &cell.Ea.y, &cell.Ea.z ) &&
      fgets_comment(buf,sizeof(buf),fptr) &&
      change_exp_format(buf) &&
      3 == sscanf(buf, "%le %le %le", &cell.Eb.x, &cell.Eb.y, &cell.Eb.z ) &&
      fgets_comment(buf,sizeof(buf),fptr) &&
      change_exp_format(buf) &&
      3 == sscanf(buf, "%le %le %le", &cell.Ec.x, &cell.Ec.y, &cell.Ec.z ) ){
    cell.vectors_changed = true;
  }
  else{
    return false;
  }
  return true;
}

bool DTTapp::Cell::save( const DTCell& cell, FILE* fptr )
{
  fprintf(fptr, "!----------- section cell\n");

  fprintf(fptr, "%f # ax\n", cell.length );
  fprintf(fptr, "%f %f %f # aa(1:3,1)\n",
	  cell.Ea.x, cell.Ea.y, cell.Ea.z );
  fprintf(fptr, "%f %f %f # aa(1:3,2)\n",
	  cell.Eb.x, cell.Eb.y, cell.Eb.z );
  fprintf(fptr, "%f %f %f # aa(1:3,3)\n",
	  cell.Ec.x, cell.Ec.y, cell.Ec.z );

  return true;
}

bool DTTapp::Symmetry::load( DTSymmetry& symmetry, FILE* fptr )
{
  char buf[1024];

  int ivalue;
  int nsv, nnp, kprint;

  if( fgets_comment(buf,sizeof(buf),fptr) &&
      4 == sscanf(buf, "%d %d %d %d",
		  &ivalue, &nsv, &nnp, &kprint ) ){
  }
  else{
    return false;
  }

  symmetry.name = "data";
  symmetry.vmatrix.resize(ivalue);

  for( int n=0; n<(int)symmetry.vmatrix.size(); n++ ){
    IMatrix imatrix;
    if( fgets_comment(buf,sizeof(buf),fptr) &&
	9 == sscanf(buf, " %d %d %d %d %d %d %d %d %d",
		    &imatrix.M[0][0], &imatrix.M[1][0], &imatrix.M[2][0],
		    &imatrix.M[0][1], &imatrix.M[1][1], &imatrix.M[2][1],
		    &imatrix.M[0][2], &imatrix.M[1][2], &imatrix.M[2][2] ) ){
    }
    else{
      return false;
    }
    symmetry.vmatrix[n] = imatrix.getTransInverse();
  }

  for( int n=0; n<(int)symmetry.vmatrix.size(); n++ ){
    int Ta, Tb, Tc;
    while(true){
      if( 3 == fscanf(fptr, " %d %d %d",
		      &Ta, &Tb, &Tc ) ) break;
      fgets(buf,sizeof(buf),fptr); // read \r\n
    }
    symmetry.vmatrix[n].T[0] = fraction( Ta, nnp );
    symmetry.vmatrix[n].T[1] = fraction( Tb, nnp );
    symmetry.vmatrix[n].T[2] = fraction( Tc, nnp );
  }
  fgets(buf,sizeof(buf),fptr); // read \r\n

  symmetry.expand();

  if( !DTSymmetry::findDB(symmetry) ){
    //    MyException::warning("unknown new symmetry");
    //    return false;
    return true;
  }

  return true;
}


bool DTTapp::Symmetry::save( const DTSymmetry& symmetry, FILE* fptr )
{
  fprintf(fptr, "!----------- section symmetry\n");

  int nsv = 1, nnp, kprint = 1;

  // SĂ̕sړq̍̕ŏ{߂B
  nnp=1;
  for( int n=0; n<(int)symmetry.vmatrix.size(); n++ ){
    nnp = fraction::common_denominator
      ( nnp, symmetry.vmatrix[n].T[0].denominator );
    nnp = fraction::common_denominator
      ( nnp, symmetry.vmatrix[n].T[1].denominator );
    nnp = fraction::common_denominator
      ( nnp, symmetry.vmatrix[n].T[2].denominator );
  }

  fprintf(fptr, "%d %d %d %d "
	  "# ng nsv nnp kprint\n",
	  (int)symmetry.vmatrix.size(), nsv, nnp, kprint );

  for( int n=0; n<(int)symmetry.vmatrix.size(); n++ ){
    const IMatrix imatrix = symmetry.vmatrix[n].getTransInverse();
    for( int i=0; i<3; i++ ){
      for( int j=0; j<3; j++ ){
	fprintf(fptr, "%2d ", imatrix.M[j][i] );
      }
    }
    fprintf(fptr, "# rg(1:3,1:3,%2d)\n",n+1);
  }
  for( int n=0; n<(int)symmetry.vmatrix.size(); n++ ){
    int Ta, Tb, Tc;

    Ta = (symmetry.vmatrix[n].T[0]*nnp).toInteger();
    Tb = (symmetry.vmatrix[n].T[1]*nnp).toInteger();
    Tc = (symmetry.vmatrix[n].T[2]*nnp).toInteger();

    fprintf(fptr, "%d %d %d  ", Ta, Tb, Tc );
    if( (n+1)%6 == 0 ){
      fprintf(fptr, "# pg(1:3,%2d:%2d)\n",n+1-5,n+1);
    }
  }
  return true;
}



DTTapp::Cutoff::Cutoff( void )
{
  clear();
}

void DTTapp::Cutoff::clear( void )
{
  qf = qf_def;
  rf = rf_def;
  qm = qm_def;
  qc = qc_def;
  beta = beta_def;
  qbtp1 = qbtp2 = qbtp3 = 0.0;
}

bool DTTapp::Cutoff::load( FILE* fptr )
{
  char buf[1024];

  if( fgets_comment(buf,sizeof(buf),fptr) &&
      change_exp_format(buf) ){
    if( 8 == sscanf(buf, "%le %le %le %le %le %le %le %le",
		    &qf, &rf, &qm, &qc, &beta,
		    &qbtp1, &qbtp2, &qbtp3 ) ){
    }
    else if( 5 == sscanf(buf, "%le %le %le %le %le",
		    &qf, &rf, &qm, &qc, &beta ) ){
    }
  }
  else{
    return false;
  }
  return true;
}

bool DTTapp::Cutoff::save( FILE* fptr )
{
  fprintf(fptr, "!----------- section cutoff\n");

  qbtp1 = 0.0;
  qbtp2 = qm;
  qbtp3 = qm;

  fprintf(fptr, "%6.3f %6.3f %6.3f %6.3f %6.3f %6.3f %6.3f %6.3f "
	  "# qf rf qm qc beta qbtp(1:3)\n",
	  qf, rf, qm, qc, beta, qbtp1, qbtp2, qbtp3 );

  return true;
}

bool DTTapp::Atoms::load( DTAtoms& atoms, FILE* fptr )
{
  char buf[1024];
  char svalue[128];

  int ivalueE, ivalueA;

  if( fgets_comment(buf,sizeof(buf),fptr) &&
      2 == sscanf(buf, "%d %d", &ivalueE, &ivalueA ) ){
  }
  else{
    return false;
  }
  atoms.clear();

  for( int n=0; n<ivalueE; n++ ){
    if( fgets_comment(buf,sizeof(buf),fptr) &&
	1 == sscanf(buf, "%s", svalue )){
      atoms.velement.push_back( DTElement(svalue) );
    }
  }
 
  int ppid;
  Coordinates coords;
  int motion;
  for( int n=0; n<ivalueA; n++ ){
    if( fgets_comment(buf,sizeof(buf),fptr) &&
	change_exp_format(buf) &&
	5 == sscanf(buf, "%d %le %le %le %d",
		    &ppid,
		    &coords.a,
		    &coords.b,
		    &coords.c,
		    &motion ) ){
      ppid--;
      if( ppid<0 || (int)ivalueE<=ppid ){
	return false;
      }
      // TAPP̋[|eVԍqԍƂĉ
      // ͂ȂTAPPɂ͌fȂ߂̏u
      atoms.vatom.push_back
	( DTAtom(atoms.velement[ppid], coords, motion) );
      atoms.vatom.back().setAsOriginal(n);
    }else{
      return false;
    }
  }

  return true;
}

bool DTTapp::Atoms::save( const DTAtoms& atoms, FILE* fptr )
{
  fprintf(fptr, "!----------- section atoms\n");

  int count =0;
  for( int n=0; n<(int)atoms.vatom.size(); n++ ){
    if( atoms.vatom[n].isCopy() ) continue; // Rs[q
    //    if( atoms.vatom[n].element == 0 ) continue; // _~[q
    count++;
  }

  fprintf(fptr, "%d %d "
	  "# nkd nsi\n", (int)atoms.velement.size(), count );

  for( int n=0; n<(int)atoms.velement.size(); n++ ){
    fprintf(fptr,"%s ", qPrintable(atoms.velement[n].name) );
    fprintf(fptr,"# ppid(%d)\n",n+1);
  }
 
  for( int n=0; n<(int)atoms.vatom.size(); n++ ){
    if( atoms.vatom[n].isCopy() ) continue; // Rs[q
    //    if( atoms.vatom[n].element == 0 ) continue; // _~[q

    fprintf(fptr, "%d %f %f %f %d ",
	    atoms.findPPID(atoms.vatom[n].element.number)+1,
	    atoms.vatom[n].coords.a,
	    atoms.vatom[n].coords.b,
	    atoms.vatom[n].coords.c,
	    atoms.vatom[n].motion );
    fprintf(fptr, "# kd(%d) asi(1:3,%d) mkd(%d)\n",
	    n+1,n+1,n+1);
  }

  return true;
}

DTTapp::Mesh::Mesh( void )
{
  clear();
}

void DTTapp::Mesh::clear( void )
{
  ndx = ndy = ndz = nd_def;
  nk = nk_def;
  nw = nw_def;
  nb1 = nb1_def, nb2 = nb2_def;
  znext = znext_def;
  dspin = dspin_def;
}

bool DTTapp::Mesh::load( FILE* fptr )
{
  char buf[1024];

  if( fgets_comment(buf,sizeof(buf),fptr) &&
      3 == sscanf(buf, "%d %d %d", &ndx, &ndy, &ndz ) &&
      fgets_comment(buf,sizeof(buf),fptr) &&
      change_exp_format(buf) &&
      6 == sscanf(buf, "%d %d %d %d %le %le",
		  &nk, &nw, &nb1, &nb2, &znext, &dspin ) ){
  }else{
    return false;
  }
  return true;
}

bool DTTapp::Mesh::save( FILE* fptr )
{
  fprintf(fptr, "!----------- section mesh\n");

  fprintf(fptr, "%d %d %d "
	  "# ndx ndy ndz\n",
	  ndx, ndy, ndz );
  fprintf(fptr, "%d %d %d %d %f %f "
	  "# nk nw nb1 nb2 znext dspin\n",
	  nk, nw, nb1, nb2, znext, dspin );
  return true;
}

DTTapp::Ewald::Ewald( void )
{
  clear();
}

void DTTapp::Ewald::clear( void )
{
  qgb = qgb_def;
  qrb = qrb_def;
  g   = g_def;
}


bool DTTapp::Ewald::load( FILE* fptr )
{
  char buf[1024];

  if( fgets_comment(buf,sizeof(buf),fptr) &&
      change_exp_format(buf) &&
      2 == sscanf(buf, "%le %le", &qgb, &qrb ) &&
      fgets_comment(buf,sizeof(buf),fptr) &&
      change_exp_format(buf) &&
      1 == sscanf(buf, "%le", &g ) ){
  }else{
    return false;
  }
  return true;
}

bool DTTapp::Ewald::save( FILE* fptr )
{
  fprintf(fptr, "!----------- section ewald\n");

  fprintf(fptr, "%f %f "
	  "# qgb qrb\n", qgb, qrb );
  fprintf(fptr, "%f "
	  "# g\n", g );

  return true;
}


DTTapp::Sampling::Sampling( void )
{
  vmesh.push_back("manual");
  vmesh.push_back("(1x1x1)");
  vmesh.push_back("(2x2x2)");
  vmesh.push_back("(3x3x3)");
  vmesh.push_back("(4x4x4)");
  vmesh.push_back("(5x5x5)");
  vmesh.push_back("(6x6x6)");
  vmesh.push_back("(7x7x7)");
  vmesh.push_back("(8x8x8)");
  vmesh.push_back("(9x9x9)");
  vmesh.push_back("(10x10x10)");

  clear();
}

void DTTapp::Sampling::clear( void )
{
  imesh = 0;
  gamma = true;

  mmm1.clear();
  mmm2.clear();
  mmm1.push_back( int3(2,2,2) );
  mmm2.push_back( int3(2,2,2) );
  index_shown = 1;
  update_shown();

  nkrnew = nkrnew_def;
  ll1 = ll2 = ll3 = ll_def;

  setByDetail(DTTapp::Mesh::nk_def);
}

bool DTTapp::Sampling::load( FILE* fptr, const int nk )
{
  mmm1.clear();
  mmm2.clear();
  index_shown = 1;
  imesh = 0;

  char buf[1024];

  int3 mmm;
  while(true){
    if( fgets_comment(buf,sizeof(buf),fptr) &&
	3 == sscanf(buf, "%d %d %d", &mmm[0], &mmm[1], &mmm[2] ) ){
      if( mmm[0] == 0 && mmm[1] == 0 && mmm[2] == 0 ) break;
      mmm1.push_back(mmm);
    }
    else{
      return false;
    }

    if( fgets_comment(buf,sizeof(buf),fptr) &&
	 3 == sscanf(buf, "%d %d %d", &mmm[0], &mmm[1], &mmm[2] ) ){
      mmm2.push_back(mmm);
    }
    else{
      return false;
    }
  }

  setByDetail(nk);

  if( fgets_comment(buf,sizeof(buf),fptr) &&
      (3 == sscanf(buf, "%d %d %d", &ll1, &ll2, &ll3 ) ||
       1 == sscanf(buf, "%d", &nkrnew )) ){

  }
  else{
    return false;
  }
  return true;
}

bool DTTapp::Sampling::save( FILE* fptr )
{
  fprintf(fptr, "!----------- section sampling\n");

  for( int n=0; n<(int)mmm1.size(); n++ ){
    fprintf(fptr, "%d %d %d ",
	    mmm1[n][0], mmm1[n][1], mmm1[n][2] );
    fprintf(fptr, "# mmm(1:3,1,%d)\n",n+1);
    fprintf(fptr, "%d %d %d ",
	    mmm2[n][0], mmm2[n][1], mmm2[n][2] );
    fprintf(fptr, "# mmm(1:3,2,%d)\n",n+1);
  }
  fprintf(fptr, "%d %d %d # mmm end\n", 0, 0, 0 );
  if( ll1 == 0 ){
    fprintf(fptr, "%d # nkrnew\n", nkrnew );
  }else{
    fprintf(fptr, "%d %d %d # ll1 ll2 ll3\n", ll1, ll2, ll3 );
  }

  return true;
}

void DTTapp::Sampling::change_shown( void )
{
  if( mmm1.empty() ) return;
  if( index_shown > (int)mmm1.size() ) index_shown = 1;

  mmm1[index_shown-1] = mmm1_shown;
  mmm2[index_shown-1] = mmm2_shown;
}

void DTTapp::Sampling::update_shown( void )
{
  if( mmm1.empty() ) return;
  if( index_shown > (int)mmm1.size() ) index_shown = 1;
    
  mmm1_shown  = mmm1[index_shown-1];
  mmm2_shown  = mmm2[index_shown-1];
}

int DTTapp::Sampling::size_shown( void )
{
  return (int)mmm1.size();
}

static int GCD( int a, int b )
{
  int na, nb, nr;

  na = a;
  nb = b;

  while( nb>0 ){
    nr = na%nb;
    na = nb;
    nb = nr;
  }

  return na;
}

static int LCM( int a, int b )
{
  int g = GCD(a,b);
  return (a/g)*(b/g)*g;
}

static int LCM( int a, int b, int c )
{
  return LCM(LCM(a,b),c);
}

void DTTapp::Sampling::setByOutline( int& nk )
{
  if( imesh == 0 ) return;

  mmm1.clear();
  mmm2.clear();
  index_shown = 1;

  mesh[0] = imesh;
  mesh[1] = imesh;
  mesh[2] = imesh;

  nk = LCM(2*mesh[0],2*mesh[1],2*mesh[2]);

  int3 m1;
  for( int i=0; i<3; i++ ){
    m1[i] = ( gamma ?
	      (mesh[i]%2==0 ? mesh[i]+0 : mesh[i]+1) :
	      (mesh[i]%2==0 ? mesh[i]+1 : mesh[i]+0)) *nk/(2*mesh[i]);
  }

  int3 m2 = int3(2,2,2);

  mmm1.push_back(m1);
  mmm2.push_back(m2);
}

void DTTapp::Sampling::setByDetail( const int nk )
{
  imesh = 0;

  mesh = int3(0,0,0);

  int3 g = int3(0,0,0);

  for( int i=0; i<3; i++ ){
    for( int m=-mmm1_shown[i]; m<=+mmm1_shown[i]; m+=mmm2_shown[i] ){
      if( m<-nk/2 || +nk/2<m ) continue;
      mesh[i]++;
      if( m==0 ) g[i] = 1;
    }
  }

  gamma = g[0] && g[1] && g[2];
}

void DTTapp::Sampling::addMMM( void )
{
  mmm1.push_back( mmm1.back() );
  mmm2.push_back( mmm2.back() );
  index_shown = (int)mmm1.size();
  update_shown();
}

void DTTapp::Sampling::delMMM( void )
{
  if( (int)mmm1.size()<=1 ) return;
  if( index_shown > (int)mmm1.size() ) index_shown = 1;
  mmm1.erase( mmm1.begin()+index_shown-1 );
  mmm2.erase( mmm2.begin()+index_shown-1 );
  update_shown();
}


DTTapp::File::File( void )
{
  clear();
}

void DTTapp::File::clear( void )
{
  iopt1 = iopt1_def;
  iopt2 = iopt2_def;
  iopt3 = iopt3_def;
  iopt4 = iopt4_def;
  iopt5 = iopt5_def;
  iopt6 = iopt6_def;
}

bool DTTapp::File::load( FILE* fptr )
{
  char buf[1024];

  int bvalue1, bvalue2, bvalue3, bvalue4, bvalue5, bvalue6;
  
  if( fgets_comment(buf,sizeof(buf),fptr) ){
    if( 6 == sscanf(buf, "%d %d %d %d %d %d",
		    &bvalue1, &bvalue2, &bvalue3, &bvalue4,
		    &bvalue5, &bvalue6 ) ){
      iopt1 = (bvalue1 != 0);
      iopt2 = (bvalue2 != 0);
      iopt3 = (bvalue3 != 0);
      iopt4 = (bvalue4 != 0);
      iopt5 = (bvalue5 != 0);
      iopt6 = (bvalue6 != 0);
    }
    else if( 4 == sscanf(buf, "%d %d %d %d",
		    &bvalue1, &bvalue2, &bvalue3, &bvalue4 ) ){
      iopt1 = (bvalue1 != 0);
      iopt2 = (bvalue2 != 0);
      iopt3 = (bvalue3 != 0);
      iopt4 = (bvalue4 != 0);
      iopt5 = false;
      iopt6 = false;
    }
    else{
      return false;
    }
  }
  else{
    return false;
  }

  return true;
}

bool DTTapp::File::save( FILE* fptr )
{
  fprintf(fptr, "!----------- section file\n");

  fprintf(fptr, "%d %d %d %d %d %d # iopt(1:6)\n",
	  (int)iopt1, (int)iopt2,
	  (int)iopt3, (int)iopt4, (int)iopt5, (int)iopt6 );

  return true;
}


DTTapp::Mass::Mass( void )
{
  clear();
}

void DTTapp::Mass::clear( void )
{
  tim.clear();
  tim.push_back( Matrix::getIdentity() );
  index_shown = 1;
}


void DTTapp::Mass::add( void )
{
  tim.push_back( tim.back() );
  index_shown = (int)tim.size();
  update_shown();
}

void DTTapp::Mass::del( void )
{
  if( (int)tim.size()<=1 ) return;
  if( index_shown > (int)tim.size() ) index_shown = 1;
  tim.erase( tim.begin()+index_shown-1 );
  update_shown();
}

bool DTTapp::Mass::load( FILE* fptr )
{
  char buf[1024];

  int ivalue;

  if( fgets_comment(buf,sizeof(buf),fptr) &&
      1 == sscanf(buf, "%d", &ivalue ) ){
    tim.resize(ivalue);
  }
  else{
    return false;
  }

  for( int n=0; n<(int)tim.size(); n++ ){
    for( int i=0; i<3; i++ ){
      if( fgets_comment(buf,sizeof(buf),fptr) &&
	  change_exp_format(buf) &&
	  3 == sscanf(buf, " %le %le %le", &tim[n](0,i),
		      &tim[n](1,i), &tim[n](2,i) ) ){
      }
      else{
	return false;
      }
    }
  }
  return true;
}
	
bool DTTapp::Mass::save( FILE* fptr )
{
  fprintf(fptr, "!----------- section mass\n");

  fprintf(fptr, "%d # nmkd\n", (int)tim.size() );
  for( int n=0; n<(int)tim.size(); n++ ){
    for( int i=0; i<3; i++ ){
      for( int j=0; j<3; j++ ){
	fprintf(fptr, "%f ", tim[n](j,i) );
      }
      fprintf(fptr, "# tim(1:3,%d,%d)\n",i+1,n+1);
    }
  }

  return true;
}

void DTTapp::Mass::change_shown( void )
{
  if( tim.empty() ) return;
  if( index_shown > (int)tim.size() ) index_shown = 1;

  tim[index_shown-1] = tim_shown;
}

void DTTapp::Mass::update_shown( void )
{
  if( tim.empty() ) return;
  if( index_shown > (int)tim.size() ) index_shown = 1;
    
  tim_shown  = tim[index_shown-1];
}

int DTTapp::Mass::size_shown( void )
{
  return (int)tim.size();
}

DTTapp::Parameter::Parameter( void )
{
  clear();
}

void DTTapp::Parameter::clear( void )
{
  eps    = eps_def;
  eepsa  = eepsa_def;
  feps   = feps_def;
  ekbt   = ekbt_def;
  decr   = decr_def;
  okatom = okatom_def;
  uptime = uptime_def;
  niter0 = niter0_def;
  niter1 = niter1_def;
  itsb   = itsb_def;
  ndiag0 = ndiag0_def;
  ndiag1 = ndiag1_def;
  ncycl  = ncycl_def;
  mrfr   = mrfr_def;
  most   = most_def;
}

bool DTTapp::Parameter::load( FILE* fptr )
{
  char buf[1024];

  if( fgets_comment(buf,sizeof(buf),fptr) &&
      change_exp_format(buf) &&
      4 == sscanf(buf, "%le %le %le %le",
		  &eps, &eepsa,
		  &feps, &ekbt ) &&
      fgets_comment(buf,sizeof(buf),fptr) &&
      change_exp_format(buf) &&
      3 == sscanf(buf, "%le %le %le",
		  &decr, &okatom, &uptime ) &&
      fgets_comment(buf,sizeof(buf),fptr) &&
      5 == sscanf(buf, "%d %d %d %d %d",
		  &niter0, &niter1, &itsb,
		  &ndiag0, &ndiag1 ) &&
      fgets_comment(buf,sizeof(buf),fptr) &&
      3 == sscanf(buf, "%d %d %d",
		  &ncycl, &mrfr, &most ) ){
  }
  else{
    return false;
  }

  return true;
}


bool DTTapp::Parameter::save( FILE* fptr )
{
  fprintf(fptr, "!----------- section parameter\n");

  fprintf(fptr, "%11.3e %11.3e %11.3e %11.3e "
	  "# eps eepsa feps ekbt\n",
	  eps, eepsa, feps, ekbt );
  fprintf(fptr, "%f %f %11.3e "
	  "# decr okatom uptime\n",
	  decr, okatom, uptime );
  fprintf(fptr, "%d %d %d %d %d "
	  "# niter0 niter1 itsb ndiag0 ndiag1\n",
	  niter0, niter1, itsb, ndiag0, ndiag1 );
  fprintf(fptr, "%d %d %d "
	  "# ncycl mrfr most\n",
	  ncycl, mrfr, most );

  return true;
}

DTTapp::Polar::Polar( void )
{
  clear();
}

void DTTapp::Polar::clear( void )
{
  vkd.clear();
  index_kd_shown = 0;

  vit.clear();
  index_it_shown = 0;
}

bool DTTapp::Polar::load( FILE* fptr )
{
  char buf[1024];

  int ivalue;

  if( fgets_comment(buf,sizeof(buf),fptr) &&
      1 == sscanf(buf, "%d", &ivalue ) ){
    vkd.resize(ivalue);
    index_kd_shown = 1;
  }
  else{
    return false;
  }

  for( int n=0; n<(int)vkd.size(); n++ ){
    if( fgets_comment(buf,sizeof(buf),fptr) &&
	change_exp_format(buf) &&
	3 == sscanf(buf, " %d %le %le",
		    &vkd[n].pol, &vkd[n].pol1, &vkd[n].pol2 ) ){
    }
    else{
      return false;
    }
  }

  if( fgets_comment(buf,sizeof(buf),fptr) &&
      1 == sscanf(buf, "%d", &ivalue ) ){
    vit.resize(ivalue);
    index_it_shown = 1;
  }
  else{
    return false;
  }

  for( int n=0; n<(int)vit.size(); n++ ){
    if( fgets_comment(buf,sizeof(buf),fptr) &&
	change_exp_format(buf) &&
	3 == sscanf(buf, " %d %le %le",
		    &vit[n].pol, &vit[n].asi1, &vit[n].asi2 ) ){
    }
    else{
      return false;
    }
  }

  return true;
}

bool DTTapp::Polar::save( FILE* fptr )
{
  fprintf(fptr, "!----------- section polar\n");

  fprintf(fptr, "# initial-spin\n");

  fprintf(fptr, "%d # nkd_pol\n",  (int)vkd.size() );
  for( int n=0; n<(int)vkd.size(); n++ ){
    fprintf(fptr, "%d %f %f ",
	    vkd[n].pol, vkd[n].pol1, vkd[n].pol2 );
    fprintf(fptr, "# kd_pol(%d) pol_kd(1,%d) pol_kd(2,%d)\n",
	    n+1,n+1,n+1);
  }

  fprintf(fptr, "%d # nsi_pol\n",  (int)vit.size() );
  for( int n=0; n<(int)vit.size(); n++ ){
    fprintf(fptr, "%d %f %f ",
	    vit[n].pol, vit[n].asi1, vit[n].asi2 );
    fprintf(fptr, "# it_pol(%d) pol_asi(1,%d) pol_asi(2,%d)\n",
	    n+1,n+1,n+1);
  }

  return true;
}


void DTTapp::Polar::addKD( void )
{
  if( vkd.empty() ){
    vkd.push_back( kd() );
  }else{
    vkd.push_back( vkd.back() );
  }
  index_kd_shown = (int)vkd.size();
  update_shown();
}

void DTTapp::Polar::delKD( void )
{
  if( (int)vkd.size()<=1 ) return;
  if( index_kd_shown > (int)vkd.size() ) index_kd_shown = 1;
  vkd.erase( vkd.begin()+index_kd_shown-1 );
  update_shown();
}

void DTTapp::Polar::addIT( void )
{
  if( vit.empty() ){
    vit.push_back( it() );
  }else{
    vit.push_back( vit.back() );
  }
  index_it_shown = (int)vit.size();
  update_shown();
}

void DTTapp::Polar::delIT( void )
{
  if( (int)vit.size()<=1 ) return;
  if( index_it_shown > (int)vit.size() ) index_it_shown = 1;
  vit.erase( vit.begin()+index_it_shown-1 );
  update_shown();
}

void DTTapp::Polar::change_shown( void )
{
  if( !vkd.empty() ){
    if( index_kd_shown > (int)vkd.size() ) index_kd_shown = 1;
    vkd[index_kd_shown-1] = kd_shown;
  }

  if( !vit.empty() ){
    if( index_it_shown > (int)vit.size() ) index_it_shown = 1;
    vit[index_it_shown-1] = it_shown;
  }
}

void DTTapp::Polar::update_shown( void )
{
  if( !vkd.empty() ){
    if( index_kd_shown > (int)vkd.size() ) index_kd_shown = 1;
    kd_shown = vkd[index_kd_shown-1];
  }

  if( !vit.empty() ){
    if( index_it_shown > (int)vit.size() ) index_it_shown = 1;
    it_shown = vit[index_it_shown-1];
  }
}

int DTTapp::Polar::size_shown_kd( void )
{
  return (int)vkd.size();
}
int DTTapp::Polar::size_shown_it( void )
{
  return (int)vit.size();
}


DTTapp::VBPEF::VBPEF( void )
{
  clear();
}

void DTTapp::VBPEF::clear( void )
{
  ifbunp = ifbunp_def;
  ifbfix = ifbfix_def;
  nw     = nw_def;
  nb2    = nb2_def;
  itsb   = itsb_def;
  ndiag  = ndiag_def;
  eps    = eps_def;
  nbba1  = nbba_def;
  nbba2  = nbba_def;
  nsph   = nsph_def;
  iaxis  = iaxis_def;
  rzero  = Position( rzero_def, rzero_def, rzero_def );
  rmin   = rmin_def;
  rmax   = rmax_def;
  ndiv   = ndiv_def;
}

bool DTTapp::VBPEF::load( DTCell& cell, FILE* fptr )
{
  char buf[1024];
  int  number_trace_block;
  int  number_band_traced;

  ifbunp=0;
  if( fgets_comment(buf,sizeof(buf),fptr) &&
      (3 == sscanf(buf, "%d %d %d", &number_trace_block, &ifbunp, &ifbfix ) ||
       2 == sscanf(buf, "%d %d", &number_trace_block, &ifbunp ) ||
       1 == sscanf(buf, "%d", &number_trace_block )) ){
  }
  else{
    return false;
  }
  number_band_traced = number_trace_block + 1;

  cell.vsymmLselected.resize(number_trace_block);
  cell.vsymmLselected_loaded = true;

  if( fgets_comment(buf,sizeof(buf),fptr) ){
    char* ptr = buf;
    int   offset;
    char svalue[32];

    for( int i=0; i<number_band_traced; i++ ){
      if( 1 == sscanf(ptr, " '%[^']'%n", svalue, &offset ) ||
	  1 == sscanf(ptr, " %s%n", svalue, &offset ) ){
	ptr += offset;
	if( i+1<number_band_traced ){
	  cell.vsymmLselected[i].labelPs = svalue;
	}
	if( i>0 ){
	  cell.vsymmLselected[i-1].labelPe = svalue;
	}
      }
      else{
	MyException::critical("broken vband nkpt.");
	return false;
      }
    }
  }
  else{
    return false;
  }

  if( fgets_comment(buf,sizeof(buf),fptr) && change_exp_format(buf) ){
    char* ptr = buf;
    int   offset;
    double x;

    for( int i=0; i<number_band_traced; i++ ){
      if( 1 == sscanf(ptr, "%le%n", &x, &offset ) ){
	ptr += offset;
	if( i+1<number_band_traced ){
	  cell.vsymmLselected[i].positions.x = x;
	}
	if( i>0 ){
	  cell.vsymmLselected[i-1].positione.x = x;
	}
      }
      else{
	MyException::critical("broken vband ak x.");
	return false;
      }
    }
  }
  else{
    return false;
  }

  if( fgets_comment(buf,sizeof(buf),fptr) && change_exp_format(buf) ){
    char* ptr = buf;
    int   offset;
    double y;

    for( int i=0; i<number_band_traced; i++ ){
      if( 1 == sscanf(ptr, "%le%n", &y, &offset ) ){
	ptr += offset;
	if( i+1<number_band_traced ){
	  cell.vsymmLselected[i].positions.y = y;
	}
	if( i>0 ){
	  cell.vsymmLselected[i-1].positione.y = y;
	}
      }
      else{
	MyException::critical("broken vband ak y.");
	return false;
      }
    }
  }
  else{
    return false;
  }



  if( fgets_comment(buf,sizeof(buf),fptr) && change_exp_format(buf) ){
    char* ptr = buf;
    int   offset;
    double z;

    for( int i=0; i<number_band_traced; i++ ){
      if( 1 == sscanf(ptr, "%le%n", &z, &offset ) ){
	ptr += offset;
	if( i+1<number_band_traced ){
	  cell.vsymmLselected[i].positions.z = z;
	}
	if( i>0 ){
	  cell.vsymmLselected[i-1].positione.z = z;
	}
      }
      else{
	MyException::critical("broken vband ak z.");
	return false;
      }
    }
  }
  else{
    return false;
  }

  if( fgets_comment(buf,sizeof(buf),fptr) && change_exp_format(buf) ){
    char* ptr = buf;
    int   offset;
    int   npoint;

    for( int i=0; i<number_trace_block; i++ ){
      if( 1 == sscanf(ptr, "%d%n", &npoint, &offset ) ){
	ptr += offset;
	cell.vsymmLselected[i].npoint = npoint;
	if( i+1==number_trace_block ){
	  cell.vsymmLselected[i+1].npoint = 0;
	}
      }
      else{
	MyException::critical("broken vband ak nkfi.");
	return false;
      }
    }
  }
  else{
    return false;
  }

  // delete jump
  for( int i=0; i<(int)cell.vsymmLselected.size(); i++ ){
    if( cell.vsymmLselected[i].labelPs 
	== cell.vsymmLselected[i].labelPe ){
      cell.vsymmLselected.erase( cell.vsymmLselected.begin()+i );
    } 
  }

  //

  if( fgets_comment(buf,sizeof(buf),fptr) &&
      2 == sscanf(buf, "%d %d", &nw, &nb2 ) ){
  }
  else{
    return false;
  }
  if( fgets_comment(buf,sizeof(buf),fptr) &&
      change_exp_format(buf) &&
      3 == sscanf(buf, "%d %d %le",
		  &itsb, &ndiag, &eps ) ){
  }
  else{
    return false;
  }

  nbba1 = 1;
  nbba2 = nw;

  switch( ifbunp ){
  case 0: {
  } break;
  case 1: {
    if( fgets_comment(buf,sizeof(buf),fptr) &&
	2 == sscanf(buf, "%d %d", &nbba1, &nbba2 ) ){
    }
    else{
      return false;
    }
    if( fgets_comment(buf,sizeof(buf),fptr) &&
	1 == sscanf(buf, "%d", &nsph ) ){
    }
    else{
      return false;
    }
    if( fgets_comment(buf,sizeof(buf),fptr) &&
	change_exp_format(buf) &&
	4 == sscanf(buf, "%d %le %le %d",
		    &iaxis, &rmin, &rmax, &ndiv ) ){
    }
    else{
      return false;
    }
  } break;
  case 2: {
    if( fgets_comment(buf,sizeof(buf),fptr) &&
	2 == sscanf(buf, "%d %d", &nbba1, &nbba2 ) ){
    }
    else{
      return false;
    }
    if( fgets_comment(buf,sizeof(buf),fptr) &&
	1 == sscanf(buf, "%d", &nsph ) ){
    }
    else{
      return false;
    }
    if( fgets_comment(buf,sizeof(buf),fptr) &&
	change_exp_format(buf) &&
	6 == sscanf(buf, "%le %le %le %le %le %d",
		    &rzero.x, &rzero.y, &rzero.z,
		    &rmin, &rmax, &ndiv ) ){
    }
    else{
      return false;
    }
  } break;
  }

  return true;
}


bool DTTapp::VBPEF::save( const DTCell& cell, FILE* fptr )
{
  fprintf(fptr, "!----------- section vbpef\n");

  fprintf(fptr, "# vbpef data\n" );


  Position last;
  {
    int number_band_traced = 0;
    for( int i=0; i<(int)cell.vsymmLselected.size(); i++ ){
      const BrillouinSegment& symmL = cell.vsymmLselected[i];

      if( i==0 ){
	number_band_traced++;
	number_band_traced++;
      }
      else{
	if( last != symmL.positions ){
	  number_band_traced++;
	}
	number_band_traced++;
      }
      last = symmL.positione;
    }
    fprintf(fptr, "%d %d %d # nbk ifbunp ifbfix\n",
	    number_band_traced, ifbunp, ifbfix );

  }
  fprintf(fptr, "/\n");



  for( int i=0; i<(int)cell.vsymmLselected.size(); i++ ){
    const BrillouinSegment& symmL = cell.vsymmLselected[i];

    if( i==0 ){
      fprintf(fptr, "%-5s ", qPrintable(("'"+symmL.labelPs+"'")) ); 
      fprintf(fptr, "%-5s ", qPrintable(("'"+symmL.labelPe+"'")) ); 
    }
    else{
      if( last != symmL.positions ){
	fprintf(fptr, "%-5s ", qPrintable(("'"+symmL.labelPs+"'")) ); 
      }
      fprintf(fptr, "%-5s ", qPrintable(("'"+symmL.labelPe+"'")) ); 
    }
    last = symmL.positione;
  }
  fprintf(fptr, "# nkpt(1:nbk+1)\n");

  for( int i=0; i<(int)cell.vsymmLselected.size(); i++ ){
    const BrillouinSegment& symmL = cell.vsymmLselected[i];

    if( i==0 ){
      fprintf(fptr, "%5.3f ", symmL.positions.x ); 
      fprintf(fptr, "%5.3f ", symmL.positione.x ); 
    }
    else{
      if( last != symmL.positions ){
	fprintf(fptr, "%5.3f ", symmL.positions.x ); 
      }
      fprintf(fptr, "%5.3f ", symmL.positione.x ); 
    }
    last = symmL.positione;
  }
  fprintf(fptr, "# ak(1,1:nbk+1)\n");

  for( int i=0; i<(int)cell.vsymmLselected.size(); i++ ){
    const BrillouinSegment& symmL = cell.vsymmLselected[i];

    if( i==0 ){
      fprintf(fptr, "%5.3f ", symmL.positions.y ); 
      fprintf(fptr, "%5.3f ", symmL.positione.y ); 
    }
    else{
      if( last != symmL.positions ){
	fprintf(fptr, "%5.3f ", symmL.positions.y ); 
      }
      fprintf(fptr, "%5.3f ", symmL.positione.y ); 
    }
    last = symmL.positione;
  }
  fprintf(fptr, "# ak(2,1:nbk+1)\n");

  for( int i=0; i<(int)cell.vsymmLselected.size(); i++ ){
    const BrillouinSegment& symmL = cell.vsymmLselected[i];

    if( i==0 ){
      fprintf(fptr, "%5.3f ", symmL.positions.z ); 
      fprintf(fptr, "%5.3f ", symmL.positione.z ); 
    }
    else{
      if( last != symmL.positions ){
	fprintf(fptr, "%5.3f ", symmL.positions.z ); 
      }
      fprintf(fptr, "%5.3f ", symmL.positione.z ); 
    }
    last = symmL.positione;
  }
  fprintf(fptr, "# ak(3,1:nbk+1)\n");

  fprintf(fptr, "     ");
  for( int i=0; i<(int)cell.vsymmLselected.size(); i++ ){
    const BrillouinSegment& symmL = cell.vsymmLselected[i];

    if( i==0 ){
      fprintf(fptr, "%-5d ", symmL.npoint ); 
    }
    else{
      if( last != symmL.positions ){
	fprintf(fptr, "%-5d ", 0 );
      }
      fprintf(fptr, "%-5d ", symmL.npoint );
    }
    last = symmL.positione;
  }
  fprintf(fptr, " # nkfi(1:nbk)\n");


  fprintf(fptr, "%d %d # nw nb2\n",
	  nw, nb2 );
  fprintf(fptr, "%d %d %11.3e # itsb ndiag eps\n",
	  itsb, ndiag, eps );
  fprintf(fptr, "%d %d # nbba1 nbba2\n",
	  nbba1, nbba2 );

  switch( ifbunp ){
  case 0: {
  } break;
  case 1: {
    fprintf(fptr, "%d # nsph\n", nsph );
    fprintf(fptr, "%d %f %f %d # iaxis, rmin, rmax, ndiv\n",
	    iaxis, rmin, rmax, ndiv );
  } break;
  case 2: {
    fprintf(fptr, "%d # nsph\n", nsph );
    fprintf(fptr, "%f %f %f %f %f %d "
	    "# rzero(1:3) rmin rmax ndiv\n",
	    rzero.x, rzero.y, rzero.z, rmin, rmax, ndiv );
  } break;
  }

  return true;
}


//-----------
DTTapp::VBWFN::VBWFN( void )
{
  clear();
}

void DTTapp::VBWFN::clear( void )
{
  vnb.clear();
  index_shown = 0;
  nraxn = nrayn = nrazn = 0;
  sqamp = false;
  wfn = false;
  sk[0] = sk[1] = sk[2] = 0.0;
  itsb = 0;
  ndiag = 0;
  eps = 0.0;
}

bool DTTapp::VBWFN::load( FILE* fptr )
{
  char buf[1024];

  int ivalue;

  if( fgets_comment(buf,sizeof(buf),fptr) &&
      3 == sscanf(buf, "%d %d %d", &nraxn, &nrayn, &nrazn ) ){
  }
  else{
    return false;
  }

  int bvalue1, bvalue2;
  if( fgets_comment(buf,sizeof(buf),fptr) &&
      change_exp_format(buf) &&
      5 == sscanf(buf, "%d %d %le %le %le",
		  &bvalue1, &bvalue2, &sk[0],&sk[1],&sk[2] ) ){
    sqamp = bvalue1 != 0;
    wfn   = bvalue2 != 0;
  }
  else{
    return false;
  }

  if( fgets_comment(buf,sizeof(buf),fptr) &&
      1 == sscanf(buf, "%d", &ivalue ) ){
    vnb.resize(ivalue);
    index_shown = 1;
  }
  else{
    return false;
  }

  for( int n=0; n<(int)vnb.size(); n++ ){
    if( fgets_comment(buf,sizeof(buf),fptr) &&
	2 == sscanf(buf, " %d %d",
		    &vnb[n].nbb1, &vnb[n].nbb2 ) ){
    }
    else{
      return false;
    }
  }

  if( fgets_comment(buf,sizeof(buf),fptr) &&
      change_exp_format(buf) &&
      3 == sscanf(buf, "%d %d %le",
		  &itsb, &ndiag, &eps ) ){
  }
  else{
    return false;
  }

  return true;
}

bool DTTapp::VBWFN::save( FILE* fptr )
{
  fprintf(fptr, "!----------- section vbwfn\n");

  fprintf(fptr, "# vbwfn data\n");

  fprintf(fptr, "%d %d %d "
	  "# nraxn nraxn nrazn\n",
	  nraxn, nraxn, nrazn );
  fprintf(fptr, "%d %d %f %f %f "
	  "# if_sqamp if_wfn sk(1:3)\n",
	  (int)sqamp, (int)wfn, sk[0], sk[1], sk[2] );
  
  fprintf(fptr, "%d # nblk\n", (int)vnb.size() );
  for( int n=0; n<(int)vnb.size(); n++ ){
    fprintf(fptr, "%d %d ",
	    vnb[n].nbb1, vnb[n].nbb2 );
    fprintf(fptr, "# nbb1(%d) nbb2(%d)\n",n+1,n+1);
  }
  fprintf(fptr, "%d %d %11.3e # itsb ndiag eps\n",
	  itsb, ndiag, eps );

  return true;
}

void DTTapp::VBWFN::add( void )
{
  if( vnb.empty() ){
    vnb.push_back( nb() );
  }else{
    vnb.push_back( vnb.back() );
  }
  index_shown = (int)vnb.size();
  update_shown();
}

void DTTapp::VBWFN::del( void )
{
  if( (int)vnb.size()<=1 ) return;
  if( index_shown > (int)vnb.size() ) index_shown = 1;
  vnb.erase( vnb.begin()+index_shown-1 );
  update_shown();
}

void DTTapp::VBWFN::change_shown( void )
{
  if( !vnb.empty() ){
    if( index_shown > (int)vnb.size() ) index_shown = 1;
    vnb[index_shown-1] = nb_shown;
  }
}

void DTTapp::VBWFN::update_shown( void )
{
  if( !vnb.empty() ){
    if( index_shown > (int)vnb.size() ) index_shown = 1;
    nb_shown = vnb[index_shown-1];
  }
}

int DTTapp::VBWFN::size_shown( void )
{
  return (int)vnb.size();
}


DTTapp::TCHRPOT::TCHRPOT( void )
{
  clear();
}

void DTTapp::TCHRPOT::clear( void )
{
  nraxn=nrayn=nrazn=0;
  tchr = false;
  lpot = false;
}

bool DTTapp::TCHRPOT::load( FILE* fptr )
{
  char buf[1024];

  if( fgets_comment(buf,sizeof(buf),fptr) &&
      3 == sscanf(buf, "%d %d %d", &nraxn, &nrayn, &nrazn ) ){
  }
  else{
    return false;
  }

  int bvalue1, bvalue2;
  if( fgets_comment(buf,sizeof(buf),fptr) &&
      2 == sscanf(buf, "%d %d", &bvalue1, &bvalue2 ) ){
    tchr = bvalue1 != 0;
    lpot = bvalue2 != 0;
  }
  else{
    return false;
  }

  return true;
}

bool DTTapp::TCHRPOT::save( FILE* fptr )
{
  fprintf(fptr, "!----------- section tchrpot\n");

  fprintf(fptr, "# tchrpot data\n");

  fprintf(fptr, "%d %d %d # nraxn, nraxn, nrazn\n",
	  nraxn, nraxn, nrazn );
  fprintf(fptr, "%d %d # if_tchr if_lpot\n", (int)tchr, (int)lpot );

  return true;
}



//----------------------------------------------------
bool DTTapp::guess( const QString& fname )
{
  FILE* fptr = fopen( fname, "rb" );
  if( fptr == NULL ){
    throw MyException("can not open a file.",fname);
  }

  char buf[256];
  int  ivalue;

  bool match = false;
  while( fgets(buf,sizeof(buf),fptr) ){
    if( buf[0] == '#' || buf[0] == '!' ) continue;

    if( 7 == sscanf(buf,"%d %d %d %d %d %d %d",
		    &ivalue, &ivalue, &ivalue, &ivalue, 
		    &ivalue, &ivalue, &ivalue ) ){
      match = true;
    }
    break;
  }

  fclose(fptr);

  return match;
}

bool DTTapp::load( DTLattice& lattice, const QString& fname )
{
  FILE* fptr = fopen( fname, "rb" );
  if( fptr == NULL ){
    throw MyException("TAPP input file is not found.",fname);
    //    return false;
  }

  clear();

  const char* error_msg = NULL;
  char buf[1024];

  if( fgets(buf,sizeof(buf),fptr) &&
      !strncmp( buf, "#CGM-3.0", 8 ) ){
  }
  else{
    error_msg = "TAPP input file is broken in header section.";
    goto error_block;
  }

  if( !option.load(fptr) ){
    error_msg = "TAPP input file is broken in OPTION section.";
    goto error_block;
  }
  if( !lsda.load(fptr) ){
    error_msg = "TAPP input file is broken in LSDA section.";
    goto error_block;
  }
  if( !cell.load(lattice.cell,fptr) ){
    error_msg = "TAPP input file is broken in CELL section.";
    goto error_block;
  }
  if( !symmetry.load(lattice.symmetry,fptr) ){
    error_msg = "TAPP input file is broken in SYMMETRY section.";
    goto error_block;
  }

  if( !cutoff.load(fptr) ){
    error_msg = "TAPP input file is broken in CUTOFF section.";
    goto error_block;
  }

  if( !atoms.load(lattice.getData(),fptr) ){
    error_msg = "TAPP input file is broken in STRUCTURE section.";
    goto error_block;
  }
    
  if( !mesh.load(fptr) ){
    error_msg = "TAPP input file is broken in MESH section.";
    goto error_block;
  }

  if( !ewald.load(fptr) ){
    error_msg = "TAPP input file is broken in EWALD section.";
    goto error_block;
  }
  if( !sampling.load(fptr,mesh.nk) ){
    error_msg = "TAPP input file is broken in SAMPLING section.";
    goto error_block;
  }
  if( !file.load(fptr) ){
    error_msg = "TAPP input file is broken in FILE section.";
    goto error_block;
  }
  if( !mass.load(fptr) ){
    error_msg = "TAPP input file is broken in MASS section.";
    goto error_block;
  }
  if( !parameter.load(fptr) ){
    error_msg = "TAPP input file is broken in PARAMETER section.";
    goto error_block;
  }

  while( fgets(buf,sizeof(buf),fptr) ){
    if( !strncmp( buf, "# vbpef data", 12 ) ){
      if( !vbpef.load(lattice.cell,fptr) ){
	error_msg = "TAPP input file is broken in VBPEF section.";
	goto error_block;
      }
      continue;
    }

    if( !strncmp( buf, "# initial-spin", 14 ) ){
      if( !polar.load(fptr) ){
	error_msg = "TAPP input file is broken in POLAR section.";
	goto error_block;
      }
      continue;
    }

    if( !strncmp( buf, "# vbwfn data", 12 ) ){
      if( !vbwfn.load(fptr) ){
	error_msg = "TAPP input file is broken in VBWFN section.";
	goto error_block;
      }
      continue;
    }

    if( !strncmp( buf, "# tchrpot data", 14 ) ){
      if( !tchrpot.load(fptr) ){
	error_msg = "TAPP input file is broken in TCHRPOT section.";
	goto error_block;
      }
      continue;
    }
   }

  fclose(fptr);

  update();

  return true;

 error_block:
  fclose(fptr);
  clear();

  throw MyException(error_msg,fname);
  return false; // dummy statement
}


bool DTTapp::save( const DTLattice& lattice, const QString& fname )
{
  FILE* fptr = fopen( fname, "wb" );
  if( fptr == NULL ){
    throw MyException("can not create a config file.",fname);
    //    return false;
  }

  fprintf(fptr, "#CGM-3.0\n");
  fprintf(fptr, "# %s\n", qPrintable(fname) );
  fprintf(fptr, "# output by TAPIOCA\n");
  fprintf(fptr, "#\n");

  option.save(fptr);
  lsda.save(fptr);
  cell.save(lattice.cell,fptr);
  symmetry.save(lattice.symmetry,fptr);
  cutoff.save(fptr);
  atoms.save(lattice.getData(),fptr);
  mesh.save(fptr);
  ewald.save(fptr);
  sampling.save(fptr);
  file.save(fptr);
  mass.save(fptr);
  parameter.save(fptr);
  vbpef.save(lattice.cell,fptr);

  if( polar.isset() ){
    polar.save(fptr);
  }
  if( vbwfn.isset() ){
    vbwfn.save(fptr);
  }
  if( tchrpot.isset() ){
    tchrpot.save(fptr);
  }

  fclose(fptr);
 
  return true;
}

bool DTTapp::loadDefaults( void )
{
  QDomDocument doc;

  if( !XML::load( doc, "defaults.qtml") ) return false;

  QDomNode root = XML::getRoot(doc);
  root = XML::getFirstElementByTagName(root,"TAPP");

  {
    QDomElement group, node;
    group = XML::getFirstElementByTagName(root,"Option");

    node  = XML::getFirstElementByTagName(group,"wfn_on_disk");
    XML::sscanf( XML::getAttribute(node,"default"), "%d", &Option::wfn_on_disk_def );
    node  = XML::getFirstElementByTagName(group,"qg_on_disk");
    XML::sscanf( XML::getAttribute(node,"default"), "%d", &Option::qg_on_disk_def );
    node  = XML::getFirstElementByTagName(group,"fsmear");
    XML::sscanf( XML::getAttribute(node,"default"), "%d", &Option::fsmear_def );
    node  = XML::getFirstElementByTagName(group,"hpc");
    XML::sscanf( XML::getAttribute(node,"default"), "%d", &Option::hpc_def );
  }

  {
    QDomElement group, node;
    group = XML::getFirstElementByTagName(root,"LSDA");

    node  = XML::getFirstElementByTagName(group,"xctype");
    XML::sscanf( XML::getAttribute(node,"default"), "%d", &LSDA::ixctype_def );

    node  = XML::getFirstElementByTagName(group,"nspin");
    XML::sscanf( XML::getAttribute(node,"default"), "%d", &LSDA::nspin_def );

    node  = XML::getFirstElementByTagName(group,"spinmode");
    XML::sscanf( XML::getAttribute(node,"default"), "%d", &LSDA::spinmode_def );

    node  = XML::getFirstElementByTagName(group,"ipmode");
    XML::sscanf( XML::getAttribute(node,"default"), "%d", &LSDA::ipmode_def );

    node  = XML::getFirstElementByTagName(group,"nfixed");
    XML::sscanf( XML::getAttribute(node,"default"), "%d", &LSDA::nfixed_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%d", &LSDA::nfixed_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%d", &LSDA::nfixed_max );
  }

  {
    QDomElement group, node;
    group = XML::getFirstElementByTagName(root,"Cutoff");

    node  = XML::getFirstElementByTagName(group,"qf");
    XML::sscanf( XML::getAttribute(node,"default"), "%lf", &Cutoff::qf_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%lf", &Cutoff::qf_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%lf", &Cutoff::qf_max );

    node  = XML::getFirstElementByTagName(group,"rf");
    XML::sscanf( XML::getAttribute(node,"default"), "%lf", &Cutoff::rf_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%lf", &Cutoff::rf_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%lf", &Cutoff::rf_max );

    node  = XML::getFirstElementByTagName(group,"qm");
    XML::sscanf( XML::getAttribute(node,"default"), "%lf", &Cutoff::qm_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%lf", &Cutoff::qm_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%lf", &Cutoff::qm_max );

    node  = XML::getFirstElementByTagName(group,"qc");
    XML::sscanf( XML::getAttribute(node,"default"), "%lf", &Cutoff::qc_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%lf", &Cutoff::qc_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%lf", &Cutoff::qc_max );

    node  = XML::getFirstElementByTagName(group,"beta");
    XML::sscanf( XML::getAttribute(node,"default"), "%lf", &Cutoff::beta_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%lf", &Cutoff::beta_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%lf", &Cutoff::beta_max );
  }

  {
    QDomElement group, node;
    group = XML::getFirstElementByTagName(root,"Mesh");

    node  = XML::getFirstElementByTagName(group,"nd");
    XML::sscanf( XML::getAttribute(node,"default"), "%d", &Mesh::nd_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%d", &Mesh::nd_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%d", &Mesh::nd_max );

    node  = XML::getFirstElementByTagName(group,"nk");
    XML::sscanf( XML::getAttribute(node,"default"), "%d", &Mesh::nk_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%d", &Mesh::nk_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%d", &Mesh::nk_max );

    node  = XML::getFirstElementByTagName(group,"nw");
    XML::sscanf( XML::getAttribute(node,"default"), "%d", &Mesh::nw_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%d", &Mesh::nw_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%d", &Mesh::nw_max );

    node  = XML::getFirstElementByTagName(group,"nb");
    XML::sscanf( XML::getAttribute(node,"default"), "%d %d",
		 &Mesh::nb1_def, &Mesh::nb2_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%d", &Mesh::nb_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%d", &Mesh::nb_max );

    node  = XML::getFirstElementByTagName(group,"znext");
    XML::sscanf( XML::getAttribute(node,"default"), "%lf", &Mesh::znext_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%lf", &Mesh::znext_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%lf", &Mesh::znext_max );

    node  = XML::getFirstElementByTagName(group,"dspin");
    XML::sscanf( XML::getAttribute(node,"default"), "%lf", &Mesh::dspin_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%lf", &Mesh::dspin_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%lf", &Mesh::dspin_max );
  }

  {
    QDomElement group, node;
    group = XML::getFirstElementByTagName(root,"Ewald");

    node  = XML::getFirstElementByTagName(group,"qgb");
    XML::sscanf( XML::getAttribute(node,"default"), "%lf", &Ewald::qgb_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%lf", &Ewald::qgb_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%lf", &Ewald::qgb_max );

    node  = XML::getFirstElementByTagName(group,"qrb");
    XML::sscanf( XML::getAttribute(node,"default"), "%lf", &Ewald::qrb_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%lf", &Ewald::qrb_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%lf", &Ewald::qrb_max );

    node  = XML::getFirstElementByTagName(group,"g");
    XML::sscanf( XML::getAttribute(node,"default"), "%lf", &Ewald::g_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%lf", &Ewald::g_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%lf", &Ewald::g_max );
  }

  {
    QDomElement group, node;
    group = XML::getFirstElementByTagName(root,"Sampling");

    node  = XML::getFirstElementByTagName(group,"nkrnew");
    XML::sscanf( XML::getAttribute(node,"default"), "%d", &Sampling::nkrnew_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%d", &Sampling::nkrnew_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%d", &Sampling::nkrnew_max );

    node  = XML::getFirstElementByTagName(group,"ll");
    XML::sscanf( XML::getAttribute(node,"default"), "%d", &Sampling::ll_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%d", &Sampling::ll_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%d", &Sampling::ll_max );
  }

  {
    QDomElement group, node;
    group = XML::getFirstElementByTagName(root,"File");

    int bvalue1, bvalue2, bvalue3, bvalue4, bvalue5, bvalue6;
    node  = XML::getFirstElementByTagName(group,"iopt");
    XML::sscanf( XML::getAttribute(node,"default"), "%d %d %d %d %d %d",
		 &bvalue1, &bvalue2, &bvalue3, &bvalue4, &bvalue5, &bvalue6 );

    File::iopt1_def = (bvalue1 != 0);
    File::iopt2_def = (bvalue2 != 0);
    File::iopt3_def = (bvalue3 != 0);
    File::iopt4_def = (bvalue4 != 0);
    File::iopt5_def = (bvalue5 != 0);
    File::iopt6_def = (bvalue6 != 0);
  }

  {
    QDomElement group, node;
    group = XML::getFirstElementByTagName(root,"Parameter");

    node  = XML::getFirstElementByTagName(group,"eps");
    XML::sscanf( XML::getAttribute(node,"default"), "%lf", &Parameter::eps_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%lf", &Parameter::eps_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%lf", &Parameter::eps_max );

    node  = XML::getFirstElementByTagName(group,"eepsa");
    XML::sscanf( XML::getAttribute(node,"default"), "%lf", &Parameter::eepsa_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%lf", &Parameter::eepsa_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%lf", &Parameter::eepsa_max );

    node  = XML::getFirstElementByTagName(group,"feps");
    XML::sscanf( XML::getAttribute(node,"default"), "%lf", &Parameter::feps_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%lf", &Parameter::feps_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%lf", &Parameter::feps_max );

    node  = XML::getFirstElementByTagName(group,"ekbt");
    XML::sscanf( XML::getAttribute(node,"default"), "%lf", &Parameter::ekbt_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%lf", &Parameter::ekbt_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%lf", &Parameter::ekbt_max );

    node  = XML::getFirstElementByTagName(group,"decr");
    XML::sscanf( XML::getAttribute(node,"default"), "%lf", &Parameter::decr_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%lf", &Parameter::decr_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%lf", &Parameter::decr_max );

    node  = XML::getFirstElementByTagName(group,"okatom");
    XML::sscanf( XML::getAttribute(node,"default"), "%lf", &Parameter::okatom_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%lf", &Parameter::okatom_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%lf", &Parameter::okatom_max );

    node  = XML::getFirstElementByTagName(group,"uptime");
    XML::sscanf( XML::getAttribute(node,"default"), "%lf", &Parameter::uptime_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%lf", &Parameter::uptime_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%lf", &Parameter::uptime_max );

    node  = XML::getFirstElementByTagName(group,"niter");
    XML::sscanf( XML::getAttribute(node,"default"), "%d %d",
		 &Parameter::niter0_def, &Parameter::niter1_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%d", &Parameter::niter_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%d", &Parameter::niter_max );

    node  = XML::getFirstElementByTagName(group,"itsb");
    XML::sscanf( XML::getAttribute(node,"default"), "%d", &Parameter::itsb_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%d", &Parameter::itsb_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%d", &Parameter::itsb_max );

    node  = XML::getFirstElementByTagName(group,"ndiag");
    XML::sscanf( XML::getAttribute(node,"default"), "%d %d",
		 &Parameter::ndiag0_def, &Parameter::ndiag1_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%d", &Parameter::ndiag_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%d", &Parameter::ndiag_max );

    node  = XML::getFirstElementByTagName(group,"ncycl");
    XML::sscanf( XML::getAttribute(node,"default"), "%d", &Parameter::ncycl_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%d", &Parameter::ncycl_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%d", &Parameter::ncycl_max );

    node  = XML::getFirstElementByTagName(group,"mrfr");
    XML::sscanf( XML::getAttribute(node,"default"), "%d", &Parameter::mrfr_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%d", &Parameter::mrfr_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%d", &Parameter::mrfr_max );

    node  = XML::getFirstElementByTagName(group,"most");
    XML::sscanf( XML::getAttribute(node,"default"), "%d", &Parameter::most_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%d", &Parameter::most_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%d", &Parameter::most_max );
  }

  {
    QDomElement group, node;
    group = XML::getFirstElementByTagName(root,"VBPEF");

    node  = XML::getFirstElementByTagName(group,"ifbunp");
    XML::sscanf( XML::getAttribute(node,"default"), "%d", &VBPEF::ifbunp_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%d", &VBPEF::ifbunp_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%d", &VBPEF::ifbunp_max );

    node  = XML::getFirstElementByTagName(group,"ifbfix");
    XML::sscanf( XML::getAttribute(node,"default"), "%d", &VBPEF::ifbfix_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%d", &VBPEF::ifbfix_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%d", &VBPEF::ifbfix_max );

    node  = XML::getFirstElementByTagName(group,"nw");
    XML::sscanf( XML::getAttribute(node,"default"), "%d", &VBPEF::nw_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%d", &VBPEF::nw_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%d", &VBPEF::nw_max );

    node  = XML::getFirstElementByTagName(group,"nb2");
    XML::sscanf( XML::getAttribute(node,"default"), "%d", &VBPEF::nb2_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%d", &VBPEF::nb2_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%d", &VBPEF::nb2_max );

    node  = XML::getFirstElementByTagName(group,"itsb");
    XML::sscanf( XML::getAttribute(node,"default"), "%d", &VBPEF::itsb_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%d", &VBPEF::itsb_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%d", &VBPEF::itsb_max );

    node  = XML::getFirstElementByTagName(group,"ndiag");
    XML::sscanf( XML::getAttribute(node,"default"), "%d", &VBPEF::ndiag_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%d", &VBPEF::ndiag_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%d", &VBPEF::ndiag_max );


    node  = XML::getFirstElementByTagName(group,"nbba");

    XML::sscanf( XML::getAttribute(node,"default"), "%d", &VBPEF::nbba_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%d", &VBPEF::nbba_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%d", &VBPEF::nbba_max );

    node  = XML::getFirstElementByTagName(group,"nsph");
    XML::sscanf( XML::getAttribute(node,"default"), "%d", &VBPEF::nsph_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%d", &VBPEF::nsph_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%d", &VBPEF::nsph_max );

    node  = XML::getFirstElementByTagName(group,"iaxis");
    XML::sscanf( XML::getAttribute(node,"default"), "%d", &VBPEF::iaxis_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%d", &VBPEF::iaxis_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%d", &VBPEF::iaxis_max );

    node  = XML::getFirstElementByTagName(group,"ndiv");
    XML::sscanf( XML::getAttribute(node,"default"), "%d", &VBPEF::ndiv_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%d", &VBPEF::ndiv_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%d", &VBPEF::ndiv_max );


    node  = XML::getFirstElementByTagName(group,"eps");
    XML::sscanf( XML::getAttribute(node,"default"), "%lf", &VBPEF::eps_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%lf", &VBPEF::eps_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%lf", &VBPEF::eps_max );

    node  = XML::getFirstElementByTagName(group,"rzero");
    XML::sscanf( XML::getAttribute(node,"default"), "%lf", &VBPEF::rzero_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%lf", &VBPEF::rzero_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%lf", &VBPEF::rzero_max );

    node  = XML::getFirstElementByTagName(group,"rmin");
    XML::sscanf( XML::getAttribute(node,"default"), "%lf", &VBPEF::rmin_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%lf", &VBPEF::rmin_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%lf", &VBPEF::rmin_max );

    node  = XML::getFirstElementByTagName(group,"rmax");
    XML::sscanf( XML::getAttribute(node,"default"), "%lf", &VBPEF::rmax_def );
    XML::sscanf( XML::getAttribute(node,"min"), "%lf", &VBPEF::rmax_min );
    XML::sscanf( XML::getAttribute(node,"max"), "%lf", &VBPEF::rmax_max );
  }

  return true;
}

