/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

/*!
 \file dttapp.h
 \brief TAPPݒf[^̃NX
*/

#ifndef __DTTAPP_H_INCLUDED
#define __DTTAPP_H_INCLUDED

#include <stdio.h>
#include <vector>
using namespace std;

#include <QtCore/QString>
#include "matrix.h"
#include "position.h"

class DTLattice;
class DTCell;
class DTSymmetry;
class DTAtoms;

class DTTapp
{
public:
  DTTapp( void );

public:
  void update( void );
  void clear( void );

public:
  bool guess( const QString& fname );
  bool load( DTLattice& lattice, const QString& fname );
  bool save( const DTLattice& lattice, const QString& fname );
  static bool loadDefaults( void );

public:
  struct Option {
    bool wfn_on_disk;
    bool qg_on_disk;
    int fsmear;
    int hpc;

    Option( void );
    void clear( void );
    bool load( FILE* fptr );
    bool save( FILE* fptr );

    static bool wfn_on_disk_def;
    static bool qg_on_disk_def;
    static int fsmear_def;
    static int hpc_def;
  } option;

  struct LSDA {
    vector<QString> vxctype;
    int  ixctype;
    int  nspin; // 1=LDA,2=LSD
    int  spinmode; // 0=free mode,1=fixed mode
    int  ipmode; // 0=bare PP,1=add neutralized ,2=add spin polarized
    int  nfixed; // number of fixed SCF iterations

    LSDA( void );
    void clear( void );
    bool load( FILE* fptr );
    bool save( FILE* fptr );

    static int ixctype_def;
    static int nspin_def;
    static int spinmode_def;
    static int ipmode_def;
    static int nfixed_def, nfixed_min, nfixed_max;
  } lsda;

  struct Cell {
    bool load( DTCell& cell, FILE* fptr );
    bool save( const DTCell& cell, FILE* fptr );
  } cell;

  struct Symmetry {
    bool load( DTSymmetry& symmetry, FILE* fptr );
    bool save( const DTSymmetry& symmetry, FILE* fptr );
  } symmetry;

  struct Cutoff {
    double qf; // cutoff for PP, 2*qm+0.1
    double rf; // cutoff for grid, [20:60]
    double qm; // cutoff for wavefunctions
    double qc; // cutoff for reciprocal vectors, > 2+qm
    double beta; // mixing, (0:1]
    double qbtp1, qbtp2, qbtp3;

    Cutoff( void );
    void clear( void );
    bool load( FILE* fptr );
    bool save( FILE* fptr );

    static double qf_def, qf_min, qf_max;
    static double rf_def, rf_min, rf_max;
    static double qm_def, qm_min, qm_max;
    static double qc_def, qc_min, qc_max;
    static double beta_def, beta_min, beta_max;
  } cutoff;

  struct Atoms {
    bool load( DTAtoms& cell, FILE* fptr );
    bool save( const DTAtoms& cell, FILE* fptr );
  } atoms;

  struct Mesh {
    int  ndx, ndy, ndz; // K-mesh for DOS
    int  nk; // number of sampling
    int  nw; // number of bands 
    int  nb1, nb2; // range of bands
    double znext; // extra electrons
    double dspin; // extra spins

    Mesh( void );
    void clear( void );
    bool load( FILE* fptr );
    bool save( FILE* fptr );

    static int  nd_def, nd_min, nd_max;
    static int  nk_def, nk_min, nk_max;
    static int  nw_def, nw_min, nw_max;
    static int  nb1_def, nb2_def, nb_min, nb_max;
    static double znext_def, znext_min, znext_max;
    static double dspin_def, dspin_min, dspin_max;
  } mesh;

  struct Ewald {
    double qgb; // size of reciprocal space
    double qrb; // size of real space
    double g; // cutoff

    Ewald( void );
    void clear( void );
    bool load( FILE* fptr );
    bool save( FILE* fptr );

    static double qgb_def, qgb_min, qgb_max;
    static double qrb_def, qrb_min, qrb_max;
    static double g_def, g_min, g_max;
  } ewald;

  struct Sampling {
    struct int3 {
      int a[3];
      int3( void ){}
      int3( const int a0, const int a1, const int a2 ){
	a[0] = a0, a[1] = a1, a[2] = a2;
      }
      int& operator [] ( const int j ){
	return a[j];
      }
      const int& operator [] ( const int j ) const {
	return a[j];
      }
    };

    vector<QString> vmesh;
    int            imesh;
    bool           gamma;
    
    int3 mesh;
    vector<int3> mmm1;
    vector<int3> mmm2;

    int nkrnew;
    int ll1, ll2, ll3;

    Sampling( void );
    void clear( void );
    bool load( FILE* fptr, const int nk );
    bool save( FILE* fptr );

    static int nkrnew_def, nkrnew_min, nkrnew_max;
    static int ll_def, ll_min, ll_max;

    int3 mmm1_shown;
    int3 mmm2_shown;
    int  index_shown;

    void change_shown( void );
    void update_shown( void );
    int size_shown( void );

    void setByOutline( int& nk );
    void setByDetail( const int nk );
    void addMMM( void );
    void delMMM( void );
  } sampling;

  struct File {
    bool iopt1; // restart, 1=yes, otherwize=no
    bool iopt2; // load wavefunctions, 1=yes, otherwize=no
    bool iopt3; // load potentials, 1=yes, otherwize=no
    bool iopt4; // save wavefunctions, 1=yes, otherwize=no
    bool iopt5;
    bool iopt6;

    File( void );
    void clear( void );
    bool load( FILE* fptr );
    bool save( FILE* fptr );

    static bool iopt1_def, iopt2_def, iopt3_def;
    static bool iopt4_def, iopt5_def, iopt6_def;
  } file;

  struct Mass {
    //    int nmkd; // number of inverse mass tensors
    vector<Matrix> tim; // inverse mass tensors

    Mass( void );
    void clear( void );
    bool load( FILE* fptr );
    bool save( FILE* fptr );

    Matrix   tim_shown;
    int    index_shown;

    void change_shown( void );
    void update_shown( void );
    int size_shown( void );
    void add( void );
    void del( void );
  } mass;

  struct Parameter {
    double eps; // criterion for electronic states, <1.0E-12
    double eepsa; // criterion for optimization energy, 1.0E-10
    double feps; // criterion for optimization force, [1.0E-3:1.0E-2]
    double ekbt; // 1.0E-5
    double decr; // <1.0E-1
    double okatom; // 0.5
    double uptime;
    int  niter0; // 50
    int  niter1; // 20
    int  itsb; // [3:5]
    int  ndiag0; // 1
    int  ndiag1; // 1
    int  ncycl;
    int  mrfr; // [3:5]
    int  most; // [5:8]

    static double eps_def, eps_min, eps_max; 
    static double eepsa_def, eepsa_min, eepsa_max; 
    static double feps_def, feps_min, feps_max; 
    static double ekbt_def, ekbt_min, ekbt_max; 
    static double decr_def, decr_min, decr_max; 
    static double okatom_def, okatom_min, okatom_max; 
    static double uptime_def, uptime_min, uptime_max; 
    static int  niter0_def, niter1_def, niter_min, niter_max; 
    static int  itsb_def, itsb_min, itsb_max; 
    static int  ndiag0_def, ndiag1_def, ndiag_min, ndiag_max; 
    static int  ncycl_def, ncycl_min, ncycl_max; 
    static int  mrfr_def, mrfr_min, mrfr_max; 
    static int  most_def, most_min, most_max; 

    Parameter( void );
    void clear( void );
    bool load( FILE* fptr );
    bool save( FILE* fptr );
  } parameter;

  struct Polar {
    struct kd {
      int pol;
      double pol1, pol2;
      kd( void ){
	pol=1, pol1=pol2=0.0;
      }
      kd( const int pol, const double pol1, const double pol2 ){
	this->pol  = pol;
	this->pol1 = pol1;
	this->pol2 = pol2;
      }
    };
    vector<kd> vkd;
    struct it {
      int pol;
      double asi1, asi2;
      it( void ){
	pol=1, asi1=asi2=0.0;
      }
      it( const int pol, const double asi1, const double asi2 ){
	this->pol  = pol;
	this->asi1 = asi1;
	this->asi2 = asi2;
      }
    };
    vector<it> vit;

    Polar( void );
    bool isset( void ){
      return !vkd.empty();
    }
    void clear( void );
    bool load( FILE* fptr );
    bool save( FILE* fptr );
    void addKD( void );
    void delKD( void );
    void addIT( void );
    void delIT( void );

    kd    kd_shown;
    int index_kd_shown;
    it    it_shown;
    int index_it_shown;

    void change_shown( void );
    void update_shown( void );
    int size_shown_kd( void );
    int size_shown_it( void );
  } polar;

  struct VBPEF {
    int  ifbunp;
    int  ifbfix;
    int  nw;
    int  nb2;
    int  itsb;
    int  ndiag;
    double eps;
    int  nbba1, nbba2;
    int  nsph;
    int  iaxis;
    Position rzero;
    double rmin, rmax;
    int  ndiv;

    static int  ifbunp_def, ifbunp_min, ifbunp_max; 
    static int  ifbfix_def, ifbfix_min, ifbfix_max; 
    static int  nw_def, nw_min, nw_max; 
    static int  nb2_def, nb2_min, nb2_max; 
    static int  itsb_def, itsb_min, itsb_max; 
    static int  ndiag_def, ndiag_min, ndiag_max; 
    static double eps_def, eps_min, eps_max; 
    static int  nbba_def, nbba_min, nbba_max; 
    static int  nsph_def, nsph_min, nsph_max; 
    static int  iaxis_def, iaxis_min, iaxis_max; 
    static double rzero_def, rzero_min, rzero_max; 
    static double rmin_def, rmin_min, rmin_max; 
    static double rmax_def, rmax_min, rmax_max; 
    static int  ndiv_def, ndiv_min, ndiv_max; 

    VBPEF( void );
    void clear( void );
    bool load( DTCell& cell, FILE* fptr );
    bool save( const DTCell& cell, FILE* fptr );
  } vbpef;

  struct VBWFN {
    struct nb {
      int nbb1, nbb2;
      nb( void ){ nbb1=0, nbb2=0; }
      nb( const int nbb1, const int nbb2 ){
	this->nbb1 = nbb1;
	this->nbb2 = nbb2;
      }
    };
    vector<nb> vnb;

    int nraxn, nrayn, nrazn;
    bool  sqamp, wfn;
    double sk[3];
    int itsb, ndiag;
    double eps;

    VBWFN( void );
    bool isset( void ){
      return !vnb.empty();
    }
    void clear( void );
    bool load( FILE* fptr );
    bool save( FILE* fptr );
    void add( void );
    void del( void );

    nb    nb_shown;
    int index_shown;

    void change_shown( void );
    void update_shown( void );
    int size_shown( void );
  } vbwfn;

  struct TCHRPOT {
    int nraxn, nrayn, nrazn;
    bool  tchr, lpot;

    TCHRPOT( void );
    bool isset( void ){
      return nraxn*nrayn*nrazn>0;
    }

    void clear( void );
    bool load( FILE* fptr );
    bool save( FILE* fptr );
  } tchrpot;


};

#endif // __DTTAPP_H_INCLUDED
