/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

/*!
 \file glatoms.cc
 \brief iq̌qGL\̃NX
*/

#include <GL/gl.h>
#include "dtmodel.h"
#include "glatoms.h"
#include <QtOpenGL/QGLWidget>

void GLAtoms::updateClustering( void )
{
  const Position& Lo = model.lattice.cell.Lo;
  const vector<DTAtom>& vatom = model.lattice.cluster.vatom;

  GLBase::beginNewList();

  glPushMatrix();
  glTranslated( Lo.x, Lo.y, Lo.z );

  for( int n=0; n<(int)vatom.size(); n++ ){
    makeAtom( vatom[n] );
  }

  glPopMatrix();

  GLBase::endNewList();
}

void GLAtoms::update( void )
{
  if( model.lattice.isclustering() ){
    updateClustering();
    return;
  }

  const Position& Lo = model.lattice.cell.Lo;
  const vector<DTAtom>& vatom = model.lattice.getAtoms();

  const bool    show_copy = model.gloption.lattice.atom.show_copy;

  GLBase::beginNewList();

  glPushMatrix();
  glTranslated( Lo.x, Lo.y, Lo.z );

  for( int n=0; n<(int)vatom.size(); n++ ){
    //    if( vatom[n].isDuplicate() ) continue;
    if( vatom[n].isCopy() && !show_copy ) continue;

    glLoadName(n);
    makeAtom( vatom[n] );
  }
  glLoadName(GLuint(-1));

  const vector<int>& viatom = model.lattice.getData().getAtomsSelected();
  for( int i=0; i<(int)viatom.size(); i++ ){
    const int iatom = viatom[i];
    const int index = vatom[iatom].getIndex();

    for( int n=0; n<(int)vatom.size(); n++ ){
      //      if( vatom[n].isDuplicate() ) continue;
      if( vatom[n].isCopy() && !show_copy ) continue;

      if( n==iatom && i+1==(int)viatom.size() ){
	makeAtomSelected( vatom[n], 0 );
      }
      else if( n==iatom ){
	makeAtomSelected( vatom[n], 1 );
      }
      else if( vatom[n].getIndex()==index ){
	makeAtomSelected( vatom[n], 2 );
      }
    }
  }

  glPopMatrix();

  GLBase::endNewList();

}

void GLAtoms::select( void )
{
  const Position& La = model.lattice.cell.La;
  const Position& Lb = model.lattice.cell.Lb;
  const Position& Lc = model.lattice.cell.Lc;
  const Position& Lo = model.lattice.cell.Lo;
  const vector<DTAtom>& vatom = model.lattice.getAtoms();

  glPushMatrix();
  glTranslated( Lo.x, Lo.y, Lo.z );

  for( int n=0; n<(int)vatom.size(); n++ ){
    if( vatom[n].isCopy() ) continue;

    glLoadName(n);
    makeAtom( vatom[n] );
  }

  if( model.gloption.location.extendA>0 ){
    for( int n=0; n<(int)vatom.size(); n++ ){
      if( vatom[n].isCopy() ) continue;

      glLoadName(n+vatom.size()*(1+3*0+9*0) );
      makeAtom( vatom[n], +La );
      glLoadName(n+vatom.size()*(2+3*0+9*0) );
      makeAtom( vatom[n], -La );
    }
  }
  if( model.gloption.location.extendB>0 ){
    for( int n=0; n<(int)vatom.size(); n++ ){
      if( vatom[n].isCopy() ) continue;

      glLoadName(n+vatom.size()*(0+3*1+9*0) );
      makeAtom( vatom[n], +Lb );
      glLoadName(n+vatom.size()*(0+3*2+9*0) );
      makeAtom( vatom[n], -Lb );
    }
  }
  if( model.gloption.location.extendC>0 ){
    for( int n=0; n<(int)vatom.size(); n++ ){
      if( vatom[n].isCopy() ) continue;

      glLoadName(n+vatom.size()*(0+3*0+9*1) );
      makeAtom( vatom[n], +Lc );
      glLoadName(n+vatom.size()*(0+3*0+9*2) );
      makeAtom( vatom[n], -Lc );
    }
  }
  if( model.gloption.location.extendA>0 &&
      model.gloption.location.extendB>0 ){
    for( int n=0; n<(int)vatom.size(); n++ ){
      if( vatom[n].isCopy() ) continue;

      glLoadName(n+vatom.size()*(1+3*1+9*0) );
      makeAtom( vatom[n], +La+Lb );
      glLoadName(n+vatom.size()*(2+3*1+9*0) );
      makeAtom( vatom[n], -La+Lb );
      glLoadName(n+vatom.size()*(1+3*2+9*0) );
      makeAtom( vatom[n], +La-Lb );
      glLoadName(n+vatom.size()*(2+3*2+9*0) );
      makeAtom( vatom[n], -La-Lb );
    }
  }
  if( model.gloption.location.extendB>0 &&
      model.gloption.location.extendC>0 ){
    for( int n=0; n<(int)vatom.size(); n++ ){
      if( vatom[n].isCopy() ) continue;

      glLoadName(n+vatom.size()*(0+3*1+9*1) );
      makeAtom( vatom[n], +Lb+Lc );
      glLoadName(n+vatom.size()*(0+3*2+9*1) );
      makeAtom( vatom[n], -Lb+Lc );
      glLoadName(n+vatom.size()*(0+3*1+9*2) );
      makeAtom( vatom[n], +Lb-Lc );
      glLoadName(n+vatom.size()*(0+3*2+9*2) );
      makeAtom( vatom[n], -Lb-Lc );
    }
  }
  if( model.gloption.location.extendC>0 &&
      model.gloption.location.extendA>0 ){
    for( int n=0; n<(int)vatom.size(); n++ ){
      if( vatom[n].isCopy() ) continue;

      glLoadName(n+vatom.size()*(1+3*0+9*1) );
      makeAtom( vatom[n], +Lc+La );
      glLoadName(n+vatom.size()*(1+3*0+9*2) );
      makeAtom( vatom[n], -Lc+La );
      glLoadName(n+vatom.size()*(2+3*0+9*1) );
      makeAtom( vatom[n], +Lc-La );
      glLoadName(n+vatom.size()*(2+3*0+9*2) );
      makeAtom( vatom[n], -Lc-La );
    }
  }
  if( model.gloption.location.extendA>0 &&
      model.gloption.location.extendB>0 &&
      model.gloption.location.extendC>0 ){
    for( int n=0; n<(int)vatom.size(); n++ ){
      if( vatom[n].isCopy() ) continue;

      glLoadName(n+vatom.size()*(1+3*1+9*1) );
      makeAtom( vatom[n], +La+Lb+Lc );
      glLoadName(n+vatom.size()*(2+3*1+9*1) );
      makeAtom( vatom[n], -La+Lb+Lc );
      glLoadName(n+vatom.size()*(1+3*2+9*1) );
      makeAtom( vatom[n], +La-Lb+Lc );
      glLoadName(n+vatom.size()*(2+3*2+9*1) );
      makeAtom( vatom[n], -La-Lb+Lc );
      glLoadName(n+vatom.size()*(1+3*1+9*2) );
      makeAtom( vatom[n], +La+Lb-Lc );
      glLoadName(n+vatom.size()*(2+3*1+9*2) );
      makeAtom( vatom[n], -La+Lb-Lc );
      glLoadName(n+vatom.size()*(1+3*2+9*2) );
      makeAtom( vatom[n], +La-Lb-Lc );
      glLoadName(n+vatom.size()*(2+3*2+9*2) );
      makeAtom( vatom[n], -La-Lb-Lc );
    }
  }

  glLoadName(GLuint(-1));

  glPopMatrix();
}

void GLAtoms::makeAtom( const DTAtom& atom )
{
  const int&  number = atom.element.number;
  const Coordinates coords = model.gloption.location.scroll(atom.coords);
  const Position pos = model.lattice.cell.getPositionLocal(coords);

  const GLOption::Lattice::Atom::byElement& element = model.gloption.lattice.atom.element(number);
  const double& radius = element.radius;
  const GLcolor& color = element.color;
  const double& scale  = model.gloption.lattice.atom.scale;
  const int     slices = (int)model.gloption.lattice.atom.slices;

  const bool&   points = model.gloption.lattice.atom.points;

  glColor4dv(color);
  glMaterialdv( GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, color );

  if( points ){
    glDisable( GL_LIGHTING );
    glPointSize(scale*32.0*radius);
    glBegin( GL_POINTS );
    glVertex3dv(pos);
    glEnd();
    glPointSize(1.0);
    glEnable( GL_LIGHTING );
  }
  else{
    gluSolidBall( scale*radius, pos, slices );
  }
}


void GLAtoms::makeAtom( const DTAtom& atom, const Position& L )
{
  const int&  number = atom.element.number;
  const Coordinates coords = model.gloption.location.scroll(atom.coords);
  const Position pos = model.lattice.cell.getPositionLocal(coords) + L;

  const GLOption::Lattice::Atom::byElement& element = model.gloption.lattice.atom.element(number);
  const double& radius = element.radius;
  const GLcolor& color = element.color;
  const double& scale  = model.gloption.lattice.atom.scale;
  const int     slices = (int)model.gloption.lattice.atom.slices;

  const bool&   points = model.gloption.lattice.atom.points;

  const double delta = +1.0/1024; // NbsO̗V

  if( model.gloption.location.extendA > 0 &&
      model.gloption.location.extendA < 100 ){
    const double a = model.lattice.cell.Ka*pos;
    if( a<0.0-double(model.gloption.location.extendA)/100.0-delta ) return;
    if( a>1.0+double(model.gloption.location.extendA)/100.0+delta ) return;
  }

  if( model.gloption.location.extendB > 0 &&
      model.gloption.location.extendB < 100 ){
    const double b = model.lattice.cell.Kb*pos;
    if( b<0.0-double(model.gloption.location.extendB)/100.0-delta ) return;
    if( b>1.0+double(model.gloption.location.extendB)/100.0+delta ) return;
  }

  if( model.gloption.location.extendC > 0 &&
      model.gloption.location.extendC < 100 ){
    const double c = model.lattice.cell.Kc*pos;
    if( c<0.0-double(model.gloption.location.extendC)/100.0-delta ) return;
    if( c>1.0+double(model.gloption.location.extendC)/100.0+delta ) return;
  }

  glColor4dv(color);
  glMaterialdv( GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, color );

  if( points ){
    glDisable( GL_LIGHTING );
    glPointSize(scale*32.0*radius);
    glBegin( GL_POINTS );
    glVertex3dv(pos);
    glEnd();
    glPointSize(1.0);
    glEnable( GL_LIGHTING );
  }
  else{
    gluSolidBall( scale*radius, pos, slices );
  }
}

void GLAtoms::makeAtomSelected( const DTAtom& atom, const int selected )
{
  const int&   number = atom.element.number;
  const Coordinates coords = model.gloption.location.scroll(atom.coords);
  const Position pos = model.lattice.cell.getPositionLocal(coords);

  const GLOption::Lattice::Atom::byElement& element = model.gloption.lattice.atom.element(number);
  const double& radius = element.radius;
  const GLcolor& color = model.gloption.lattice.atom.color_selected;
  const double& scale  = model.gloption.lattice.atom.scale;

  switch( selected ){
  case 0 : glColor4dv( color ); break;
  case 1 : glColor4dv( GLcolor::orange ); break;
  case 2 : glColor4dv( GLcolor::red ); break;
  }
  glDisable( GL_LIGHTING );
  gluWireCube( scale*2*radius, pos );
  glEnable( GL_LIGHTING );
}



void GLAtoms::draw( void )
{
  GLBase::draw();

  glPushMatrix();
  const Position& Lo = model.lattice.cell.Lo;
  glTranslated( Lo.x, Lo.y, Lo.z );

  double bond_length=0.0;
  switch( model.lattice.getData().getDragMode() ){
  case 0 : break;
  case 1 : break;
  case 2 : {
    Position pos0, pos1, pos2;

    glDisable( GL_LIGHTING );
    glColor4dv( GLcolor::white );

    glBegin( GL_LINES );
    {
      const Coordinates coords =
	model.lattice.getData().getDragCoordinate(0);
      const Coordinates scroll =
	model.gloption.location.scroll(coords);
      pos0 = model.lattice.cell.getPositionLocal(scroll);
      glVertex3dv( pos0 );
    }
    {
      const Coordinates coords =
	model.lattice.getData().getDragCoordinate(1);
      const Coordinates scroll =
	model.gloption.location.scroll(coords);
      pos1 = model.lattice.cell.getPositionLocal(scroll);
      glVertex3dv( pos1 );
    }
    {
      const Coordinates coords =
	model.lattice.getData().getDragCoordinate(2);
      const Coordinates scroll =
	model.gloption.location.scroll(coords);
      pos2 = model.lattice.cell.getPositionLocal(scroll);
      glVertex3dv( pos0 );
      glVertex3dv( pos2 );
    }
    glEnd();

    bond_length = Position::length(pos2-pos0);

    glEnable( GL_LIGHTING );
  } break;

  case 3 : {
    Position pos0, pos1, pos2, pos3;

    glDisable( GL_LIGHTING );
    glColor4dv( GLcolor::white );

    glBegin( GL_LINE_LOOP );
    {
      const Coordinates coords =
	model.lattice.getData().getDragCoordinate(0);
      const Coordinates scroll =
	model.gloption.location.scroll(coords);
      pos0 = model.lattice.cell.getPositionLocal(scroll);
      glVertex3dv( pos0 );
    }
    {
      const Coordinates coords =
	model.lattice.getData().getDragCoordinate(1);
      const Coordinates scroll =
	model.gloption.location.scroll(coords);
      pos1 = model.lattice.cell.getPositionLocal(scroll);
      glVertex3dv( pos1 );
    }
    {
      const Coordinates coords =
	model.lattice.getData().getDragCoordinate(2);
      const Coordinates scroll =
	model.gloption.location.scroll(coords);
      pos2 = model.lattice.cell.getPositionLocal(scroll);
      glVertex3dv( pos2 );
    }
    glEnd();

    glBegin( GL_LINES );
    {
      const Coordinates coords =
	model.lattice.getData().getDragCoordinate(3);
      const Coordinates scroll =
	model.gloption.location.scroll(coords);
      pos3 = model.lattice.cell.getPositionLocal(scroll);
      glVertex3dv( pos1 );
      glVertex3dv( pos3 );
    }
    glEnd();

    bond_length = Position::length(pos3-pos1);

    glEnable( GL_LIGHTING );
  } break;


  case 4 : {
    Position pos0, pos1, pos2, pos3, pos4;

    glDisable( GL_LIGHTING );
    glColor4dv( GLcolor::white );

    glBegin( GL_LINE_LOOP );
    {
      const Coordinates coords =
	model.lattice.getData().getDragCoordinate(0);
      const Coordinates scroll =
	model.gloption.location.scroll(coords);
      pos0 = model.lattice.cell.getPositionLocal(scroll);
      glVertex3dv( pos0 );
    }
    {
      const Coordinates coords =
	model.lattice.getData().getDragCoordinate(1);
      const Coordinates scroll =
	model.gloption.location.scroll(coords);
      pos1 = model.lattice.cell.getPositionLocal(scroll);
      glVertex3dv( pos1 );
    }
    {
      const Coordinates coords =
	model.lattice.getData().getDragCoordinate(2);
      const Coordinates scroll =
	model.gloption.location.scroll(coords);
      pos2 = model.lattice.cell.getPositionLocal(scroll);
      glVertex3dv( pos2 );
    }
    glEnd();


    glBegin( GL_LINE_LOOP );
    {
      const Coordinates coords =
	model.lattice.getData().getDragCoordinate(1);
      const Coordinates scroll =
	model.gloption.location.scroll(coords);
      pos1 = model.lattice.cell.getPositionLocal(scroll);
      glVertex3dv( pos1 );
    }
    {
      const Coordinates coords =
	model.lattice.getData().getDragCoordinate(2);
      const Coordinates scroll =
	model.gloption.location.scroll(coords);
      pos2 = model.lattice.cell.getPositionLocal(scroll);
      glVertex3dv( pos2 );
    }
    {
      const Coordinates coords =
	model.lattice.getData().getDragCoordinate(3);
      const Coordinates scroll =
	model.gloption.location.scroll(coords);
      pos3 = model.lattice.cell.getPositionLocal(scroll);
      glVertex3dv( pos3 );
    }
    glEnd();

    glBegin( GL_LINES );
    {
      const Coordinates coords =
	model.lattice.getData().getDragCoordinate(4);
      const Coordinates scroll =
	model.gloption.location.scroll(coords);
      pos4 = model.lattice.cell.getPositionLocal(scroll);
      glVertex3dv( pos2 );
      glVertex3dv( pos4 );
    }
    glEnd();

    bond_length = Position::length(pos4-pos2);

    glEnable( GL_LIGHTING );
  } break;
  }

  glPopMatrix();


  // draw extra objects on time.
  int value;
  glGetIntegerv( GL_RENDER_MODE, &value );

  if( value == GL_RENDER &&
      model.lattice.getData().getDragMode() >= 2 ){

    const QFont font("Helvetica", 12);

    const int win_width  = glGetWindowWidth();
    const int win_height = glGetWindowHeight();

    glColor4dv( GLcolor::white );

    glMatrixMode( GL_MODELVIEW );
    glPushMatrix();
    glLoadIdentity();
  
    glMatrixMode( GL_PROJECTION );
    glPushMatrix();
    glLoadIdentity();
    gluOrtho2D( 0.0, (GLfloat)win_width, (GLfloat)win_height, 0.0 );

    // qԋ̐l\
    char str[32];
    snprintf( str, sizeof(str), "L=%10.6f angstrom", bond_length );

    int xo = 16;
    int yo = win_height-16;

    qglwidget->renderText( xo, yo, QString(str), font );

    glPopMatrix();
    glMatrixMode( GL_MODELVIEW );
    glPopMatrix();
  }
}
