/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

/*!
 \file glband.cc
 \brief ohGL\̃NX
*/

#include "dtmodel.h"
#include "glmisc.h"
#include "qtmisc.h"
#include "glband.h"
#include "gleps.h"
#include "qtexception.h"


GLBand::GLBand( DTModel& _model ) : model(_model)
{
  show_up = true;
  show_dw = true;

  setFocusPolicy( Qt::ClickFocus );
  setMinimumSize( model.gloption.window.width-256, model.gloption.window.height-32 );
  setSizePolicy( QSizePolicy::Expanding, QSizePolicy::Expanding );

  popupmenu = new QMenu(this);
  popupmenu->addSeparator();
  popupmenu->addAction( new QAction(tr("zoom energy in"),this) );
  popupmenu->addAction( new QAction(tr("zoom energy out"),this) );
  popupmenu->addAction( new QAction(tr("zoom k-axis in"),this) );
  popupmenu->addAction( new QAction(tr("zoom k-axis out"),this) );
  popupmenu->addAction( new QAction(tr("zoom reset"),this) );
  popupmenu->addSeparator();

  popupmenu->addAction( action_both_spin=new QAction(tr("both spin"),this) );
  action_both_spin->setCheckable(true);
  action_both_spin->setChecked( show_up && show_dw );
  popupmenu->addAction( action_up_spin=new QAction(tr("up   spin"),this) );
  action_up_spin->setCheckable(true);
  action_up_spin->setChecked( show_up && !show_dw );
  popupmenu->addAction( action_dw_spin=new QAction(tr("down spin"),this) );
  action_dw_spin->setCheckable(true);
  action_dw_spin->setChecked( !show_up && show_dw );

  popupmenu->addSeparator();

  popupmenu->addAction( action_def_ord=new QAction(tr("default order"),this) );
  action_def_ord->setCheckable(true);
  action_def_ord->setChecked( model.band.order == DTBand::DEFAULT_ORDER );

  popupmenu->addAction( action_asc_ord=new QAction(tr("ascending order"),this) );
  action_asc_ord->setCheckable(true);
  action_asc_ord->setChecked( model.band.order == DTBand::ASCENDING_ORDER );

  popupmenu->addAction( action_nat_ord=new QAction(tr("natural order"),this) );
  action_nat_ord->setCheckable(true);
  action_nat_ord->setChecked( model.band.order == DTBand::NATUAL_ORDER );

  popupmenu->addSeparator();

  popupmenu->addAction( action_colorful=new QAction(tr("band colorful"),this) );
  action_colorful->setCheckable(true);
  action_colorful->setChecked( !model.gloption.band.monochrome );
  popupmenu->addAction( action_lines=new QAction(tr("band lines"),this) );
  action_lines->setCheckable(true);
  action_lines->setChecked( model.gloption.band.line.show );
  popupmenu->addAction( action_points=new QAction(tr("band points"),this) );
  action_points->setCheckable(true);
  action_points->setChecked( model.gloption.band.point.show );

  popupmenu->addSeparator();
  popupmenu->addAction( new QAction(tr("save image"),this) );
  connect(popupmenu, SIGNAL(triggered(QAction*)), this, SLOT(menuEvent(QAction*)));
}

void GLBand::update( void )
{
  makeCurrent();
  updateGL();
}

void GLBand::initializeGL( void )
{
}

void GLBand::resizeGL(int width, int height)
{
  glViewport( 0, 0, width, height );

  glMatrixMode( GL_PROJECTION );
  glLoadIdentity();

  gluOrtho2D( 0.0, (GLfloat)width, 0.0, (GLfloat)height );

  glMatrixMode( GL_MODELVIEW );

  glClear( GL_COLOR_BUFFER_BIT );


  // E㉺̗]̍Œx
  marginL = 96;
  marginR = 16;
  marginT = 36;
  marginB = 36;

  // EBhẼTCY
  win_width  = glGetWindowWidth();
  win_height = glGetWindowHeight();

  // Ot͈
  grp_width  = win_width  - marginL - marginR;
  grp_height = win_height - marginT - marginB;
}

static double round( double& a, double& b )
{
  double d = a-b;
  double u = pow( 10.0, rint(log10(d))-1.0 );

  while( d/u >= 8.0 ){
    u *= 2.0;
  }

  a = ceil (a/u+1.0e-8)*u;
  b = floor(b/u)*u;

  return u;
}

void GLBand::paintGL( void )
{
  char label[32];

  glClearColor( model.gloption.band.background );
  glClear( GL_COLOR_BUFFER_BIT );

  if( !model.band.isset() ) return;

  const double kmin = model.gloption.band.kmin;
  const double kmax = model.gloption.band.kmax;
  const double emin = model.gloption.band.emin;
  const double emax = model.gloption.band.emax;

  //Ot_`W_Ɉړ
  glPushMatrix();
  glLoadIdentity();
  glTranslated( (double)marginL, (double)marginB, 0.0 );

  // Otg̕`
  {
    glColor4dv( model.gloption.band.foreground );
    glBegin( GL_LINE_LOOP );
    glVertex2d( 0, -1 );
    glVertex2d( grp_width, -1 );
    glVertex2d( grp_width, grp_height+1 );
    glVertex2d( 0, grp_height+1 );
    glEnd();
  }

  // ̖O̕`
  {
    glColor4dv( model.gloption.band.foreground );
    // 
    renderText( grp_width/2, -24, 0.0, QString("k-point"));

    // c
    renderText( -40, grp_height+5, 0.0, QString("energy"));
  }

  // Wn̕ύXAf[^̂̂vbgŏ̍Wɕ`ł
  {
    glPushMatrix();
    glScaled( double(grp_width)/(kmax-kmin),
	      double(grp_height)/(emax-emin), 0.0 );
    glTranslated( -kmin, -emin, 0.0 );

    // ڐ胉x̕`
    {
      // c
      glPushMatrix();
      glTranslateR( -70, -10, 0.0 );

      const int N = 8;

      double dE = emax - emin;
      double E0 = 0.0;
      round( dE, E0 );
      dE = dE/N;

      const int is=(int)ceil (emin/dE);
      const int ie=(int)floor(emax/dE);

      glColor4dv( model.gloption.band.foreground );
      for( int i=is; i<=ie; i++ ){
	double E = dE*i;
	sprintf( label, "%+.*e", 1, E );
	renderText( kmin, E, 0.0, QString(label));
      }
      glPopMatrix();

      // 
      glPushMatrix();
      glTranslateR( -5, -12, 0.0 );

      for( int i=0; i<model.band.sizeX(); i++ ){
	double x = model.band.data[0][i].x;
	if( x<kmin || kmax<x ) continue;

	if( model.band.data[0][i].xlabel != "" ){
	  sprintf( label, "%s", qPrintable(model.band.data[0][i].xlabel) );
	  renderText( x, emin, 0.0, QString(label) );
	}
      }
      glPopMatrix();

    }
    
    // Obh̕`
    {
      const int N = 8;

      // Obh
      {      
	double dE = emax - emin;
	double E0 = 0.0;
	round( dE, E0 );
	dE = dE/N;
      
	const int is=(int)ceil (emin/dE);
	const int ie=(int)floor(emax/dE);
      

	glColor4dv( model.gloption.band.background*0.75
		    +model.gloption.band.foreground*0.25 );
	glBegin( GL_LINES );
	for( int i=is; i<=ie; i++ ){
	  double E = dE*i;
	  glVertex2d( kmin, E );
	  glVertex2d( kmax, E );
	}
	glEnd();
      }

      // Obhc
      {
	glBegin( GL_LINES );
	for( int i=0; i<model.band.sizeX(); i++ ){
	  double x = model.band.data[0][i].x;
	  if( x<kmin || kmax<x ) continue;

	  if( model.band.data[0][i].xlabel != "" ){
	    glVertex2d( x, emin );
	    glVertex2d( x, emax );
	  }
	}
	glEnd();
      }
    }

    // f[^̕`
    for( int j=0; j<model.band.sizeY(); j++ ){
      if( model.gloption.band.monochrome ){
	glColor4dv(model.gloption.band.foreground );
      }
      else{
	const double d = double(j)/(model.band.sizeY()-1);
	glColor4dv(model.gloption.band.gradation(d));
      }

      if( model.gloption.band.line.show ){
	glLineWidth((double)model.gloption.band.line.size);

	if( show_up ){
	  int n=0;
	  double Kpre=0.0,Epre=0.0;
	  bool   Fpre=false;
	  for( int i=0; i<model.band.sizeX(); i++ ){
	    double k = model.band.data[0][i].x;
	    double E = model.band.data[0][i].vy[j];
	    // ͈͊O
	    if( k<kmin || kmax<k ){
	      // ͈͓Oɏoꍇ
	      if( n>0 ){
		// Ԃ_܂Ő`
		if( Fpre && k>kmax ){
		  glVertex2d( kmax, (E-Epre)/(k-Kpre)*(kmax-Kpre)+Epre );
		  glEnd();
		  n=0;
		}
	      }
	      Kpre=k;
	      Epre=E;
	      Fpre=true;
	      continue;
	    }

	    // c͈͓
	    if( emin<=E && E<=emax ){
	      // ߂Ă͈͓̔
	      if( n==0 ){
		glBegin( GL_LINE_STRIP );
		if( false );
		// O_c͈͊OȀ͈͓ꍇɕԂ_`
		else if( Fpre && Epre>emax && k>=kmin ){
		  glVertex2d( (k-Kpre)/(E-Epre)*(emax-Epre)+Kpre, emax );
		}
		// O_c͈͊OȀ͈͓ꍇɕԂ_`
		else if( Fpre && Epre<emin && k>=kmin ){
		  glVertex2d( (k-Kpre)/(E-Epre)*(emin-Epre)+Kpre, emin );
		}
		// O_͈͊ȌꍇɕԂ_`
		else if( Fpre && Kpre<kmin ){
		  glVertex2d( kmin, (E-Epre)/(k-Kpre)*(kmin-Kpre)+Epre );
		}
	      }
	      // `
	      glVertex2d( k, E );
	      n++;
	    }
	    // c͈͊O
	    else if(n>0){
	      if( false );
	      // ͈͊ȌꍇɕԂ_܂Ő`
	      else if(Fpre && E>emax){
		glVertex2d( (k-Kpre)/(E-Epre)*(emax-Epre)+Kpre, emax );
	      }
	      // ͈͊ȌꍇɕԂ_܂Ő`
	      else if(Fpre && E<emin){
		glVertex2d( (k-Kpre)/(E-Epre)*(emin-Epre)+Kpre, emin );
	      }
	      glEnd();
	      n=0;
	    }
	    Kpre=k;
	    Epre=E;
	    Fpre=true;
	  }
	  if(n>0){
	    glEnd(),n=0;
	  }
	}
	if( show_dw && model.band.isset2() ){
	  int n=0;
	  double Kpre=0.0,Epre=0.0;
	  bool   Fpre=false;
	  for( int i=0; i<model.band.sizeX(); i++ ){
	    double k = model.band.data[0][i].x;
	    double E = model.band.data[1][i].vy[j];
	    if( k<kmin || kmax<k ){
	      if( n>0 ){
		if( Fpre && k>kmax ){
		  glVertex2d( kmax, (E-Epre)/(k-Kpre)*(kmax-Kpre)+Epre );
		  glEnd();
		  n=0;
		}
	      }
	      Kpre=k;
	      Epre=E;
	      Fpre=true;
	      continue;
	    }
	    if( emin<=E && E<=emax ){
	      if( n==0 ){
		glBegin( GL_LINE_STRIP );
		if( false );
		else if( Fpre && Epre>emax && k>=kmin ){
		  glVertex2d( (k-Kpre)/(E-Epre)*(emax-Epre)+Kpre, emax );
		}
		else if( Fpre && Epre<emin && k>=kmin ){
		  glVertex2d( (k-Kpre)/(E-Epre)*(emin-Epre)+Kpre, emin );
		}
		else if( Fpre && Kpre<kmin ){
		  glVertex2d( kmin, (E-Epre)/(k-Kpre)*(kmin-Kpre)+Epre );
		}
	      }
	      glVertex2d( k, E );
	      n++;
	    }
	    else if(n>0){
	      if( false );
	      else if(Fpre && E>emax){
		glVertex2d( (k-Kpre)/(E-Epre)*(emax-Epre)+Kpre, emax );
	      }
	      else if(Fpre && E<emin){
		glVertex2d( (k-Kpre)/(E-Epre)*(emin-Epre)+Kpre, emin );
	      }
	      glEnd();
	      n=0;
	    }
	    Kpre=k;
	    Epre=E;
	    Fpre=true;
	  }
	  if(n>0){
	    glEnd(),n=0;
	  }
	}

	glLineWidth(1.0);
      }
    
      if( model.gloption.band.point.show ){
	glPointSize((double)model.gloption.band.point.size);

	if( show_up ){
	  glBegin( GL_POINTS );
	  for( int i=0; i<model.band.sizeX(); i++ ){
	    double x = model.band.data[0][i].x;
	    double E = model.band.data[0][i].vy[j];
	    if( x<kmin || kmax<x ) continue;
	    if( E<emin || emax<E ) continue;
	    glVertex2d( x, E );
	  }
	  glEnd();
	}
	if( show_dw && model.band.isset2() ){
	  glBegin( GL_POINTS );
	  for( int i=0; i<model.band.sizeX(); i++ ){
	    double x = model.band.data[0][i].x;
	    double E = model.band.data[1][i].vy[j];
	    if( x<kmin || kmax<x ) continue;
	    if( E<emin || emax<E ) continue;
	    glVertex2d( x, E );
	  }
	  glEnd();
	}
	glPointSize(1.0);
      }
    } // end for

    glPopMatrix();
  }

  glPopMatrix();
}

bool GLBand::paintEPS( const QString& fname )
{
  char label[32];

  if( !model.band.isset() ) return false;

  const double kmin = model.gloption.band.kmin;
  const double kmax = model.gloption.band.kmax;
  const double emin = model.gloption.band.emin;
  const double emax = model.gloption.band.emax;

  GLEPS eps;
  if( !eps.open( qPrintable(fname),
		 win_width, win_height, 100.0, 100.0*win_height/win_width ) ){
    return false;
  }

  eps.font( "/Times-Roman", 16.0 );

  //Ot_`W_Ɉړ
  eps.pushMatrix();
  eps.loadIdentity();
  eps.translated( (double)marginL, (double)marginB, 0.0 );
  eps.lineWidth((double)model.gloption.band.line.size);

  // Otg̕`
  {
    eps.color4dv( model.gloption.band.foreground );
    eps.begin( GL_LINE_LOOP );
    eps.vertex2d( 0, -1 );
    eps.vertex2d( grp_width, -1 );
    eps.vertex2d( grp_width, grp_height+1 );
    eps.vertex2d( 0, grp_height+1 );
    eps.end();
  }

  // ̖O̕`
  {
    eps.color4dv( model.gloption.band.foreground );
    // 
    eps.renderText( grp_width/2, -30, 0.0, QString("k-point"));

    // c
    eps.renderText( -45, grp_height+5, 0.0, QString("energy"));
  }


  // Wn̕ύXAf[^̂̂vbgŏ̍Wɕ`ł
  {
    eps.pushMatrix();
    eps.scaled( double(grp_width)/(kmax-kmin),
	      double(grp_height)/(emax-emin), 0.0 );
    eps.translated( -kmin, -emin, 0.0 );

    // ڐ胉x̕`
    {
      const int N = 8;

      // c
      {
	eps.pushMatrix();
	eps.translateR( -70, -10, 0.0 );

	double dE = emax - emin;
	double E0 = 0.0;
	round( dE, E0 );
	dE = dE/N;

	const int is=(int)ceil (emin/dE);
	const int ie=(int)floor(emax/dE);

	eps.color4dv( model.gloption.band.foreground );
	for( int i=is; i<=ie; i++ ){
	  double E = dE*i;
	  sprintf( label, "%+.*e", 1, E );
	  eps.renderText( kmin, E, 0.0, QString(label));
	}
	eps.popMatrix();
      }

      // 
      {
	eps.pushMatrix();
	eps.translateR( -5, -15, 0.0 );

	for( int i=0; i<model.band.sizeX(); i++ ){
	  double x = model.band.data[0][i].x;
	  if( x<kmin || kmax<x ) continue;
	  if( model.band.data[0][i].xlabel != "" ){
	    sprintf( label, "%s", qPrintable(model.band.data[0][i].xlabel) );
	    eps.renderText( x, emin, 0.0, QString(label) );
	  }
	}
	eps.popMatrix();
      }
    }
    
    // Obh̕`
    {
      const int N = 8;

      // Obh
      {      
	double dE = emax - emin;
	double E0 = 0.0;
	round( dE, E0 );
	dE = dE/N;
      
	const int is=(int)ceil (emin/dE);
	const int ie=(int)floor(emax/dE);

	eps.color4dv( model.gloption.band.background*0.75
		      +model.gloption.band.foreground*0.25 );
	eps.begin( GL_LINES );
	for( int i=is; i<=ie; i++ ){
	  double E = dE*i;
	  eps.vertex2d( kmin, E );
	  eps.vertex2d( kmax, E );
	}
	eps.end();
      }

      // Obhc
      {
	eps.begin( GL_LINES );
	for( int i=0; i<model.band.sizeX(); i++ ){
	  double x = model.band.data[0][i].x;
	  if( x<kmin || kmax<x ) continue;
	  if( model.band.data[0][i].xlabel != "" ){
	    eps.vertex2d( x, emin );
	    eps.vertex2d( x, emax );
	  }
	}
	eps.end();
      }
    }

    // f[^̕`
    for( int j=0; j<model.band.sizeY(); j++ ){
      if( model.gloption.band.monochrome ){
	eps.color4dv(model.gloption.band.foreground );
      }
      else{
	const double d = double(j)/(model.band.sizeY()-1);
	eps.color4dv(model.gloption.band.gradation(d));
      }

      if( model.gloption.band.line.show ){
	if( show_up ){
	  int n=0;
	  double Kpre=0.0,Epre=0.0;
	  bool   Fpre=false;
	  for( int i=0; i<model.band.sizeX(); i++ ){
	    double k = model.band.data[0][i].x;
	    double E = model.band.data[0][i].vy[j];
	    if( k<kmin || kmax<k ){
	      if( n>0 ){
		if( Fpre && k>kmax ){
		  eps.vertex2d( kmax, (E-Epre)/(k-Kpre)*(kmax-Kpre)+Epre );
		  eps.end();
		  n=0;
		}
	      }
	      Kpre=k;
	      Epre=E;
	      Fpre=true;
	      continue;
	    }
	    if( emin<=E && E<=emax ){
	      if( n==0 ){
		eps.begin( GL_LINE_STRIP );
		if( false );
		else if( Fpre && Epre>emax && k>=kmin ){
		  eps.vertex2d( (k-Kpre)/(E-Epre)*(emax-Epre)+Kpre, emax );
		}
		else if( Fpre && Epre<emin && k>=kmin ){
		  eps.vertex2d( (k-Kpre)/(E-Epre)*(emin-Epre)+Kpre, emin );
		}
		else if( Fpre && Kpre<kmin ){
		  eps.vertex2d( kmin, (E-Epre)/(k-Kpre)*(kmin-Kpre)+Epre );
		}
	      }
	      eps.vertex2d( k, E );
	      n++;
	    }
	    else if(n>0){
	      if( false );
	      else if(Fpre && E>emax){
		eps.vertex2d( (k-Kpre)/(E-Epre)*(emax-Epre)+Kpre, emax );
	      }
	      else if(Fpre && E<emin){
		eps.vertex2d( (k-Kpre)/(E-Epre)*(emin-Epre)+Kpre, emin );
	      }
	      eps.end();
	      n=0;
	    }
	    Kpre=k;
	    Epre=E;
	    Fpre=true;
	  }
	  if(n>0){
	    eps.end(),n=0;
	  }
	}
	if( show_dw && model.band.isset2() ){
	  int n=0;
	  double Kpre=0.0,Epre=0.0;
	  bool   Fpre=false;
	  for( int i=0; i<model.band.sizeX(); i++ ){
	    double k = model.band.data[0][i].x;
	    double E = model.band.data[1][i].vy[j];
	    if( k<kmin || kmax<k ){
	      if( n>0 ){
		if( Fpre && k>kmax ){
		  eps.vertex2d( kmax, (E-Epre)/(k-Kpre)*(kmax-Kpre)+Epre );
		  eps.end();
		  n=0;
		}
	      }
	      Kpre=k;
	      Epre=E;
	      Fpre=true;
	      continue;
	    }
	    if( emin<=E && E<=emax ){
	      if( n==0 ){
		eps.begin( GL_LINE_STRIP );
		if( false );
		else if( Fpre && Epre>emax && k>=kmin ){
		  eps.vertex2d( (k-Kpre)/(E-Epre)*(emax-Epre)+Kpre, emax );
		}
		else if( Fpre && Epre<emin && k>=kmin ){
		  eps.vertex2d( (k-Kpre)/(E-Epre)*(emin-Epre)+Kpre, emin );
		}
		else if( Fpre && Kpre<kmin ){
		  eps.vertex2d( kmin, (E-Epre)/(k-Kpre)*(kmin-Kpre)+Epre );
		}
	      }
	      eps.vertex2d( k, E );
	      n++;
	    }
	    else if(n>0){
	      if( false );
	      else if(Fpre && E>emax){
		eps.vertex2d( (k-Kpre)/(E-Epre)*(emax-Epre)+Kpre, emax );
	      }
	      else if(Fpre && E<emin){
		eps.vertex2d( (k-Kpre)/(E-Epre)*(emin-Epre)+Kpre, emin );
	      }
	      eps.end();
	      n=0;
	    }
	    Kpre=k;
	    Epre=E;
	    Fpre=true;
	  }
	  if(n>0){
	    eps.end(),n=0;
	  }
	}
      }
      
      if( model.gloption.band.point.show ){
	eps.pointSize((double)model.gloption.band.point.size);

	if( show_up ){
	  eps.begin( GL_POINTS );
	  for( int i=0; i<model.band.sizeX(); i++ ){
	    double x = model.band.data[0][i].x;
	    double E = model.band.data[0][i].vy[j];
	    if( x<kmin || kmax<x ) continue;
	    if( E<emin || emax<E ) continue;
	    eps.vertex2d( x, E );
	  }
	  eps.end();
	}
	if( show_dw && model.band.isset2() ){
	  eps.begin( GL_POINTS );
	  for( int i=0; i<model.band.sizeX(); i++ ){
	    double x = model.band.data[0][i].x;
	    double E = model.band.data[1][i].vy[j];
	    if( x<kmin || kmax<x ) continue;
	    if( E<emin || emax<E ) continue;
	    eps.vertex2d( x, E );
	  }
	  eps.end();
	}

	eps.pointSize(1.0);
      }
    }

    eps.popMatrix();
  }

  eps.popMatrix();

  eps.close();

  return true;
}


void GLBand::mousePressEvent( QMouseEvent* ev )
{
  x_down = ev->x();
  y_down = ev->y();
}
void GLBand::mouseReleaseEvent( QMouseEvent* ev )
{
}


void GLBand::mouseMoveEvent( QMouseEvent* ev )
{
  int x = ev->x();
  int y = ev->y();

  int dx = x - x_down;
  int dy = y - y_down;

  if( abs(dy)<4 && abs(dx)<4 ) return;

  scroll(dx, dy);

  x_down = ev->x();
  y_down = ev->y();

  updateGL();
}

void GLBand::mouseDoubleClickEvent( QMouseEvent* ev )
{
}

void GLBand::contextMenuEvent( QContextMenuEvent* ev )
{
  popupmenu->exec(ev->globalPos());
}

void GLBand::menuEvent( QAction* action )
{
  double& kmin = model.gloption.band.kmin;
  double& kmax = model.gloption.band.kmax;
  double& emin = model.gloption.band.emin;
  double& emax = model.gloption.band.emax;

  static bool first = true;
  if( first ){
    path = QDir::currentPath();
    first = false;
  }

  if( false );

  else if( action->text() == "zoom energy in" ){
    const double yalpha = double(win_height-marginB-y_down)/(grp_height);
    const double erange = emax - emin;

    emax = emin + (yalpha*erange + 0.5/2*erange);
    emin = emin + (yalpha*erange - 0.5/2*erange);

    updateGL();
  }
  else if( action->text() == "zoom k-axis in" ){
    const double xalpha = double(x_down-marginL)/(grp_width);
    const double krange = kmax - kmin;
    kmax = kmin + xalpha*krange + 0.5/4*krange;
    kmin = kmin + xalpha*krange - 0.5/4*krange;
    if( kmin < model.band.kmin ){
      kmin = model.band.kmin;
    }
    if( kmax > model.band.kmax ){
      kmax = model.band.kmax;
    }
    round( kmax, kmin );

    updateGL();
  }
  else if( action->text() == "zoom energy out" ){
    const double yalpha = double(win_height-marginB-y_down)/(grp_height);
    const double erange = emax - emin;

    emax = emin + (yalpha*erange + 0.5*2*erange);
    emin = emin + (yalpha*erange - 0.5*2*erange);

    updateGL();
  }
  else if( action->text() == "zoom k-axis out" ){
    const double xalpha = double(x_down-marginL)/(grp_width);
    const double krange = kmax - kmin;
    kmax = kmin + xalpha*krange + 0.5*4*krange;
    kmin = kmin + xalpha*krange - 0.5*4*krange;
    if( kmin < model.band.kmin ){
      kmin = model.band.kmin;
    }
    if( kmax > model.band.kmax ){
      kmax = model.band.kmax;
    }
    round( kmax, kmin );

    updateGL();
  }
  else if( action->text() == "zoom reset" ){
    emin = model.band.emin;
    emax = model.band.emax;
    kmin = model.band.kmin;
    kmax = model.band.kmax;

    updateGL();
  }

  else if( action->text() == "both spin" ){
    show_up = true;
    show_dw = true;
    action_both_spin->setChecked( show_up && show_dw );
    action_up_spin->setChecked( show_up && !show_dw );
    action_dw_spin->setChecked( !show_up && show_dw );
    updateGL();
  }
  else if( action->text() == "up   spin" ){
    show_up = true;
    show_dw = false;
    action_both_spin->setChecked( show_up && show_dw );
    action_up_spin->setChecked( show_up && !show_dw );
    action_dw_spin->setChecked( !show_up && show_dw );
    updateGL();
  }
  else if( action->text() == "down spin" ){
    show_up = false;
    show_dw = true;
    action_both_spin->setChecked( show_up && show_dw );
    action_up_spin->setChecked( show_up && !show_dw );
    action_dw_spin->setChecked( !show_up && show_dw );
    updateGL();
  }

  else if( action->text() == "default order" ){
    model.band.order = DTBand::DEFAULT_ORDER;
    model.band.update();
    action_def_ord->setChecked( model.band.order == DTBand::DEFAULT_ORDER );
    action_asc_ord->setChecked( model.band.order == DTBand::ASCENDING_ORDER );
    action_nat_ord->setChecked( model.band.order == DTBand::NATUAL_ORDER );
    updateGL();
  }
  else if( action->text() == "ascending order" ){
    model.band.order = DTBand::ASCENDING_ORDER;
    model.band.update();
    action_def_ord->setChecked( model.band.order == DTBand::DEFAULT_ORDER );
    action_asc_ord->setChecked( model.band.order == DTBand::ASCENDING_ORDER );
    action_nat_ord->setChecked( model.band.order == DTBand::NATUAL_ORDER );
    updateGL();
  }
  else if( action->text() == "natural order" ){
    model.band.order = DTBand::NATUAL_ORDER;
    model.band.update();
    action_def_ord->setChecked( model.band.order == DTBand::DEFAULT_ORDER );
    action_asc_ord->setChecked( model.band.order == DTBand::ASCENDING_ORDER );
    action_nat_ord->setChecked( model.band.order == DTBand::NATUAL_ORDER );
    updateGL();
  }

  else if( action->text() == "band colorful" ){
    model.gloption.band.monochrome  = !model.gloption.band.monochrome;
    action_colorful->setChecked( !model.gloption.band.monochrome );
    updateGL();
  }
  else if( action->text() == "band lines" ){
    model.gloption.band.line.show  = !model.gloption.band.line.show;
    action_lines->setChecked( model.gloption.band.line.show );
    updateGL();
  }
  else if( action->text() == "band points" ){
    model.gloption.band.point.show = !model.gloption.band.point.show;
    action_points->setChecked( model.gloption.band.point.show );
    updateGL();
  }
  else if( action->text() == "save image" ){
    const QString fname = QFileDialog::getSaveFileName
      ( this, tr("Save image in file"), path,
	tr("Windows Bitmap file(*.bmp);;"
	   "Encapsulated PostScript file(*.eps);;"
	   "All files(*)"));
    
    if( fname.endsWith( ".bmp" ) ){
      if( glSavePixels( qPrintable(fname) ) ){
	path = getDirName(fname);
      }
      else{
	MyException::critical("can not create a file.",fname);
      }
    }
    else if( fname.endsWith( ".eps" ) ){
      if( paintEPS( qPrintable(fname) ) ){
	path = getDirName(fname);
      }
      else{
	MyException::critical("can not create a file.",fname);
      }
    }
    else if( fname !="" ){
      MyException::critical("unknown suffix.",fname);
    }
  }
}


void GLBand::scroll( int ix, int iy )
{
  double& kmin = model.gloption.band.kmin;
  double& kmax = model.gloption.band.kmax;
  double& emin = model.gloption.band.emin;
  double& emax = model.gloption.band.emax;

  const double dy = double(iy)/(grp_height)* (emax-emin);
  const double dx = double(ix)/(grp_width)* (kmax-kmin)*(-1.0);
  kmin += dx;
  kmax += dx;
  emin += dy;
  emax += dy;
  if( kmin<model.band.kmin || model.band.kmax<kmax ){
    kmin -= dx;
    kmax -= dx;
  }
  if( emin<model.band.emin || model.band.emax<emax ){
    emin -= dy;
    emax -= dy;
  }
}


void GLBand::wheelEvent( QWheelEvent* ev )
{
  if( ev->orientation() == Qt::Horizontal ) return;

  const int dy = 1*(ev->delta()/120);
  const int dx = 0;

  scroll( dx, dy );

  updateGL();
}

void GLBand::keyPressEvent( QKeyEvent* ev )
{
  switch( ev->key() ){
  case Qt::Key_Left     :  scroll( -4,  0 ); break;
  case Qt::Key_Right    :  scroll( +4,  0 ); break;
  case Qt::Key_Up       :  scroll(  0, -4 ); break;
  case Qt::Key_Down     :  scroll(  0, +4 ); break;
    //  case Qt::Key_PageDown : break;
    //  case Qt::Key_PageUp   : break;
  default : break;
  }

  updateGL();
}


void GLBand::keyReleaseEvent( QKeyEvent* ev )
{
  if( ev->isAutoRepeat() ) return;

  switch( ev->key() ){
  case Qt::Key_Left   : break;
  case Qt::Key_Right  : break;
  case Qt::Key_Down   : break;
  case Qt::Key_Up     : break;
    //  case Qt::Key_PageDown : break;
    //  case Qt::Key_PageUp   : break;
  default             : return;
  }
}
