/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

/*!
 \file glbonds.cc
 \brief iq̌GL\̃NX
*/

#include <GL/gl.h>
#include "dtmodel.h"
#include "glbonds.h"

void GLBonds::update( void )
{
  const Position& La = model.lattice.cell.La;
  const Position& Lb = model.lattice.cell.Lb;
  const Position& Lc = model.lattice.cell.Lc;
  const Position& Lo = model.lattice.cell.Lo;
  const vector<DTAtom>& vatom = model.lattice.getAtoms();
  const bool    show_copy = model.gloption.lattice.bond.show_copy;

  GLBase::beginNewList();

  glPushMatrix();
  glTranslated( Lo.x, Lo.y, Lo.z );

  int natom=0;
  for( int i=0; i<(int)vatom.size(); i++ ){
    if( vatom[i].isDuplicate() ) continue;
    if( vatom[i].isCopy() && !show_copy ) continue;
    natom++;
  }

  // in case a small molecule
  if( natom<100 ){
    for( int i=0; i<(int)vatom.size(); i++ ){
      if( vatom[i].isDuplicate() ) continue;

      for( int j=i+1; j<(int)vatom.size(); j++ ){
	if( vatom[j].isDuplicate() ) continue;
	if( vatom[j].isCopy() && !show_copy ) continue;
	makeBond( vatom[i], vatom[j] );
      }
    }

    if( model.gloption.location.extendA>0 ){
      for( int i=0; i<(int)vatom.size(); i++ ){
	if( vatom[i].isDuplicate() ) continue;
	for( int j=0; j<(int)vatom.size(); j++ ){
	  if( vatom[j].isDuplicate() ) continue;
	  if( vatom[j].isCopy() && !show_copy ) continue;
	  makeBond( vatom[i], vatom[j], La );
	}
      }
    }

    if( model.gloption.location.extendB>0 ){
      for( int i=0; i<(int)vatom.size(); i++ ){
	if( vatom[i].isDuplicate() ) continue;
	for( int j=0; j<(int)vatom.size(); j++ ){
	  if( vatom[j].isDuplicate() ) continue;
	  if( vatom[j].isCopy() && !show_copy ) continue;
	  makeBond( vatom[i], vatom[j], Lb );
	}
      }
    }

    if( model.gloption.location.extendC>0 ){
      for( int i=0; i<(int)vatom.size(); i++ ){
	if( vatom[i].isDuplicate() ) continue;
	for( int j=0; j<(int)vatom.size(); j++ ){
	  if( vatom[j].isDuplicate() ) continue;
	  if( vatom[j].isCopy() && !show_copy ) continue;
	  makeBond( vatom[i], vatom[j], Lc );
	}
      }
    }

    if( model.gloption.location.extendA>0 &&
	model.gloption.location.extendB>0 ){
      for( int i=0; i<(int)vatom.size(); i++ ){
	if( vatom[i].isDuplicate() ) continue;
	for( int j=0; j<(int)vatom.size(); j++ ){
	  if( vatom[j].isDuplicate() ) continue;
	  if( vatom[j].isCopy() && !show_copy ) continue;
	  makeBond( vatom[i], vatom[j], La+Lb );
	  makeBond( vatom[i], vatom[j], La-Lb );
	}
      }
    }

    if( model.gloption.location.extendB>0 &&
	model.gloption.location.extendC>0 ){
      for( int i=0; i<(int)vatom.size(); i++ ){
	if( vatom[i].isDuplicate() ) continue;
	for( int j=0; j<(int)vatom.size(); j++ ){
	  if( vatom[j].isDuplicate() ) continue;
	  if( vatom[j].isCopy() && !show_copy ) continue;
	  makeBond( vatom[i], vatom[j], Lb+Lc );
	  makeBond( vatom[i], vatom[j], Lb-Lc );
	}
      }
    }

    if( model.gloption.location.extendC>0 &&
	model.gloption.location.extendA>0 ){
      for( int i=0; i<(int)vatom.size(); i++ ){
	if( vatom[i].isDuplicate() ) continue;
	for( int j=0; j<(int)vatom.size(); j++ ){
	  if( vatom[j].isDuplicate() ) continue;
	  if( vatom[j].isCopy() && !show_copy ) continue;
	  makeBond( vatom[i], vatom[j], Lc+La );
	  makeBond( vatom[i], vatom[j], Lc-La );
	}
      }
    }
    if( model.gloption.location.extendA>0 &&
	model.gloption.location.extendB>0 &&
	model.gloption.location.extendC>0 ){
      for( int i=0; i<(int)vatom.size(); i++ ){
	if( vatom[i].isDuplicate() ) continue;
	for( int j=0; j<(int)vatom.size(); j++ ){
	  if( vatom[j].isDuplicate() ) continue;
	  if( vatom[j].isCopy() && !show_copy ) continue;
	  makeBond( vatom[i], vatom[j], La+Lb+Lc );
	  makeBond( vatom[i], vatom[j], La-Lb+Lc );
	  makeBond( vatom[i], vatom[j], La+Lb-Lc );
	  makeBond( vatom[i], vatom[j], La-Lb-Lc );
	}
      }
    }

  }
  glPopMatrix();

  GLBase::endNewList();
}


void GLBonds::makeBond( const DTAtom& atom1, const DTAtom& atom2 )
{
  const int   number1 = atom1.element.number;
  const int   number2 = atom2.element.number;
  const Coordinates coords1 = model.gloption.location.scroll(atom1.coords);
  const Coordinates coords2 = model.gloption.location.scroll(atom2.coords);

  const Position pos1 = model.lattice.cell.getPositionLocal(coords1);
  const Position pos2 = model.lattice.cell.getPositionLocal(coords2);

  const GLOption::Lattice::Bond::byElement& element = model.gloption.lattice.bond.element(number1,number2);
  const double& length = element.length;
  const GLcolor& color = element.color;

  const double scale   = model.gloption.lattice.bond.scale;
  const int    slices  = (int)model.gloption.lattice.bond.slices;
  const bool   lines   = model.gloption.lattice.bond.lines;

  const double eps = 1.0e-4;
  if( Position::length2( pos1, pos2 ) > length*length+eps ) return;

  glColor4dv( color );
  glMaterialdv( GL_FRONT, GL_AMBIENT_AND_DIFFUSE, color );

  if( lines ){
    glDisable( GL_LIGHTING );
    glLineWidth( scale*16.0);
    glBegin( GL_LINES );
    glVertex3dv( pos1 );
    glVertex3dv( pos2 );
    glEnd();
    glLineWidth(1.0);
    glEnable( GL_LIGHTING );
  }else{
    gluSolidStick( scale, pos1, pos2, slices );
  }
}


void GLBonds::makeBond( const DTAtom& atom1, const DTAtom& atom2, const Position& L )
{
  const int   number1 = atom1.element.number;
  const int   number2 = atom2.element.number;
  const Coordinates coords1 = model.gloption.location.scroll(atom1.coords);
  const Coordinates coords2 = model.gloption.location.scroll(atom2.coords);

  const Position pos1 = model.lattice.cell.getPositionLocal(coords1);
  const Position pos2 = model.lattice.cell.getPositionLocal(coords2) + L;

  const GLOption::Lattice::Bond::byElement& element = model.gloption.lattice.bond.element(number1,number2);
  const double& length = element.length;
  const GLcolor& color = element.color;

  const double scale   = model.gloption.lattice.bond.scale;
  const int    slices  = (int)model.gloption.lattice.bond.slices;
  const bool   lines   = model.gloption.lattice.bond.lines;

  const double eps = 1.0e-4;
  if( Position::length2( pos1, pos2 ) > length*length+eps ) return;

  glColor4dv( color );
  glMaterialdv( GL_FRONT, GL_AMBIENT_AND_DIFFUSE, color );

  if( lines ){
    glDisable( GL_LIGHTING );
    glLineWidth( scale*16.0);
    glBegin( GL_LINES );
    glVertex3dv( pos1 );
    glVertex3dv( pos2 );
    glEnd();
    glLineWidth(1.0);
    glEnable( GL_LIGHTING );
  }else{
    gluSolidStick( scale, pos1, pos2, slices );
  }
}
