/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

/*!
 \file glbrillouin.cc
 \brief uA][GL\̃NX
*/

#include <GL/gl.h>
#include "dtmodel.h"
#include "qtmisc.h"
#include "glbrillouin.h"
#include <QtOpenGL/QGLWidget>

GLBrillouin::GLBrillouin( DTModel& _model ) : model(_model)
{
  setFocusPolicy( Qt::ClickFocus );
  setWindowTitle( model.gloption.window.title );
  setMinimumSize( 512-128, 256+128 );
  setSizePolicy( QSizePolicy::Expanding, QSizePolicy::Expanding );

  resize( model.gloption.window.width, model.gloption.window.height );

  connect( &model, SIGNAL(changed()),   this, SLOT(update()) );
  connect(   this, SIGNAL(changed()), &model, SLOT(update()) );

  popupmenu = new QMenu(this);
  popupmenu->addSeparator();
  popupmenu->addAction( new QAction(tr("change perspective/orthogonal"),this) );
  popupmenu->addAction( new QAction(tr("reset view"),this) );
  popupmenu->addAction( new QAction(tr("save image"),this) );
  connect(popupmenu, SIGNAL(triggered(QAction*)), this, SLOT(menuEvent(QAction*)));

  perspective = false;
}

void GLBrillouin::update( void )
{
  makeCurrent();
  updateGL();
}



void GLBrillouin::initializeGL( void )
{
  glClearColor( model.gloption.window.background );

  glMatrixMode( GL_MODELVIEW );
  glLoadIdentity();
  gluLookAt( model.gloption.location.eye,
	     model.gloption.location.gaze,
	     model.gloption.location.up );

  glEnable( GL_DEPTH_TEST );
  glEnable( GL_NORMALIZE );

  glShadeModel( GL_SMOOTH );
  glLightModeli(  GL_LIGHT_MODEL_TWO_SIDE, 1 );
  glLightModeli(  GL_LIGHT_MODEL_LOCAL_VIEWER, 1 );
  glLightModeldv( GL_LIGHT_MODEL_AMBIENT, model.gloption.light.ambient );
  glEnable( GL_LIGHTING );

  glLightdv( GL_LIGHT0, GL_POSITION, model.gloption.light.direction );
  glLightdv( GL_LIGHT0, GL_DIFFUSE,  model.gloption.light.diffuse );
  glLightdv( GL_LIGHT0, GL_SPECULAR, model.gloption.light.specular );
  glEnable( GL_LIGHT0 );

  glLightdv( GL_LIGHT1, GL_POSITION, model.gloption.light.second_direction );

  glLightdv( GL_LIGHT1, GL_DIFFUSE,  model.gloption.light.diffuse );
  glLightdv( GL_LIGHT1, GL_SPECULAR, model.gloption.light.specular );
  if( model.gloption.light.second ){
    glEnable( GL_LIGHT1 );
  }

  glFogf( GL_FOG_MODE, GL_LINEAR );
  glFogf( GL_FOG_START,   model.gloption.location.world );
  glFogf( GL_FOG_END, 1.6*model.gloption.location.world );
  glFogfv( GL_FOG_COLOR,  model.gloption.window.background );
  if( model.gloption.light.fog ){
    glEnable( GL_FOG );
  }

  glMaterialdv( GL_FRONT_AND_BACK, GL_SPECULAR,  model.gloption.light.specular  );
  glMateriald ( GL_FRONT_AND_BACK, GL_SHININESS, model.gloption.light.shininess );

  glClearColor( model.gloption.window.background );
  glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
}


void GLBrillouin::gluProjection( int width, int height, bool perspective )
{
  if( perspective ){
    const double aspect = (double)width/height;
    gluPerspective( model.gloption.location.pangle,
		    aspect,
		    model.gloption.location.pnear,
		    model.gloption.location.pfar );
  }else{
    const double t = tan(model.gloption.location.pangle*0.5*M_PI/180.0);
    const double aspect = (double)width/height;
    const double top    = 0.65*sqrt(model.gloption.location.pnear*model.gloption.location.pfar) * t;
    const double bottom = -top;
    const double right  = top*aspect;
    const double left   = -right;

    glOrtho( left, right, bottom, top, 
	     model.gloption.location.pnear,
	     model.gloption.location.pfar );
  }
}

void GLBrillouin::resizeGL(int width, int height)
{
  glViewport( 0, 0, width, height );

  glMatrixMode( GL_PROJECTION );
  glLoadIdentity();

  gluProjection( width, height, perspective );

  glMatrixMode( GL_MODELVIEW );
  
  glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
}

void GLBrillouin::paintGL( void )
{
  if( model.gloption.light.second ){
    glEnable( GL_LIGHT1 );
  }else{
    glDisable( GL_LIGHT1 );
  }
  if( model.gloption.light.fog ){
    glFogfv( GL_FOG_COLOR, model.gloption.window.background );
    glEnable( GL_FOG );
  }else{
    glDisable( GL_FOG );
  }
  glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

  draw();
}

// mouse drag begin
void GLBrillouin::mousePressEvent( QMouseEvent* ev )
{
  if( !(ev->buttons()&Qt::LeftButton) ) return;

  int x = ev->x(), y = ev->y();

  x_down = x, y_down = y;
}

// mouse selection
void GLBrillouin::mouseDoubleClickEvent( QMouseEvent* ev )
{
  static GLuint data[1024];
  glSelectBuffer( sizeof(data)/sizeof(GLuint), data );

  glRenderMode(GL_SELECT);
  glInitNames();
  glPushName( GLuint(-1) );

  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();

  struct { GLint xo, yo, width, height; } viewport;
  glGetIntegerv( GL_VIEWPORT, (GLint*)&viewport );
  
  int xo, yo;
  int xw, yw;

  xo = ev->x();
  yo = ev->y();
  xw = 8;
  yw = 8;

  gluPickMatrix( xo, viewport.height-1 - yo,
		  xw, yw, (GLint*)&viewport );

  gluProjection( viewport.width, viewport.height, perspective );
  glMatrixMode(GL_MODELVIEW);

  select();

  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);

  GLuint hits = glRenderMode(GL_RENDER);
  GLuint selected = GLuint(-1);
  GLuint zminmin  = GLuint(-1);

  if( hits == 0 ){
    model.lattice.getData().unselectAtom();
    emit changed();
    makeCurrent();
    return;
  }

  for( GLuint i=0,j=0; i<hits; i++ ){
    GLuint n    = data[j++];
    GLuint zmin = data[j++];
    j++; // zmax
    if( n==1 ){
      // ԋ߂̂I
      if( zminmin>zmin && data[j] != GLuint(-1) ){
	zminmin = zmin;
	selected = data[j++];
      }else{
	j++;
      }
    }
  }

  if( model.lattice.selectBrillouin((int)selected) ){
    emit changed();
    makeCurrent();
  }

  x_down = ev->x(), y_down = ev->y();
}

// mouse drag end
void GLBrillouin::mouseReleaseEvent( QMouseEvent* ev )
{
}

// mouse drag
void GLBrillouin::mouseMoveEvent( QMouseEvent* ev )
{
  if( !(ev->buttons()&Qt::LeftButton) ) return;

  int x = ev->x(), y = ev->y();

  const double dx = (1.0/4)*(x-x_down);
  const double dy = (1.0/4)*(y-y_down);

  if( fabs(dx) < 1.0 && fabs(dy) < 1.0 ) return;

  glRotateR( dy, 1.0, 0.0, 0.0 );
  glRotateR( dx, 0.0, 1.0, 0.0 );

  x_down = x, y_down = y;

  updateGL();
}

void GLBrillouin::wheelEvent( QWheelEvent* ev )
{
  if( ev->orientation() == Qt::Horizontal ) return;

  const double dy = (-2.0) * (double)ev->delta() / 120.0;

  {
    const double f = 1.0 + (1.0/256)*dy;
    glScaled(f,f,f);
  }

  updateGL();
}

void GLBrillouin::keyPressEvent( QKeyEvent* ev )
{
  double dx=0.0, dy=0.0, dz=0.0;
  switch( ev->key() ){
  case Qt::Key_Left     : dx=-1.0; break;
  case Qt::Key_Right    : dx=+1.0; break;
  case Qt::Key_Up       : dy=-1.0; break;
  case Qt::Key_Down     : dy=+1.0; break;
  case Qt::Key_PageDown : dz=-1.0; break;
  case Qt::Key_PageUp   : dz=+1.0; break;
  default : break;
  }

  if( ev->modifiers() & Qt::ControlModifier ) {
    glRotateR( 15*dy, 1.0, 0.0, 0.0 );
    glRotateR( 15*dx, 0.0, 1.0, 0.0 );
    glRotateR( 15*dz, 0.0, 0.0, 1.0 );
  }
  else{
    switch( ev->key() ){
    case Qt::Key_PageDown :  {
      const double f = 1.0 + (1.0/8);
      glScaled(f,f,f);
    } break; 
    case Qt::Key_PageUp :  {
      const double f = 1.0 - (1.0/8);
      glScaled(f,f,f);
    } break;
    case Qt::Key_Down     : {
      const double dy=-15.0;
      glTranslateR( 0.0, (1.0/4)*(dy), 0.0 );
    } break;
    case Qt::Key_Up       : {
      const double dy=+15.0;
      glTranslateR( 0.0, (1.0/4)*(dy), 0.0 );
    } break;
    case Qt::Key_Left     : {
      const double dx=-15.0;
      glTranslateR( (1.0/4)*(dx), 0.0, 0.0 );
    } break;
    case Qt::Key_Right    : {
      const double dx=+15.0;
      glTranslateR( (1.0/4)*(dx), 0.0, 0.0 );
    } break;

    default : break;
    }
  }

  updateGL();
}


void GLBrillouin::keyReleaseEvent( QKeyEvent* ev )
{
  if( ev->isAutoRepeat() ) return;

  switch( ev->key() ){
  case Qt::Key_Left   : break;
  case Qt::Key_Right  : break;
  case Qt::Key_Down   : break;
  case Qt::Key_Up     : break;
  case Qt::Key_PageDown : break;
  case Qt::Key_PageUp   : break;
  default             : return;
  }
}

void GLBrillouin::contextMenuEvent( QContextMenuEvent* ev )
{
  popupmenu->exec(ev->globalPos());
}

void GLBrillouin::menuEvent( QAction* action )
{
  if( false ){
  }
  else if( action->text() == "reset view" ){
    glLoadIdentity();
    gluLookAt( model.gloption.location.eye,
	       model.gloption.location.gaze,
	       model.gloption.location.up );
    model.gloption.light.direction =
      model.gloption.light.direction_orig;
    glLightdv( GL_LIGHT0, GL_POSITION,
	       model.gloption.light.direction );
  }
  else if( action->text() == "change perspective/orthogonal" ){
    perspective = !perspective;
    resizeGL( glGetWindowWidth(), glGetWindowHeight() );
  }
  else if( action->text() == "save image" ){
    save();
  }
}


void GLBrillouin::select( void )
{
  const Position& Ka = model.lattice.cell.Ka;
  const Position& Kb = model.lattice.cell.Kb;
  const Position& Kc = model.lattice.cell.Kc;
  const Position& Ka_conv = model.lattice.cell.Ka_conv;
  const Position& Kb_conv = model.lattice.cell.Kb_conv;
  const Position& Kc_conv = model.lattice.cell.Kc_conv;
  const vector<BrillouinSegment>& vsymmL = model.lattice.cell.vsymmL;
  const vector_both<BrillouinSegment>& vsymmLselected = model.lattice.cell.vsymmLselected;

  glPushMatrix();

  double Kscale=0.0;
  {
    for( int ika=-1;ika<=+1; ika+=2 ){
      for( int ikb=-1;ikb<=+1; ikb+=2 ){
	for( int ikc=-1;ikc<=+1; ikc+=2 ){
	  const Position L = 
	    +Ka_conv*double(ika)
	    +Kb_conv*double(ikb)
	    +Kc_conv*double(ikc);
	  const double length = Position::length(L);
	  if( Kscale < length ){
	    Kscale = length;
	  }
	}
      }
    }
  }

  const double f = model.gloption.location.world/Kscale*(0.25);
  glScaled(f,f,f);

  // symmetric lines
  int s;
  for( s=0; s<(int)vsymmL.size(); s++ ){
    Position ks =
      + Ka * vsymmL[s].positions.a
      + Kb * vsymmL[s].positions.b 
      + Kc * vsymmL[s].positions.c;
    Position ke =
      + Ka * vsymmL[s].positione.a
      + Kb * vsymmL[s].positione.b 
      + Kc * vsymmL[s].positione.c;
    glLoadName(s);
    glBegin( GL_LINES );
    glVertex3dv( ks );
    glVertex3dv( ke );
    glEnd();
  }

  // symmetric lines selected
  for( int i=0; i<(int)vsymmLselected.size(); i++ ){
    const BrillouinSegment& symmL = vsymmLselected[i];

    Position ks =
      + Ka * symmL.positions.a
      + Kb * symmL.positions.b 
      + Kc * symmL.positions.c;
    Position ke =
      + Ka * symmL.positione.a
      + Kb * symmL.positione.b 
      + Kc * symmL.positione.c;

    glLoadName(s);
    glBegin( GL_LINES );
    glVertex3dv( ks );
    glVertex3dv( ke );
    glEnd();
  }

  glLoadName(GLuint(-1));

  glPopMatrix();
}

void GLBrillouin::draw( void )
{
  const Position& Ka = model.lattice.cell.Ka;
  const Position& Kb = model.lattice.cell.Kb;
  const Position& Kc = model.lattice.cell.Kc;
  const Position& Ka_conv = model.lattice.cell.Ka_conv;
  const Position& Kb_conv = model.lattice.cell.Kb_conv;
  const Position& Kc_conv = model.lattice.cell.Kc_conv;

  const vector<BrillouinFacet>& vfacet = model.lattice.cell.vfacet;
  const vector<BrillouinSegment>& vsymmL = model.lattice.cell.vsymmL;
  const vector_both<BrillouinSegment>& vsymmLselected = model.lattice.cell.vsymmLselected;
  GLcolor color = model.gloption.window.foreground;
  const QFont font("Helvetica", 12);

  glDisable( GL_LIGHTING );
  glDisable( GL_DEPTH_TEST );

  glPushMatrix();

  double Kscale=0.0;
  {
    for( int ika=-1;ika<=+1; ika+=2 ){
      for( int ikb=-1;ikb<=+1; ikb+=2 ){
	for( int ikc=-1;ikc<=+1; ikc+=2 ){
	  const Position L = 
	    +Ka_conv*double(ika)
	    +Kb_conv*double(ikb)
	    +Kc_conv*double(ikc);
	  const double length = Position::length(L);
	  if( Kscale < length ){
	    Kscale = length;
	  }
	}
      }
    }
  }

  const double f = model.gloption.location.world/Kscale*(0.25);
  glScaled(f,f,f);

  glLineWidth(1.0);
  glColor4dv( GLcolor::cyan );


  // primitive k-axes
  glBegin( GL_LINES );
  {
    glVertex3dv( Position(0.0,0.0,0.0) );
    glVertex3dv( Ka );
    glVertex3dv( Position(0.0,0.0,0.0) );
    glVertex3dv( Kb );
    glVertex3dv( Position(0.0,0.0,0.0) );
    glVertex3dv( Kc );
  }
  glEnd();

  {
    this->renderText( Ka.x, Ka.y, Ka.z, QString("Ka"), font );
    this->renderText( Kb.x, Kb.y, Kb.z, QString("Kb"), font );
    this->renderText( Kc.x, Kc.y, Kc.z, QString("Kc"), font );
  }

  glColor4dv( GLcolor::yellow );
  // symmetric lines
  for( int s=0; s<(int)vsymmL.size(); s++ ){
    Position ks =
      + Ka * vsymmL[s].positions.a
      + Kb * vsymmL[s].positions.b 
      + Kc * vsymmL[s].positions.c;
    Position ke =
      + Ka * vsymmL[s].positione.a
      + Kb * vsymmL[s].positione.b 
      + Kc * vsymmL[s].positione.c;
    glBegin( GL_LINES );
    glVertex3dv( ks );
    glVertex3dv( ke );
    glEnd();
  }

  glColor4dv( GLcolor::orange );
  glLineWidth(4.0);

  // symmetric lines selected
  for( int i=0; i<(int)vsymmLselected.size(); i++ ){
    const BrillouinSegment& symmL = vsymmLselected[i];

    Position ks =
      + Ka * symmL.positions.a
      + Kb * symmL.positions.b 
      + Kc * symmL.positions.c;
    Position ke =
      + Ka * symmL.positione.a
      + Kb * symmL.positione.b 
      + Kc * symmL.positione.c;
    glBegin( GL_LINES );
    glVertex3dv( ks );
    glVertex3dv( ke );
    glEnd();
  }
  glLineWidth(1.0);

  // start point of symmetric lines selected
  if( !vsymmLselected.empty() ){
    const BrillouinSegment& symmL = vsymmLselected.front();
    Position ks =
      + Ka * symmL.positions.a
      + Kb * symmL.positions.b
      + Kc * symmL.positions.c;

    glPointSize(12.0);
    glColor4dv( GLcolor::red );
    glBegin( GL_POINTS );
    glVertex3dv( ks );
    glEnd();
    glPointSize(1.0);
  }

  // end point of symmetric lines selected
  if( !vsymmLselected.empty() ){
    const BrillouinSegment& symmL = vsymmLselected.back();
    Position ke =
      + Ka * symmL.positione.a
      + Kb * symmL.positione.b
      + Kc * symmL.positione.c;
    glPointSize(8.0);
    glColor4dv( GLcolor::yellow );
    glBegin( GL_POINTS );
    glVertex3dv( ke );
    glEnd();
    glPointSize(1.0);
  }

  // facet
  glColor4dv( model.gloption.lattice.cell.color );
  for( int f=0; f<(int)vfacet.size(); f++ ){
    glBegin( GL_LINE_LOOP );
    for( int s=0; s<(int)vfacet[f].vsegment.size(); s++ ){
      glVertex3dv( vfacet[f].vsegment[s].positions );
    }
    glEnd();
  }

  glColor4dv( color );

  // labels of symmetric lines
  for( int s=0; s<(int)vsymmL.size(); s++ ){
    Position ks =
      + Ka * vsymmL[s].positions.a
      + Kb * vsymmL[s].positions.b 
      + Kc * vsymmL[s].positions.c;
    Position ke =
      + Ka * vsymmL[s].positione.a
      + Kb * vsymmL[s].positione.b 
      + Kc * vsymmL[s].positione.c;
    Position kc = (ks+ke)*0.5;

    this->renderText( ks.x, ks.y, ks.z, vsymmL[s].labelPs, font );
    this->renderText( kc.x, kc.y, kc.z, vsymmL[s].labelL , font  );

    if( (s+1==(int)vsymmL.size()) ||
	(s+1< (int)vsymmL.size() &&
	 !Position::match( vsymmL[s].positione, vsymmL[s+1].positions )) ){


      this->renderText( ke.x, ke.y, ke.z, vsymmL[s].labelPe, font );
    }
  }

  // labels of symmetric lines selected
  int count=0;
  for( int i=0; i<(int)vsymmLselected.size(); i++ ){
    const BrillouinSegment& symmL = vsymmLselected[i];

    Position ks =
      + Ka * symmL.positions.a
      + Kb * symmL.positions.b 
      + Kc * symmL.positions.c;
    Position ke =
      + Ka * symmL.positione.a
      + Kb * symmL.positione.b 
      + Kc * symmL.positione.c;
    Position kc = ks*0.75 +ke*0.25;

    char number[8];
    sprintf(number,"%d",++count);
    this->renderText( kc.x, kc.y, kc.z, QString(number), font );
  }

  glPopMatrix();

  {
    const int win_width  = glGetWindowWidth();
    const int win_height = glGetWindowHeight();

    int xo = 16;
    int yo = win_height-16;

    glColor4dv( GLcolor::white );

    glMatrixMode( GL_MODELVIEW );
    glPushMatrix();
    glLoadIdentity();
  
    glMatrixMode( GL_PROJECTION );
    glPushMatrix();
    glLoadIdentity();
    gluOrtho2D( 0.0, (GLfloat)win_width, (GLfloat)win_height, 0.0 );

    char label[256];
    sprintf(label,"Bravais lattice: %s %s %s",
	    qPrintable(model.lattice.cell.bravais.shape),
	    qPrintable(model.lattice.cell.bravais.center),
	    qPrintable(model.lattice.cell.bravais.subtype) );

    this->renderText( xo, yo, QString(label), font );

    glPopMatrix();
    glMatrixMode( GL_MODELVIEW );
    glPopMatrix();
  }

  glEnable( GL_DEPTH_TEST );
  glEnable( GL_LIGHTING );
}


bool GLBrillouin::save( void )
{
  static int cell=0;
  static char fname[256];

  snprintf( fname, sizeof(fname), "brillouin%02d.bmp", cell++ );

  return glSavePixels( fname );
}
