/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

/*!
 \file glcrosec.cc
 \brief OtB[h̒fʐ}GL\̃NX
*/

#include <GL/gl.h>
#include "dtmodel.h"
#include "glmisc.h"
#include "glcrosec.h"
#include "glspline.h"
#include "vector2d.h"

void GLCroSec::makeFrame
( const vector2d<double>& data, const Coordinates& scroll )
{
  glEnable( GL_BLEND );
  glBlendFunc( GL_SRC_ALPHA, GL_ONE );

  glBegin( GL_LINE_LOOP );
  glColor4dv( GLcolor( GLcolor::white, 1.0/8 ) );
  glVertex3d( 0.0, 0.0, 0.0 );
  glVertex3d( 1.0, 0.0, 0.0 );
  glVertex3d( 1.0, 1.0, 0.0 );
  glVertex3d( 0.0, 1.0, 0.0 );
  glEnd();

  glDisable( GL_BLEND );
}

void GLCroSec::makePlaneR
( const vector2d<double>& data, const Coordinates& scroll )
{
  if( model.gloption.field.crosec.plane.alpha == 1.0 ){
    glDisable( GL_BLEND );
    glDepthMask( true );
  }
  else{
    glEnable( GL_BLEND );
    glBlendFunc( GL_SRC_ALPHA, GL_ONE );
    glDepthMask( false );
  }

  const int Nx = data.sizeX();
  const int Ny = data.sizeY();
  const double dLx = 1.0/Nx;
  const double dLy = 1.0/Ny;

  glBegin( GL_QUADS );
  for( int ix=0; ix<Nx; ix++ ){
    const double x0 = fmod(ix*dLx+scroll.x,1.0);
    const double x1 = x0+dLx;
    for( int iy=0; iy<Ny; iy++ ){
      const double y0 = fmod(iy*dLy+scroll.y,1.0);
      const double y1 = y0+dLy;

      GLcolor color;
      color   = model.gloption.field.gradation(data(ix,iy));
      color.A = model.gloption.field.crosec.plane.alpha;
      glColor4dv( color );
      glVertex3d( x0, y0, 0.0 );

      color   = model.gloption.field.gradation(data((ix+1)%Nx,iy));
      color.A = model.gloption.field.crosec.plane.alpha;
      glColor4dv( color );
      glVertex3d( x1, y0, 0.0 );

      color   = model.gloption.field.gradation(data((ix+1)%Nx,(iy+1)%Ny));
      color.A = model.gloption.field.crosec.plane.alpha;
      glColor4dv( color );
      glVertex3d( x1, y1, 0.0 );

      color   = model.gloption.field.gradation(data(ix,(iy+1)%Ny));
      color.A = model.gloption.field.crosec.plane.alpha;
      glColor4dv( color );
      glVertex3d( x0, y1, 0.0 );
    }
  }
  glEnd();

  glDisable( GL_BLEND );
}


void GLCroSec::makePlaneC
( const vector2d<double>& data_abs,
  const vector2d<double>& data_arg,
  const Coordinates& scroll )
{
  if( model.gloption.field.crosec.plane.alpha == 1.0 ){
    glDisable( GL_BLEND );
    glDepthMask( true );
  }
  else{
    glEnable( GL_BLEND );
    glBlendFunc( GL_SRC_ALPHA, GL_ONE );
    glDepthMask( false );
  }

  const int Nx = data_abs.sizeX();
  const int Ny = data_abs.sizeY();
  const double dLx = 1.0/Nx;
  const double dLy = 1.0/Ny;

  const Gradation& gradation = model.gloption.field.gradation_arg;

  GLcolor color;  
  glBegin( GL_QUADS );
  for( int ix=0; ix<Nx; ix++ ){
    const double x0 = fmod(ix*dLx+scroll.x,1.0);
    const double x1 = x0+dLx;

    for( int iy=0; iy<Ny; iy++ ){
      const double y0 = fmod(iy*dLy+scroll.y,1.0);
      const double y1 = y0+dLy;

      {
	const double c = data_abs(ix+0,iy+0);
	const double f = data_arg(ix+0,iy+0);
	color   = (gradation(f)*2.0 + GLcolor::white)*c;
	color.A = model.gloption.field.crosec.plane.alpha;
	glColor4dv( color );
	glVertex3d( x0, y0, 0.0 );
      }
      {
	const double c = data_abs((ix+1)%Nx,iy);
	const double f = data_arg((ix+1)%Nx,iy);
	color   = (gradation(f)*2.0 + GLcolor::white)*c;
	color.A = model.gloption.field.crosec.plane.alpha;
	glColor4dv( color );
	glVertex3d( x1, y0, 0.0 );
      }
      {
	const double c = data_abs((ix+1)%Nx,(iy+1)%Ny);
	const double f = data_arg((ix+1)%Nx,(iy+1)%Ny);
	color   = (gradation(f)*2.0 + GLcolor::white)*c;
	color.A = model.gloption.field.crosec.plane.alpha;
	glColor4dv( color );
	glVertex3d( x1, y1, 0.0 );
      }
      {
	const double c = data_abs(ix,(iy+1)%Ny);
	const double f = data_arg(ix,(iy+1)%Ny);
	color   = (gradation(f)*2.0 + GLcolor::white)*c;
	color.A = model.gloption.field.crosec.plane.alpha;
	glColor4dv( color );
	glVertex3d( x0, y1, 0.0 );
      }
    }
  }
  glEnd();

  glDisable( GL_BLEND );
}


void GLCroSec::makeIsolines
( const vector2d<double>& data, const Coordinates& scroll )
{
  glEnable( GL_BLEND );
  glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );

  for( double f=1.0/8; f<1.0; f+=1.0/model.gloption.field.crosec.isoline.lines ){
    glColor4dv( model.gloption.field.gradation(f) );
    GLCroSec::makeIsoline( data, f, scroll );
  }

  glDisable( GL_BLEND );
}


void GLCroSec::makeGradients
( const vector2d<double>& data, const Coordinates& scroll )
{
  glEnable( GL_BLEND );
  glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );

  const int Nx = data.sizeX();
  const int Ny = data.sizeY();
  const double dLx = 1.0/Nx;
  const double dLy = 1.0/Ny;

  glLineWidth( model.gloption.field.crosec.gradient.thick );
  glPointSize( model.gloption.field.crosec.gradient.thick+1.0 );

  glBlendFunc( GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA );

  for( int ix=1; ix+1<Nx; ix++ ){
    const double x = fmod(ix*dLx+scroll.x,1.0);
    for( int iy=1; iy+1<Ny; iy++ ){
      const double y = fmod(iy*dLy+scroll.y,1.0);

      const double gx = data(ix+1,iy) - data(ix-1,iy);
      const double gy = data(ix,iy+1) - data(ix,iy-1);
      const double g2 = hypot(gx,gy);

      glColor4dv( model.gloption.field.gradation(g2) );
      glBegin( GL_LINES );

      glVertex3d( x, y, 0.0 );
      glVertex3d( x+gx*model.gloption.field.crosec.gradient.scale, y+gy*model.gloption.field.crosec.gradient.scale, 0.0 );

      glEnd();
      glBegin( GL_POINTS );
      glVertex3d( x, y, 0.0 );
      glEnd();
    }	    
  }

  glPointSize(1.0);
  glLineWidth(1.0);


  glDisable( GL_BLEND );
}

void GLCroSec::makeIsoline
( const vector2d<double>& data, const double iso_level,
  const Coordinates& scroll )
{
  const double dLx = 1.0/data.sizeX();
  const double dLy = 1.0/data.sizeY();

  vector<double> X, Y;
  X.resize(data.sizeX()+1);
  Y.resize(data.sizeY()+1);
  for( int ix=0; ix<=data.sizeX(); ix++ ){
    X[ix] = dLx*ix;
  }
  for( int iy=0; iy<=data.sizeY(); iy++ ){
    Y[iy] = dLy*iy;
  }

  nodex.resize( data.sizeX(), data.sizeY()+1 );
  nodey.resize( data.sizeX()+1, data.sizeY() );

  // X̊eړ_̈ʒuvZ
  for( int ix=0; ix<data.sizeX(); ix++ ){
    const int px0 = ix;
    const int px1 = ix+1==data.sizeX() ? 0 : ix+1;

    for( int iy=0; iy<=data.sizeY(); iy++ ){
      const int py = iy==data.sizeY() ? 0 : iy;

      const double vo = data(px0,py) - iso_level;
      const double vx = data(px1,py) - iso_level;
      
      if( vo*vx <= 0.0 && (vo<0.0 || vx<0.0) ){
	nodex(ix,iy).vertex = Position( X[ix]+dLx*vo/(vo-vx), Y[iy], 0.0 );
	nodex(ix,iy).flag   = true;
      }
      else{
	nodex(ix,iy).flag   = false;
      }
    }
  }

  // Y̊eړ_̈ʒuvZ
  for( int ix=0; ix<=data.sizeX(); ix++ ){
    const int px = ix==data.sizeX() ? 0 : ix;

    for( int iy=0; iy<data.sizeY(); iy++ ){
      const int py0 = iy;
      const int py1 = iy+1==data.sizeY() ? 0 : iy+1;

      const double vo = data(px,py0) - iso_level;
      const double vy = data(px,py1) - iso_level;

      if( vo*vy <= 0.0 && (vo<0.0 || vy<0.0) ){
	nodey(ix,iy).vertex = Position( X[ix], Y[iy]+dLy*vo/(vo-vy), 0.0 );
	nodey(ix,iy).flag   = true;
      }
      else{
	nodey(ix,iy).flag   = false;
      }
    }
  }

  glBegin( GL_LINES );
  for( int ix=0; ix<data.sizeX(); ix++ ){
    const double dx = fmod(X[ix]+scroll.x,1.0)-X[ix];
    for( int iy=0; iy<data.sizeY(); iy++ ){
      const double dy = fmod(Y[iy]+scroll.y,1.0)-Y[iy];

      makeElement(ix,iy,dx,dy);
    }
  }
  glEnd();
}

int GLCroSec::indexSelected
( const Node& node0, const Node& node1, const Node& node2, const Node& node3 )
{
  if( node1.flag && node2.flag && node3.flag ){
    double d1 = Node::distance( node0, node1 ) + Node::distance( node3, node2 );
    double d2 = Node::distance( node0, node2 ) + Node::distance( node3, node1 );
    if( d1 < d2 ){
      return 1;
    }else{
      return 2;
    }
  }else{
    if( false );
    else if( node1.flag ) return 1;
    else if( node2.flag ) return 2;
    else if( node3.flag ) return 3;
  }
  return 0;
}

void GLCroSec::makeElement( int ix, int iy, double dx, double dy )
{
  static Node node[4];

  node[0] = nodex(ix,iy);
  node[1] = nodey(ix,iy);
  node[2] = nodey(ix+1,iy);
  node[3] = nodex(ix,iy+1);

  static int conn[4][4] = {
    {0,1,2,3},
    {1,3,0,2},
    {2,0,3,1},
    {3,2,1,0},
  };

  if( node[0].flag && node[1].flag && node[2].flag && node[3].flag ){
    int sol = indexSelected(node[0],node[1],node[2],node[3]);

    glVertex2d( node[0].vertex.x+dx, node[0].vertex.y+dy );
    if( sol==1 ){
      glVertex2d( node[1].vertex.x+dx, node[1].vertex.y+dy );
      glVertex2d( node[2].vertex.x+dx, node[2].vertex.y+dy );
    }else{
      glVertex2d( node[2].vertex.x+dx, node[2].vertex.y+dy );
      glVertex2d( node[1].vertex.x+dx, node[1].vertex.y+dy );
    }
    glVertex2d( node[3].vertex.x+dx,node[3].vertex.y+dy );

    return;
  }

  for( int i=0; i<4; i++ ){
    if( !node[conn[i][0]].flag ) continue;

    int sol = indexSelected(node[conn[i][0]],node[conn[i][1]],
			    node[conn[i][2]],node[conn[i][3]]);

    const Position& v0 = node[conn[i][0]].vertex;
    const Position& v1 = node[conn[i][sol]].vertex;
    glVertex2d( v0.x+dx, v0.y+dy );
    glVertex2d( v1.x+dx, v1.y+dy );

    return;
  }
}


double GLCroSec::data_normalized
( const vector3d<double>& data,
  const int ix, const int iy, const int iz )
{
  double min, max;

  if( model.gloption.field.range_manual ){
    min = model.gloption.field.data_min;
    max = model.gloption.field.data_max;
  }
  else{
    min = data.min();
    max = data.max();
  }

  return (data(ix,iy,iz) - min)/(max-min);
}


void GLCroSecA::update( void )
{
  GLBase::beginNewList();

  const Position& La = model.field.La;
  const Position& Lb = model.field.Lb;
  const Position& Lc = model.field.Lc;
  const Position& Lo = model.field.Lo;
  const double  scale = model.gloption.field.crosec.scaleA;
  const Coordinates origin =
    model.gloption.location.scroll(Coordinates(scale,0.0,0.0));
  const Coordinates scroll =
    Coordinates(origin.b,origin.c,origin.a);
  
  glDisable( GL_LIGHTING );
  glDepthMask( false );
  glDepthFunc( GL_LEQUAL );

  glPushMatrix();
  {
    // (0,0,0)-(1,1,0)̐`̈ɊG`ΖړI̒fʂ`ϊs̐ݒ
    double mat[4][4];
    mat[0][0] = Lb.x, mat[0][1] = Lb.y, mat[0][2] = Lb.z, mat[0][3] = 0.0;
    mat[1][0] = Lc.x, mat[1][1] = Lc.y, mat[1][2] = Lc.z, mat[1][3] = 0.0;
    mat[2][0] = La.x, mat[2][1] = La.y, mat[2][2] = La.z, mat[2][3] = 0.0;
    mat[3][0] = Lo.x, mat[3][1] = Lo.y, mat[3][2] = Lo.z, mat[3][3] = 1.0;
    glMultMatrixd((double*)mat);
    glTranslated( 0.0, 0.0, scroll.c );
    glNormal3d( 0.0, 0.0, 1.0 );
  }

  switch( model.field.getFieldType() ){
  case DTField::FIELD_REAL : {
    model.field.nextfield(true);
    do{
      const vector3d<double>& data = model.field.data();

      // 3f[^2fʃf[^̒o,2fʂ̒lKiB
      vector2d<double> data2d;
      data2d.resize( data.sizeY(), data.sizeZ() );

      int ix = int( (data.sizeX()-1)*scale );
      if( ix==data.sizeX()-1 ) ix--;
      const double f2 = (data.sizeX()-1)*scale - (double)ix;
      const double f1 = 1.0-f2;

      for( int iy=0; iy<data.sizeY(); iy++ ){
	for( int iz=0; iz<data.sizeZ(); iz++ ){
	  const double d1 = data_normalized( data, ix+0,iy,iz );
	  const double d2 = data_normalized( data, ix+1,iy,iz );

	  data2d(iy,iz) = d1*f1 + d2*f2;
	}
      }

      // 2fʃf[^̃XvCԂɂ鍂x
      const int NS=2;
      vector2d<double> data2dSP;
      Spline2D::convert( data2dSP, data2d, NS );

      if( true ){  // fʂ̘g̕`
	makeFrame( data2dSP, scroll );
      }

      if( model.gloption.field.crosec.plane.show ){  // fʂ̖ʂ̕`
	makePlaneR( data2dSP, scroll );
      }

      if( model.gloption.field.crosec.isoline.show ){  // fʂ̓l̕`
	makeIsolines( data2d, scroll );
      }

      if( model.gloption.field.crosec.gradient.show ){  // fʂ̌z̕`
	makeGradients( data2d, scroll );
      }

    }while(model.field.nextfield(false));
  } break;

  case DTField::FIELD_COMPLEX : {
    model.field.nextfield(true);
    do{
      const vector3d<double>& data_abs = model.field.data_abs();
      const vector3d<double>& data_arg = model.field.data_arg();

      // 3f[^2fʃf[^̒o,2fʂ̒lKiB
      vector2d<double> data2d_abs;
      vector2d<double> data2d_arg;
      data2d_abs.resize( data_abs.sizeY(), data_abs.sizeZ() );
      data2d_arg.resize( data_arg.sizeY(), data_arg.sizeZ() );

      int ix = int( (data_abs.sizeX()-1)*scale );
      if( ix==data_abs.sizeX()-1 ) ix--;
      const double f2 = (data_abs.sizeX()-1)*scale - (double)ix;
      const double f1 = 1.0-f2;

      for( int iy=0; iy<data_abs.sizeY(); iy++ ){
	for( int iz=0; iz<data_abs.sizeZ(); iz++ ){
	  const double abs1 = data_normalized( data_abs, ix+0,iy,iz );
	  const double abs2 = data_normalized( data_abs, ix+1,iy,iz );

	  data2d_abs(iy,iz) = abs1*f1 + abs2*f2;

	  double arg1 = data_arg(ix+0,iy,iz);
	  double arg2 = data_arg(ix+1,iy,iz);
	  double arg;

	  if( fabs(arg1-arg2) > 0.75 ){
	    if( arg1>arg2 ){
	      arg = fmod( arg1*f1 + (arg2+1.0)*f2, 1.0 );
	    }
	    else{
	      arg = fmod( (arg1+1.0)*f1 + arg2*f2, 1.0 );
	    }
	  }
	  else{
	    arg = arg1*f1 + arg2*f2;
	  }
	  data2d_arg(iy,iz) = arg;
	}
      }

      if( true ){  // fʂ̘g̕`
	makeFrame( data2d_abs, scroll );
      }

      if( model.gloption.field.crosec.plane.show ){  // fʂ̖ʂ̕`
	makePlaneC( data2d_abs, data2d_arg, scroll );
      }

      if( model.gloption.field.crosec.isoline.show ){  // fʂ̓l̕`
	makeIsolines( data2d_abs, scroll );
      }

      if( model.gloption.field.crosec.gradient.show ){  // fʂ̌z̕`
	makeGradients( data2d_abs, scroll );
      }
    }while(model.field.nextfield(false));
  } break;
  }


  glPopMatrix();

  glDepthFunc( GL_LESS );
  glDepthMask( true );
  glEnable( GL_LIGHTING );

  GLBase::endNewList();
}



void GLCroSecB::update( void )
{
  GLBase::beginNewList();

  const Position& La = model.field.La;
  const Position& Lb = model.field.Lb;
  const Position& Lc = model.field.Lc;
  const Position& Lo = model.field.Lo;

  const double  scale = model.gloption.field.crosec.scaleB;
  const Coordinates origin =
    model.gloption.location.scroll(Coordinates(0.0,scale,0.0));
  const Coordinates scroll = Coordinates(origin.c,origin.a,origin.b);
  
  glDisable( GL_LIGHTING );
  glDepthMask( false );
  glDepthFunc( GL_LEQUAL );

  glPushMatrix();
  {
    // (0,0,0)-(1,1,0)̐`̈ɊG`ΖړI̒fʂ`ϊs̐ݒ
    double mat[4][4];
    mat[0][0] = Lc.x, mat[0][1] = Lc.y, mat[0][2] = Lc.z, mat[0][3] = 0.0;
    mat[1][0] = La.x, mat[1][1] = La.y, mat[1][2] = La.z, mat[1][3] = 0.0;
    mat[2][0] = Lb.x, mat[2][1] = Lb.y, mat[2][2] = Lb.z, mat[2][3] = 0.0;
    mat[3][0] = Lo.x, mat[3][1] = Lo.y, mat[3][2] = Lo.z, mat[3][3] = 1.0;
    glMultMatrixd((double*)mat);
    glTranslated( 0.0, 0.0, scroll.c );
    glNormal3d( 0.0, 0.0, 1.0 );
  }

  switch( model.field.getFieldType() ){
  case DTField::FIELD_REAL : {
    model.field.nextfield(true);
    do{
      const vector3d<double>& data = model.field.data();

      // 3f[^2fʃf[^̒o,2fʂ̒lKiB
      vector2d<double> data2d;
      data2d.resize( data.sizeZ(), data.sizeX() );

      int iy = int( (data.sizeY()-1)*scale );
      if( iy==data.sizeY()-1 ) iy--;
      const double f2 = (data.sizeY()-1)*scale - (double)iy;
      const double f1 = 1.0-f2;

      for( int iz=0; iz<data.sizeZ(); iz++ ){
	for( int ix=0; ix<data.sizeX(); ix++ ){
	  const double d1 =
	    data_normalized( data, ix,iy+0,iz );
	  const double d2 =
	    data_normalized( data, ix,iy+1,iz );

	  data2d(iz,ix) = d1*f1 + d2*f2;
	}
      }

      // 2fʃf[^̃XvCԂɂ鍂x
      const int NS=2;
      vector2d<double> data2dSP;
      Spline2D::convert( data2dSP, data2d, NS );

      if( true ){  // fʂ̘g̕`
	makeFrame( data2dSP, scroll );
      }

      if( model.gloption.field.crosec.plane.show ){  // fʂ̖ʂ̕`
	makePlaneR( data2dSP, scroll );
      }

      if( model.gloption.field.crosec.isoline.show ){  // fʂ̓l̕`
	makeIsolines( data2d, scroll );
      }

      if( model.gloption.field.crosec.gradient.show ){  // fʂ̌z̕`
	makeGradients( data2d, scroll );
      }
    }while(model.field.nextfield(false));
  } break;

  case DTField::FIELD_COMPLEX : {
    model.field.nextfield(true);
    do{
      const vector3d<double>& data_abs = model.field.data_abs();
      const vector3d<double>& data_arg = model.field.data_arg();

      // 3f[^2fʃf[^̒o,2fʂ̒lKiB
      vector2d<double> data2d_abs;
      vector2d<double> data2d_arg;
      data2d_abs.resize( data_abs.sizeZ(), data_abs.sizeX() );
      data2d_arg.resize( data_arg.sizeZ(), data_arg.sizeX() );

      int iy = int( (data_abs.sizeY()-1)*scale );
      if( iy==data_abs.sizeY()-1 ) iy--;
      const double f2 = (data_abs.sizeY()-1)*scale - (double)iy;
      const double f1 = 1.0-f2;

      for( int iz=0; iz<data_abs.sizeZ(); iz++ ){
	for( int ix=0; ix<data_abs.sizeX(); ix++ ){
	  const double abs1 = data_normalized( data_abs, ix,iy+0,iz );
	  const double abs2 = data_normalized( data_abs, ix,iy+1,iz );

	  data2d_abs(iz,ix) = abs1*f1 + abs2*f2;

	  double arg1 = data_arg(ix,iy+0,iz);
	  double arg2 = data_arg(ix,iy+1,iz);
	  double arg;

	  if( fabs(arg1-arg2) > 0.75 ){
	    if( arg1>arg2 ){
	      arg = fmod( arg1*f1 + (arg2+1.0)*f2, 1.0 );
	    }
	    else{
	      arg = fmod( (arg1+1.0)*f1 + arg2*f2, 1.0 );
	    }
	  }
	  else{
	    arg = arg1*f1 + arg2*f2;
	  }
	  data2d_arg(iz,ix) = arg;
	}
      }

      if( true ){  // fʂ̘g̕`
	makeFrame( data2d_abs, scroll );
      }

      if( model.gloption.field.crosec.plane.show ){  // fʂ̖ʂ̕`
	makePlaneC( data2d_abs, data2d_arg, scroll );
      }

      if( model.gloption.field.crosec.isoline.show ){  // fʂ̓l̕`
	makeIsolines( data2d_abs, scroll );
      }

      if( model.gloption.field.crosec.gradient.show ){  // fʂ̌z̕`
	makeGradients( data2d_abs, scroll );
      }
    }while(model.field.nextfield(false));
  } break;
  }


  glPopMatrix();

  glDepthFunc( GL_LESS );
  glDepthMask( true );
  glEnable( GL_LIGHTING );
  
  GLBase::endNewList();
}


void GLCroSecC::update( void )
{
  GLBase::beginNewList();

  const Position& La = model.field.La;
  const Position& Lb = model.field.Lb;
  const Position& Lc = model.field.Lc;
  const Position& Lo = model.field.Lo;
  const double  scale = model.gloption.field.crosec.scaleC;
  const Coordinates origin =
    model.gloption.location.scroll(Coordinates(0.0,0.0,scale));
  const Coordinates scroll =
    Coordinates(origin.a,origin.b,origin.c);
  
  glDisable( GL_LIGHTING );
  glDepthMask( false );
  glDepthFunc( GL_LEQUAL );

  glPushMatrix();
  {
    // (0,0,0)-(1,1,0)̐`̈ɊG`ΖړI̒fʂ`ϊs̐ݒ
    double mat[4][4];
    mat[0][0] = La.x, mat[0][1] = La.y, mat[0][2] = La.z, mat[0][3] = 0.0;
    mat[1][0] = Lb.x, mat[1][1] = Lb.y, mat[1][2] = Lb.z, mat[1][3] = 0.0;
    mat[2][0] = Lc.x, mat[2][1] = Lc.y, mat[2][2] = Lc.z, mat[2][3] = 0.0;
    mat[3][0] = Lo.x, mat[3][1] = Lo.y, mat[3][2] = Lo.z, mat[3][3] = 1.0;
    glMultMatrixd((double*)mat);
    glTranslated( 0.0, 0.0, scroll.c );
    glNormal3d( 0.0, 0.0, 1.0 );
  }

  switch( model.field.getFieldType() ){
  case DTField::FIELD_REAL : {
    model.field.nextfield(true);
    do{
      const vector3d<double>& data = model.field.data();

      // 3f[^2fʃf[^̒o,2fʂ̒lKiB
      vector2d<double> data2d;
      data2d.resize( data.sizeX(), data.sizeY() );

      int iz = int( (data.sizeZ()-1)*scale );
      if( iz==data.sizeZ()-1 ) iz--;
      const double f2 = (data.sizeZ()-1)*scale - (double)iz;
      const double f1 = 1.0-f2;

      for( int ix=0; ix<data.sizeX(); ix++ ){
	for( int iy=0; iy<data.sizeY(); iy++ ){
	  const double d1 = data_normalized( data, ix,iy,iz+0 );
	  const double d2 = data_normalized( data, ix,iy,iz+1 );
	  data2d(ix,iy) = d1*f1 + d2*f2;
	}
      }

      // 2fʃf[^̃XvCԂɂ鍂x
      const int NS=2;
      vector2d<double> data2dSP;
      Spline2D::convert( data2dSP, data2d, NS );

      if( true ){  // fʂ̘g̕`
	makeFrame( data2dSP, scroll );
      }

      if( model.gloption.field.crosec.plane.show ){  // fʂ̖ʂ̕`
	makePlaneR( data2dSP, scroll );
      }

      if( model.gloption.field.crosec.isoline.show ){  // fʂ̓l̕`
	makeIsolines( data2d, scroll );
      }

      if( model.gloption.field.crosec.gradient.show ){  // fʂ̌z̕`
	makeGradients( data2d, scroll );
      }
    }while(model.field.nextfield(false));
  } break;

  case DTField::FIELD_COMPLEX : {
    model.field.nextfield(true);
    do{
      const vector3d<double>& data_abs = model.field.data_abs();
      const vector3d<double>& data_arg = model.field.data_arg();

      // 3f[^2fʃf[^̒o,2fʂ̒lKiB
      vector2d<double> data2d_abs;
      vector2d<double> data2d_arg;
      data2d_abs.resize( data_abs.sizeX(), data_abs.sizeY() );
      data2d_arg.resize( data_arg.sizeX(), data_arg.sizeY() );

      int iz = int( (data_abs.sizeZ()-1)*scale );
      if( iz==data_abs.sizeZ()-1 ) iz--;
      const double f2 = (data_abs.sizeZ()-1)*scale - (double)iz;
      const double f1 = 1.0-f2;

      for( int ix=0; ix<data_abs.sizeX(); ix++ ){
	for( int iy=0; iy<data_abs.sizeY(); iy++ ){
	  const double abs1 = data_normalized( data_abs, ix,iy,iz+0 );
	  const double abs2 = data_normalized( data_abs, ix,iy,iz+1 );

	  data2d_abs(ix,iy) = abs1*f1 + abs2*f2;

	  double arg1 = data_arg(ix,iy,iz+0);
	  double arg2 = data_arg(ix,iy,iz+1);
	  double arg;

	  if( fabs(arg1-arg2) > 0.75 ){
	    if( arg1>arg2 ){
	      arg = fmod( arg1*f1 + (arg2+1.0)*f2, 1.0 );
	    }
	    else{
	      arg = fmod( (arg1+1.0)*f1 + arg2*f2, 1.0 );
	    }
	  }
	  else{
	    arg = arg1*f1 + arg2*f2;
	  }
	  data2d_arg(ix,iy) = arg;
	}
      }

      if( true ){  // fʂ̘g̕`
	makeFrame( data2d_abs, scroll );
      }

      if( model.gloption.field.crosec.plane.show ){  // fʂ̖ʂ̕`
	makePlaneC( data2d_abs, data2d_arg, scroll );
      }

      if( model.gloption.field.crosec.isoline.show ){  // fʂ̓l̕`
	makeIsolines( data2d_abs, scroll );
      }

      if( model.gloption.field.crosec.gradient.show ){  // fʂ̌z̕`
	makeGradients( data2d_abs, scroll );
      }
    }while(model.field.nextfield(false));
  } break;
  }

  glPopMatrix();

  glDepthFunc( GL_LESS );
  glDepthMask( true );
  glEnable( GL_LIGHTING );
  
  GLBase::endNewList();
}
