/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

/*!
 \file gleps.cc
 \brief }`EPS`ł̃t@CoNX
*/

#include <stdio.h>
#include <GL/gl.h>
#include "gleps.h"

bool GLEPS::open
( const char* fname,
  const int win_w, const int win_h, const double eps_w, const double eps_h )
{
  const double DOT_PER_INCH = 72.0;
  const double INCH_TO_MM = 25.4;
  const double MM_TO_PT = DOT_PER_INCH/INCH_TO_MM;

  X_TO_PT = eps_w*MM_TO_PT/win_w;
  Y_TO_PT = eps_h*MM_TO_PT/win_h;

  fptr = fopen(fname,"wb");
  if( fptr == NULL ) return false;

  fprintf( fptr, "%%!PS-Adobe-2.0 EPSF-2.0\n"
	  "%%%%BoundingBox: 0.0 0.0 %f %f\n",
	   eps_w*MM_TO_PT, eps_h*MM_TO_PT );
  fprintf( fptr, "/Dict 200 dict def\n"
	  "Dict begin\n"
	  "Dict /mtrx matrix put\n"
	  "/sc   {scale} bind def\n"
	  "/gr   {grestore} bind def\n"
	  "/gs   {gsave} bind def\n"
	  "/srgb {setrgbcolor} bind def\n"
	  "/sd   {setdash} bind def\n"
	  "/slw  {setlinewidth} bind def\n"
	  "/slc  {setlinecap} bind def\n"
	  "/slj  {setlinejoin} bind def\n"
	  "/ff   {findfont} bind def\n"
	  "/sf   {setfont} bind def\n"
	  "/scf  {scalefont} bind def\n"
	  "/n    {newpath} bind def\n"
	  "/m    {moveto} bind def\n"
	  "/l    {lineto} bind def\n"
	  "/p    {moveto 0 currentlinewidth rlineto stroke} bind def\n"
	  "/cp   {closepath} bind def\n"
	  "/s    {stroke} bind def\n"
	  "/ef   {eofill} bind def\n"
	  "/sh   {show} bind def\n" );

  return true;
}

void GLEPS::close( void )
{
  fprintf( fptr, "end\n" );
  fclose(fptr);
  fptr = NULL;
}

void GLEPS::color3d( const double R, const double G, const double B )
{
  fprintf( fptr, "%5.3f %5.3f %5.3f srgb\n", R, G, B );
}
void GLEPS::color4dv( const double color[4] )
{
  fprintf( fptr, "%5.3f %5.3f %5.3f srgb\n", color[0], color[1], color[2] );
}

void GLEPS::begin( const int mode )
{
  this->mode  = mode;
  this->first = true;
  fprintf( fptr, "n" );
}

void GLEPS::end( void )
{
  switch( this->mode ){
  case GL_LINE_LOOP  : {
    fprintf( fptr, " cp");
  } break;
  }
  fprintf( fptr, " s\n");
}

void GLEPS::loadIdentity( void )
{
  sx = sy = 1.0;
  tx = ty = 0.0;
}
void GLEPS::translated( const double tx, const double ty, const double tz )
{
  this->tx += this->sx * tx;
  this->ty += this->sy * ty;
}

void GLEPS::translateR( const double tx, const double ty, const double tz )
{
  this->tx += tx;
  this->ty += ty;
}

void GLEPS::scaled( const double sx, const double sy, const double sz )
{
  this->sx *= sx;
  this->sy *= sy;
}

void GLEPS::pushMatrix( void )
{
  stack_tx.push(tx);
  stack_ty.push(ty);
  stack_sx.push(sx);
  stack_sy.push(sy);
}

void GLEPS::popMatrix( void )
{
  tx = stack_tx.top();
  ty = stack_ty.top();
  sx = stack_sx.top();
  sy = stack_sy.top();

  stack_tx.pop();
  stack_ty.pop();
  stack_sx.pop();
  stack_sy.pop();
}

void GLEPS::vertex2d( const double x, const double y )
{
  switch( this->mode ){
  case GL_POINTS : {
    fprintf( fptr, " %f %f p", X(x), Y(y)-lw*0.5 );
  } break;

  case GL_LINE_LOOP  :
  case GL_LINE_STRIP : {
    if( this->first ){
      fprintf( fptr, " %f %f m", X(x), Y(y) );
      this->first = false;
    }
    else{
      fprintf( fptr, " %f %f l", X(x), Y(y) );
    }
  } break;
  case GL_LINES : {
    if( this->first ){
      fprintf( fptr, " %f %f m", X(x), Y(y) );
      this->first = false;
    }
    else{
      fprintf( fptr, " %f %f l", X(x), Y(y) );
      this->first = true;
    }
  } break;
  }
}

void GLEPS::pointSize( double width )
{
  lw = width*X_TO_PT;
  fprintf( fptr, " %f slw\n", lw );
}
void GLEPS::lineWidth( double width )
{
  lw = width*X_TO_PT;
  fprintf( fptr, " %f slw\n", lw );
}

void GLEPS::linestipple( double on, double off )
{
  fprintf( fptr, "[%f %f] 0 sd\n", on*X_TO_PT, off*X_TO_PT );
}

void GLEPS::font( const char* font, const double size )
{
  fprintf( fptr, "%s ff %f scf sf\n", font, size*Y_TO_PT );
}

void GLEPS::renderText( const double x, const double y, const double z, const QString& str )
{
  fprintf( fptr, "n %f %f m (%s) gs sh gr\n", X(x), Y(y), qPrintable(str));
}

