/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

/*!
 \file glfog.cc
 \brief OtB[h̃tHOGL\̃NX
*/

#include <GL/gl.h>
#include "dtmodel.h"
#include "glmisc.h"
#include "glfog.h"

double GLFog::data_normalized
( const vector3d<double>& data,
  const int ix, const int iy, const int iz )
{
  double min, max;

  if( model.gloption.field.range_manual ){
    min = model.gloption.field.data_min;
    max = model.gloption.field.data_max;
  }
  else{
    min = data.min();
    max = data.max();
  }

  return (data(ix,iy,iz) - min)/(max-min);
}

void GLFog::update( void )
{
  GLBase::beginNewList();

  glDisable( GL_LIGHTING );
  glEnable( GL_BLEND );
  glBlendFunc( GL_SRC_ALPHA, GL_ONE );
  glDepthMask( false );

  glPointSize(1.0);

  const Position& La = model.field.La;
  const Position& Lb = model.field.Lb;
  const Position& Lc = model.field.Lc;
  const Position& Lo = model.field.Lo;

  glPushMatrix(); {
    double mat[4][4];
  
    mat[0][0] = La.x, mat[0][1] = La.y, mat[0][2] = La.z, mat[0][3] = 0.0;
    mat[1][0] = Lb.x, mat[1][1] = Lb.y, mat[1][2] = Lb.z, mat[1][3] = 0.0;
    mat[2][0] = Lc.x, mat[2][1] = Lc.y, mat[2][2] = Lc.z, mat[2][3] = 0.0;
    mat[3][0] = Lo.x, mat[3][1] = Lo.y, mat[3][2] = Lo.z, mat[3][3] = 1.0;
    glMultMatrixd((double*)mat);
  }

  const double scale = model.gloption.field.fog.scale;

  switch( model.field.getFieldType() ){
  case DTField::FIELD_REAL : {
    model.field.nextfield(true);
    do{
      const vector3d<double>& data = model.field.data();
      const int Na = data.sizeX();
      const int Nb = data.sizeY();
      const int Nc = data.sizeZ();
      const double dA = 1.0/Na;
      const double dB = 1.0/Nb;
      const double dC = 1.0/Nc;
    
      glBegin( GL_POINTS );  
      for( int ia=0; ia<Na; ia++ ){
	for( int ib=0; ib<Nb; ib++ ){
	  for( int ic=0; ic<Nc; ic++ ){
	    const Coordinates scroll =
	      model.gloption.location.scroll(Coordinates(ia*dA,ib*dB,ic*dC));

	    double d = data_normalized( data, ia,ib,ic );
	    if( d < 0.0 ) d = 0.0;
	    if( d > 1.0 ) d = 1.0;

	    GLcolor color = model.gloption.field.gradation(d);
	    color.A = model.gloption.field.fog.alpha;
	    glColor4dv(color);

	    //	    const int n = int( 16 * scale * d );
	    const int n = int( 4*16 * scale * d );

	    for( int i=0; i<n; i++ ){
	      glVertex3d( scroll.a+dA*urand(), scroll.b+dB*urand(), scroll.c+dC*urand());
	    }
	  }
	}
      }
      glEnd();
    }while(model.field.nextfield(false));
  } break;

  case DTField::FIELD_COMPLEX : {
    model.field.nextfield(true);
    do{
      const vector3d<double>& data_abs = model.field.data_abs();
      const vector3d<double>& data_arg = model.field.data_arg();

      const int Na = data_abs.sizeX();
      const int Nb = data_abs.sizeY();
      const int Nc = data_abs.sizeZ();
      const double dA = 1.0/Na;
      const double dB = 1.0/Nb;
      const double dC = 1.0/Nc;

      const Gradation& gradation = model.gloption.field.gradation_arg;
      GLcolor color;  
    
      glBegin( GL_POINTS );  
      for( int ia=0; ia<Na; ia++ ){
	for( int ib=0; ib<Nb; ib++ ){
	  for( int ic=0; ic<Nc; ic++ ){
	    const Coordinates scroll =
	      model.gloption.location.scroll(Coordinates(ia*dA,ib*dB,ic*dC));

	    double d = data_normalized( data_abs, ia,ib,ic );
	    if( d < 0.0 ) d = 0.0;
	    if( d > 1.0 ) d = 1.0;

	    const double f = data_arg(ia,ib,ic);
	    color   = (gradation(f)*2.0 + GLcolor::white)*d;
	    color.A = model.gloption.field.fog.alpha;
	    glColor4dv( color );

	    const int n = int( 16 * scale * d );

	    for( int i=0; i<n; i++ ){
	      glVertex3d( scroll.a+dA*urand(), scroll.b+dB*urand(), scroll.c+dC*urand());
	    }
	  }
	}
      }
      glEnd();
    }while(model.field.nextfield(false));
  } break;
  }

  glPopMatrix();
  
  glDepthMask( true );
  glDisable( GL_BLEND );
  glEnable( GL_LIGHTING );

  GLBase::endNewList();
}
