/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

/*!
  \file glisosurf.cc
  \brief OtB[h̓lʂGL\̃NX
*/

#include <GL/gl.h>
#include "dtmodel.h"
#include "glmisc.h"
#include "glisosurf.h"

vector3d<GLIsosurf::Node> GLIsosurf::nodex;
vector3d<GLIsosurf::Node> GLIsosurf::nodey;
vector3d<GLIsosurf::Node> GLIsosurf::nodez;

double phase_mean( double f1, double f2, double t )
{
  if( fabs(f1-f2) > 0.75 ){
    if( f1>f2 ){
      return fmod( f1*(1.0-t) + (f2+1.0)*(t), 1.0 );
    }
    else{
      return fmod( (f1+1.0)*(1.0-t) + f2*(t), 1.0 );
    }
  }
  else{
    return f1*(1.0-t) + f2*(t);
  }
  return 0.0;
}

void GLIsosurf::update( void )
{
  GLBase::beginNewList();

  const Position& La = model.field.La;
  const Position& Lb = model.field.Lb;
  const Position& Lc = model.field.Lc;
  const Position& Lo = model.field.Lo;
  const Coordinates origin =
    model.gloption.location.scroll(Coordinates(0.0,0.0,0.0));

  glEnable(GL_LIGHTING);

  if( model.gloption.field.isosurf.alpha == 1.0 ){
    glDisable( GL_BLEND );
    glDepthMask( true );
  }
  else{
    glEnable( GL_BLEND );
    glBlendFunc( GL_SRC_ALPHA, GL_ONE );
    glDepthMask( false );
  }

  glPushMatrix(); {
    double mat[4][4];
  
    mat[0][0] = La.x, mat[0][1] = La.y, mat[0][2] = La.z, mat[0][3] = 0.0;
    mat[1][0] = Lb.x, mat[1][1] = Lb.y, mat[1][2] = Lb.z, mat[1][3] = 0.0;
    mat[2][0] = Lc.x, mat[2][1] = Lc.y, mat[2][2] = Lc.z, mat[2][3] = 0.0;
    mat[3][0] = Lo.x, mat[3][1] = Lo.y, mat[3][2] = Lo.z, mat[3][3] = 1.0;
    glMultMatrixd((double*)mat);
  }

  switch( model.field.getFieldType() ){
  case DTField::FIELD_REAL : {
    const vector<double>& vvalue =
      model.gloption.field.isosurf.vvalue;

    model.field.nextfield(true);
    do{
      const vector3d<double>& data = model.field.data();
      const double min = model.field.data_min();
      const double max = model.field.data_max();

      for( int i=0; i<(int)vvalue.size(); i++ ){
	double scale = 0.0;

	switch( model.field.multi.mode ){
	case 0 : {
	  scale = (vvalue[i]-min)/(max-min);
	} break;
	case 1 : {
	  scale = (double)model.field.idata/(model.field.vdata.size()-1);
	} break;
	case 2 : {
	  scale = (vvalue[i]-min)/(max-min);
	} break;
	}

	makeIsosurfR( data, vvalue[i], scale, origin );
      }
    }while(model.field.nextfield(false));
  } break;

  case DTField::FIELD_COMPLEX : {
    const vector<double>& vvalue =
      model.gloption.field.isosurf.vvalue;

    model.field.nextfield(true);
    do{
      const vector3d<double>& data_abs = model.field.data_abs();
      const vector3d<double>& data_arg = model.field.data_arg();
      const double min = model.field.data_min();
      const double max = model.field.data_max();

      for( int i=0; i<(int)vvalue.size(); i++ ){
	double scale = 0.0;
	switch( model.field.multi.mode ){
	case 0 : {
	  scale = (vvalue[i]-min)/(max-min);
	} break;
	case 1 : {
	  scale = (double)model.field.idata/(model.field.vdata.size()-1);
	} break;
	case 2 : {
	  scale = (vvalue[i]-min)/(max-min);
	} break;
	}

	makeIsosurfC( data_abs, data_arg, vvalue[i], scale, origin );
      }
    }while(model.field.nextfield(false));
  } break;
  }

  glPopMatrix();

  //  glDisable( GL_CULL_FACE );
  glDepthMask( true );
  glDisable( GL_BLEND );

  GLBase::endNewList();
}



void GLIsosurf::makeIsosurfR
( const vector3d<double>& data,
  const double iso_value,
  const double iso_scale,
  const Coordinates& scroll )
{
  icomplex_field = false;
  value = iso_value;
  scale = iso_scale;

  const double dLx = 1.0/data.sizeX();
  const double dLy = 1.0/data.sizeY();
  const double dLz = 1.0/data.sizeZ();

  vector<double> X, Y, Z;
  X.resize(data.sizeX()+1);
  Y.resize(data.sizeY()+1);
  Z.resize(data.sizeZ()+1);
  for( int ix=0; ix<=data.sizeX(); ix++ ){
    X[ix] = dLx*ix;
  }
  for( int iy=0; iy<=data.sizeY(); iy++ ){
    Y[iy] = dLy*iy;
  }
  for( int iz=0; iz<=data.sizeZ(); iz++ ){
    Z[iz] = dLz*iz;
  }

  nodex.resize( data.sizeX(), data.sizeY()+1, data.sizeZ()+1 );
  nodey.resize( data.sizeX()+1, data.sizeY(), data.sizeZ()+1 );
  nodez.resize( data.sizeX()+1, data.sizeY()+1, data.sizeZ() );

  // X̊eړ_̈ʒuvZ
  for( int ix=0; ix<data.sizeX(); ix++ ){
    const int px0 = ix;
    const int px1 = ix+1==data.sizeX() ? 0 : ix+1;

    for( int iy=0; iy<=data.sizeY(); iy++ ){
      const int py = iy==data.sizeY() ? 0 : iy;

      for( int iz=0; iz<=data.sizeZ(); iz++ ){
	const int pz = iz==data.sizeZ() ? 0 : iz;

	const double vo = data(px0,py,pz) - iso_value;
	const double vx = data(px1,py,pz) - iso_value;

	if( vo*vx <= 0.0 && (vo<0.0 || vx<0.0) ){
	  nodex(ix,iy,iz).vertex = Position( X[ix]+dLx*vo/(vo-vx), Y[iy], Z[iz] );
	  nodex(ix,iy,iz).normal = calcNormal(data,px0,py,pz);
	  nodex(ix,iy,iz).flag   = true;
	}else{
	  nodex(ix,iy,iz).flag   = false;
	}
      }
    }
  }

  // Y̊eړ_̈ʒuvZ
  for( int ix=0; ix<=data.sizeX(); ix++ ){
    const int px = ix==data.sizeX() ? 0 : ix;

    for( int iy=0; iy<data.sizeY(); iy++ ){
      const int py0 = iy;
      const int py1 = iy+1==data.sizeY() ? 0 : iy+1;

      for( int iz=0; iz<=data.sizeZ(); iz++ ){
	const int pz = iz==data.sizeZ() ? 0 : iz;

	const double vo = data(px,py0,pz) - iso_value;
	const double vy = data(px,py1,pz) - iso_value;

	if( vo*vy <= 0.0 && (vo<0.0 || vy<0.0) ){
	  nodey(ix,iy,iz).vertex = Position( X[ix], Y[iy]+dLy*vo/(vo-vy), Z[iz] );
	  nodey(ix,iy,iz).normal = calcNormal(data,px,py0,pz);
	  nodey(ix,iy,iz).flag   = true;
	}else{
	  nodey(ix,iy,iz).flag   = false;
	}
      }
    }
  }

  // Z̊eړ_̈ʒuvZ
  for( int ix=0; ix<=data.sizeX(); ix++ ){
    const int px = ix==data.sizeX() ? 0 : ix;

    for( int iy=0; iy<=data.sizeY(); iy++ ){
      const int py = iy==data.sizeY() ? 0 : iy;

      for( int iz=0; iz<data.sizeZ(); iz++ ){
	const int pz0 = iz;
	const int pz1 = iz+1==data.sizeZ() ? 0 : iz+1;

	const double vo = data(px,py,pz0) - iso_value;
	const double vz = data(px,py,pz1) - iso_value;

	if( vo*vz <= 0.0 && (vo<0.0 || vz<0.0) ){
	  nodez(ix,iy,iz).vertex = Position( X[ix], Y[iy], Z[iz]+dLz*vo/(vo-vz) );
	  nodez(ix,iy,iz).normal = calcNormal(data,px,py,pz0);
	  nodez(ix,iy,iz).flag   = true;
	}else{
	  nodez(ix,iy,iz).flag   = false;
	}
      }
    }
  }

  GLcolor color = model.gloption.field.gradation(iso_scale);
  color.A = model.gloption.field.isosurf.alpha;
  glColor4dv(color);
  glMaterialdv( GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, color );

  glBegin( GL_TRIANGLES );
  for( int ix=0; ix<data.sizeX(); ix++ ){
    const double dx = fmod(X[ix]+scroll.x,1.0) - X[ix];
    for( int iy=0; iy<data.sizeY(); iy++ ){
      const double dy = fmod(Y[iy]+scroll.y,1.0) - Y[iy];
      for( int iz=0; iz<data.sizeZ(); iz++ ){
	const double dz = fmod(Z[iz]+scroll.z,1.0) - Z[iz];

	makeElement(ix,iy,iz,Position(dx,dy,dz));
      }
    }
  }
  glEnd();
}


void GLIsosurf::makeIsosurfC
( const vector3d<double>& data_abs,
  const vector3d<double>& data_arg,
  const double iso_value,
  const double iso_scale,
  const Coordinates& scroll )
{
  icomplex_field = true;

  const double dLx = 1.0/data_abs.sizeX();
  const double dLy = 1.0/data_abs.sizeY();
  const double dLz = 1.0/data_abs.sizeZ();

  vector<double> X, Y, Z;
  X.resize(data_abs.sizeX()+1);
  Y.resize(data_abs.sizeY()+1);
  Z.resize(data_abs.sizeZ()+1);
  for( int ix=0; ix<=data_abs.sizeX(); ix++ ){
    X[ix] = dLx*ix;
  }
  for( int iy=0; iy<=data_abs.sizeY(); iy++ ){
    Y[iy] = dLy*iy;
  }
  for( int iz=0; iz<=data_abs.sizeZ(); iz++ ){
    Z[iz] = dLz*iz;
  }

  nodex.resize( data_abs.sizeX(), data_abs.sizeY()+1, data_abs.sizeZ()+1 );
  nodey.resize( data_abs.sizeX()+1, data_abs.sizeY(), data_abs.sizeZ()+1 );
  nodez.resize( data_abs.sizeX()+1, data_abs.sizeY()+1, data_abs.sizeZ() );

  // X̊eړ_̈ʒuvZ
  for( int ix=0; ix<data_abs.sizeX(); ix++ ){
    const int px0 = ix;
    const int px1 = ix+1==data_abs.sizeX() ? 0 : ix+1;

    for( int iy=0; iy<=data_abs.sizeY(); iy++ ){
      const int py = iy==data_abs.sizeY() ? 0 : iy;

      for( int iz=0; iz<=data_abs.sizeZ(); iz++ ){
	const int pz = iz==data_abs.sizeZ() ? 0 : iz;

	const double vo = data_abs(px0,py,pz) - iso_value;
	const double vx = data_abs(px1,py,pz) - iso_value;

	if( vo*vx <= 0.0 && (vo<0.0 || vx<0.0) ){
	  const double t = vo/(vo-vx);
	  nodex(ix,iy,iz).flag   = true;
	  nodex(ix,iy,iz).phase  = 
	    phase_mean( data_arg(px0,py,pz), data_arg(px1,py,pz),  t );
	  nodex(ix,iy,iz).vertex = Position( X[ix]+dLx*t, Y[iy], Z[iz] );
	  nodex(ix,iy,iz).normal = calcNormal(data_abs,px0,py,pz);
	}else{
	  nodex(ix,iy,iz).flag   = false;
	}
      }
    }
  }

  // Y̊eړ_̈ʒuvZ
  for( int ix=0; ix<=data_abs.sizeX(); ix++ ){
    const int px = ix==data_abs.sizeX() ? 0 : ix;

    for( int iy=0; iy<data_abs.sizeY(); iy++ ){
      const int py0 = iy;
      const int py1 = iy+1==data_abs.sizeY() ? 0 : iy+1;

      for( int iz=0; iz<=data_abs.sizeZ(); iz++ ){
	const int pz = iz==data_abs.sizeZ() ? 0 : iz;

	const double vo = data_abs(px,py0,pz) - iso_value;
	const double vy = data_abs(px,py1,pz) - iso_value;

	if( vo*vy <= 0.0 && (vo<0.0 || vy<0.0) ){
	  const double t = vo/(vo-vy);
	  nodey(ix,iy,iz).flag   = true;
	  nodey(ix,iy,iz).phase  = 
	    phase_mean( data_arg(px,py0,pz), data_arg(px,py1,pz),  t );
	  nodey(ix,iy,iz).vertex = Position( X[ix], Y[iy]+dLy*t, Z[iz] );
	  nodey(ix,iy,iz).normal = calcNormal(data_abs,px,py0,pz);
	}else{
	  nodey(ix,iy,iz).flag   = false;
	}
      }
    }
  }

  // Z̊eړ_̈ʒuvZ
  for( int ix=0; ix<=data_abs.sizeX(); ix++ ){
    const int px = ix==data_abs.sizeX() ? 0 : ix;

    for( int iy=0; iy<=data_abs.sizeY(); iy++ ){
      const int py = iy==data_abs.sizeY() ? 0 : iy;

      for( int iz=0; iz<data_abs.sizeZ(); iz++ ){
	const int pz0 = iz;
	const int pz1 = iz+1==data_abs.sizeZ() ? 0 : iz+1;

	const double vo = data_abs(px,py,pz0) - iso_value;
	const double vz = data_abs(px,py,pz1) - iso_value;

	if( vo*vz <= 0.0 && (vo<0.0 || vz<0.0) ){
	  const double t = vo/(vo-vz);
	  nodez(ix,iy,iz).flag   = true;
	  nodez(ix,iy,iz).phase  =
	    phase_mean( data_arg(px,py,pz0), data_arg(px,py,pz1),  t );
	  nodez(ix,iy,iz).vertex = Position( X[ix], Y[iy], Z[iz]+dLz*t );
	  nodez(ix,iy,iz).normal = calcNormal(data_abs,px,py,pz0);
	}else{
	  nodez(ix,iy,iz).flag   = false;
	}
      }
    }
  }

  glBegin( GL_TRIANGLES );
  for( int ix=0; ix<data_abs.sizeX(); ix++ ){
    const double dx = fmod(X[ix]+scroll.x,1.0) - X[ix];
    for( int iy=0; iy<data_abs.sizeY(); iy++ ){
      const double dy = fmod(Y[iy]+scroll.y,1.0) - Y[iy];
      for( int iz=0; iz<data_abs.sizeZ(); iz++ ){
	const double dz = fmod(Z[iz]+scroll.z,1.0) - Z[iz];

	makeElement(ix,iy,iz,Position(dx,dy,dz));
      }
    }
  }
  glEnd();
}


// w4_̌ړ_K؂Ȑڑ@ԓ郁\bh
int GLIsosurf::indexSelected
( const Node& node0, const Node& node1,
  const Node& node2, const Node& node3 )
{
  if( node1.flag && node2.flag && node3.flag ){
    double d1 = Node::distance( node0, node1 ) + Node::distance( node3, node2 );
    double d2 = Node::distance( node0, node2 ) + Node::distance( node3, node1 );
    if( d1 < d2 ){
      return 1;
    }else{
      return 2;
    }
  }else{
    if( false );
    else if( node1.flag ) return 1;
    else if( node2.flag ) return 2;
    else if( node3.flag ) return 3;
  }
  return 0;
}

// @vZ郁\bh
Position GLIsosurf::calcNormal
( const vector3d<double>& data, int ix, int iy, int iz )
{
  const int ixn = ix+1 == data.sizeX() ? 0    : ix+1;
  const int ixp = ix   == 0  ? data.sizeX()-1 : ix-1;
  const int iyn = iy+1 == data.sizeY() ? 0    : iy+1;
  const int iyp = iy   == 0  ? data.sizeY()-1 : iy-1;
  const int izn = iz+1 == data.sizeZ() ? 0    : iz+1;
  const int izp = iz   == 0  ? data.sizeZ()-1 : iz-1;

  Position n;
  n.x = ( data(ixn,iy,iz) - data(ixp,iy,iz) )*data.sizeX()*0.5;
  n.y = ( data(ix,iyn,iz) - data(ix,iyp,iz) )*data.sizeY()*0.5;
  n.z = ( data(ix,iy,izn) - data(ix,iy,izp) )*data.sizeZ()*0.5;

  return n.normal();
}

// evfł̓ʃ|S쐬郁\bh
void GLIsosurf::makeElement( int ix, int iy, int iz, const Position& dr )
{
  //eߓ_̐ڑ\\f[^
  static int conn[12][2][4] = {
    {{ 0, 1, 7, 6}, { 0, 2, 8, 3}},  //  0
    {{ 1, 2, 5, 4}, { 1, 0, 6, 7}},  //  1
    {{ 2, 0, 3, 8}, { 2, 1, 4, 5}},  //  2
    {{ 3, 8, 2, 0}, { 3, 4,10, 9}},  //  3
    {{ 4, 3, 9,10}, { 4, 5, 2, 1}},  //  4
    {{ 5, 4, 1, 2}, { 5, 6, 9,11}},  //  5
    {{ 6, 5,11, 9}, { 6, 7, 1, 0}},  //  6
    {{ 7, 6, 0, 1}, { 7, 8,11,10}},  //  7
    {{ 8, 7,10,11}, { 8, 3, 0, 2}},  //  8
    {{ 9,10, 4, 3}, { 9,11, 5, 6}},  //  9
    {{10,11, 8, 7}, {10, 9, 3, 4}},  // 10
    {{11, 9, 6, 5}, {11,10, 7, 8}}   // 11
  };
  static Node node[12];
  static Node polygon[12];

  node[ 0] = nodex(ix,iy,iz);
  node[ 1] = nodey(ix,iy,iz);
  node[ 2] = nodez(ix,iy,iz);
  node[ 3] = nodex(ix,iy,iz+1);
  node[ 4] = nodey(ix,iy,iz+1);
  node[ 5] = nodez(ix,iy+1,iz);
  node[ 6] = nodex(ix,iy+1,iz);
  node[ 7] = nodey(ix+1,iy,iz);
  node[ 8] = nodez(ix+1,iy,iz);
  node[ 9] = nodex(ix,iy+1,iz+1);
  node[10] = nodey(ix+1,iy,iz+1);
  node[11] = nodez(ix+1,iy+1,iz);

  for( int is=0; is<12; is++ ){
    if( !node[is].flag ) continue;

    int n=0, i=is, m=0;
    do {
      polygon[n++]= node[i];

      int sol = indexSelected(node[conn[i][m][0]],node[conn[i][m][1]],
			      node[conn[i][m][2]],node[conn[i][m][3]]);

      i = conn[i][m][sol];
      if( sol == 2 ) m ^= 1;
      node[i].flag = false;
    }while( i!=is );

    drawElement( n, polygon, dr );
  }
}

// |S쐬郁\bh
void GLIsosurf::drawElement( int n, Node polygon[], const Position& dr )
{
  int inv;

  {
    int i=0;
    if( ((polygon[(i+n-1)%n].vertex - polygon[i].vertex) %
	 (polygon[(i+n+1)%n].vertex - polygon[i].vertex))*
	polygon[i].normal > 0.0 ){
      inv = +1;
    }else{
      inv = -1;
    }
  }

  for( int i=0; i<n; i++ ){
    if( ((polygon[(i+n-1)%n].vertex - polygon[i].vertex) %
	 (polygon[(i+n+1)%n].vertex - polygon[i].vertex))*
	polygon[i].normal > 0.0 ){
      if( inv == -1 ) polygon[i].normal *= -1;
    }else{
      if( inv == +1 ) polygon[i].normal *= -1;
    }
  }

  const Gradation& gradation = model.gloption.field.gradation_arg;

  if( inv == -1 ){
    if( n==3 ){
      for( int i=0; i<n; i++ ){
	if( icomplex_field ){
	  GLcolor color = gradation(polygon[i].phase)
	    + GLcolor::white*(0.5+0.5*scale);
	  color.A = model.gloption.field.isosurf.alpha;
	  glMaterialdv( GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, color );
	}
	glNormal3dv( polygon[i].normal );
	glVertex3dv( polygon[i].vertex+dr );
      }
    }else{
      // lp`ȏȂ璆S_܂ TRIANGLE_FAN Ƃ
      Position mid_ver = Position(0,0,0);
      Position mid_nor = Position(0,0,0);
      for( int i=0; i<n; i++ ){
	mid_ver += polygon[i].vertex;
	mid_nor += polygon[i].normal;
      }
      mid_ver *= (1.0/n);
      mid_nor = mid_nor.normal();

      for( int i=0; i<n; i++ ){
	if( icomplex_field ){
	  GLcolor color = gradation(polygon[i].phase) 
	    + GLcolor::white*(0.5+0.5*scale);
	  color.A = model.gloption.field.isosurf.alpha;
	  glMaterialdv( GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, color );
	}
	glNormal3dv( mid_nor );
	glVertex3dv( mid_ver+dr );
	glNormal3dv( polygon[i].normal );
	glVertex3dv( polygon[i].vertex+dr );

	if( icomplex_field ){
	  GLcolor color = gradation(polygon[(i+1)%n].phase)
	    + GLcolor::white*(0.5+0.5*scale);
	  color.A = model.gloption.field.isosurf.alpha;
	  glMaterialdv( GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, color );
	}
	glNormal3dv( polygon[(i+1)%n].normal );
	glVertex3dv( polygon[(i+1)%n].vertex+dr );
      }
    }
  }else{
    if( n==3 ){
      for( int i=n-1; i>=0; i-- ){
	if( icomplex_field ){
	  GLcolor color = gradation(polygon[i].phase)
	    + GLcolor::white*(0.5+0.5*scale);
	  color.A = model.gloption.field.isosurf.alpha;
	  glMaterialdv( GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, color );
	}
	glNormal3dv( polygon[i].normal );
	glVertex3dv( polygon[i].vertex+dr );
      }
    }else{
      // lp`ȏȂ璆S_܂ TRIANGLE_FAN Ƃ
      Position mid_ver = Position(0,0,0);
      Position mid_nor = Position(0,0,0);
      for( int i=0; i<n; i++ ){
	mid_ver += polygon[i].vertex;
	mid_nor += polygon[i].normal;
      }
      mid_ver *= (1.0/n);
      mid_nor = mid_nor.normal();

      for( int i=n-1; i>=0; i-- ){
	if( icomplex_field ){
	  GLcolor color = gradation(polygon[(i+1)%n].phase)
	    + GLcolor::white*(0.5+0.5*scale);
	  color.A = model.gloption.field.isosurf.alpha;
	  glMaterialdv( GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, color );
	}
	glNormal3dv( mid_nor );
	glVertex3dv( mid_ver+dr );

	glNormal3dv( polygon[(i+1)%n].normal );
	glVertex3dv( polygon[(i+1)%n].vertex+dr );
	if( icomplex_field ){
	  GLcolor color = gradation(polygon[i].phase)
	    + GLcolor::white*(0.5+0.5*scale);
	  color.A = model.gloption.field.isosurf.alpha;
	  glMaterialdv( GL_FRONT_AND_BACK, GL_AMBIENT_AND_DIFFUSE, color );
	}
	glNormal3dv( polygon[i].normal );
	glVertex3dv( polygon[i].vertex+dr );
      }

    }
  }  
}
