/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

/*!
 \file glmisc.cc
 \brief GL̂̑̏
*/

#include <stdio.h>
#include <math.h>
#include <GL/glu.h>
#include "glmisc.h"

// {F\ϐ̒`
const GLcolor GLcolor::black( 0.0, 0.0, 0.0 );
const GLcolor GLcolor::white( 1.0, 1.0, 1.0 );

const GLcolor GLcolor::dimblack( 0.2, 0.2, 0.2 );
const GLcolor GLcolor::dimgray( 0.4, 0.4, 0.4 );
const GLcolor GLcolor::gray( 0.6, 0.6, 0.6 );
const GLcolor GLcolor::lightgray( 0.8, 0.8, 0.8 );

const GLcolor GLcolor::red( 1.0, 0.0, 0.0 );
const GLcolor GLcolor::green( 0.0, 1.0, 0.0 );
const GLcolor GLcolor::blue( 0.0, 0.0, 1.0 );

const GLcolor GLcolor::yellow( 1.0, 1.0, 0.0 );
const GLcolor GLcolor::cyan( 0.0, 1.0, 1.0 );
const GLcolor GLcolor::magenta( 1.0, 0.0, 1.0 );

const GLcolor GLcolor::orange( 1.0, 0.5, 0.0 );


void glTranslateR
( const double dx, const double dy, const double dz )
{
  double mat[4][4];

  glGetDoublev( GL_MODELVIEW_MATRIX, (double*)mat );
  mat[3][0] += dx;  mat[3][1] += dy;  mat[3][2] += dz;
  glLoadMatrixd((double*)mat);
}

void glRotateR
( const double theta, const double x, const double y, const double z )
{
  const double c = cos(theta*M_PI/180.0);
  const double s = sin(theta*M_PI/180.0);

  const double xrot[4][4] = {
    { 1.0, 0.0, 0.0, 0.0 },
    { 0.0,  +c,  +s, 0.0 },
    { 0.0,  -s,  +c, 0.0 },
    { 0.0, 0.0, 0.0, 1.0 },
  };
  const double yrot[4][4] = {
    {  +c, 0.0,  -s, 0.0 },
    { 0.0, 1.0, 0.0, 0.0 },
    {  +s, 0.0,  +c, 0.0 },
    { 0.0, 0.0, 0.0, 1.0 },
  };
  const double zrot[4][4] = {
    {  +c,  +s, 0.0, 0.0 },
    {  -s,  +c, 0.0, 0.0 },
    { 0.0, 0.0, 1.0, 0.0 },
    { 0.0, 0.0, 0.0, 1.0 },
  };

  const double (*drot)[4] = xrot;

  if( x != 0.0 ){ // rotation around x-axis
    drot = xrot;
  }
  if( y != 0.0 ){ // rotation around y-axis
    drot = yrot;
  }
  if( z != 0.0 ){ // rotation around z-axis
    drot = zrot;
  }

  double mat[4][4], rot[4][4];

  glGetDoublev( GL_MODELVIEW_MATRIX, (double*)mat );

  for( int i=0; i<3; i++ ){
    for( int j=0; j<4; j++ ){
      rot[i][j] = 0.0;
      for( int k=0; k<4; k++ ){
	rot[i][j] += mat[i][k] * drot[k][j];
      }
    }
  }

  {
    int i=3;
    for( int j=0; j<4; j++ ){
      rot[i][j] = mat[i][j];
    }
  }
  
  glLoadMatrixd( (double*)rot );
}


struct BitmapFileHeader
{
  unsigned int   size;
  unsigned short reserved1;
  unsigned short reserved2;
  unsigned int   offBits;

  BitmapFileHeader( void ){}
  BitmapFileHeader( const unsigned int width, const unsigned int height ){
    size      = 14 + 40 + width*height*3;
    reserved1 = 0;
    reserved2 = 0;
    offBits   = 14 + 40;
  }
};

struct BitmapInfoHeader
{
  unsigned int   size;
  int     width;
  int     height;
  unsigned short planes;
  unsigned short bitCount;
  unsigned int   compression;
  unsigned int   sizeImage;
  int     xPixPerMeter;
  int     yPixPerMeter;
  unsigned int   clrUsed;
  unsigned int   clrImportant;

  BitmapInfoHeader( void ){}
  BitmapInfoHeader( const unsigned int _width, const unsigned int _height ){
    size = 40;
    width  = _width;
    height = _height;
    planes = 1;
    bitCount = 24;
    compression = 0;
    sizeImage = _width * _height * 3;
    xPixPerMeter = 3780;
    yPixPerMeter = 3780;
    clrUsed = 0;
    clrImportant = 0;
  }
};

bool glSavePixels( const char* fname )
{
#ifdef WIN32
  FILE* fptr =  fopen( fname, "wb" );
#else
  FILE* fptr =  fopen( fname, "w" );
#endif

  if( fptr == NULL ) return false;

  struct {
    int x, y, width, height;
  } view;
  glGetIntegerv( GL_VIEWPORT, (GLint*)&view );
  view.width = 4*((view.width-1)/4+1);

  const int image_size = view.width * view.height * 3;

  char magic[2] = {'B','M'};

  BitmapFileHeader header( view.width, view.height );
  BitmapInfoHeader info( view.width, view.height );

  GLubyte* buf = new GLubyte [ image_size ];

  for( int i=0; i<image_size; i++ ){
    buf[i] = 0;
  }

  glReadPixels( 0, 0, view.width, view.height, GL_RGB, GL_UNSIGNED_BYTE, buf );

  for( int i=0; i<image_size; i+=3 ){
    GLubyte t = buf[i+0];
    buf[i+0] = buf[i+2];
    buf[i+2] = t;
  }

  fwrite( magic, sizeof(magic), 1, fptr );
  fwrite( &header, sizeof(header), 1, fptr );
  fwrite( &info, sizeof(info), 1, fptr );
  fwrite( buf, image_size, 1, fptr );

  fclose(fptr);

  delete [] buf;

  return true;
}


GLuint glLoadTexture2D( const char* fname )
{
#ifdef WIN32
  FILE* fptr =  fopen( fname, "rb" );
#else
  FILE* fptr =  fopen( fname, "r" );
#endif
  if( fptr == NULL ){
    fprintf( stderr, "Error: can not open file %s.\n", fname );
    return 0;
  }

  char magic[2];
  BitmapFileHeader header;
  BitmapInfoHeader info;

  fread( magic, sizeof(magic), 1, fptr );
  fread( &header, sizeof(header), 1, fptr );
  fread( &info, sizeof(info), 1, fptr );

  if( magic[0] != 'B' || magic[1] != 'M' ){
    fprintf( stderr, "Error: file %s is not BMP\n", fname );
    fclose(fptr);
    return 0;
  }
  if( (info.width  & (info.width-1)  ) ||
      (info.height & (info.height-1) ) ){
    fprintf( stderr, "Error: width or height of the image file %s is not 2^n. width=%d, height=%d.\n",
	     fname, info.width, info.height );
    fclose(fptr);
    return 0;
  }

  GLubyte* image = new GLubyte [ info.sizeImage ];

  fread( image, info.sizeImage, 1, fptr );
  // swap R and B
  for( GLuint i=0; i<info.sizeImage; i+=3 ){
    GLubyte t = image[i+0];
    image[i+0] = image[i+2];
    image[i+2] = t;
  }

  fclose(fptr);

  GLuint list;
  glGenTextures( 1, &list );
  glBindTexture( GL_TEXTURE_2D, list );

  glPixelStorei( GL_UNPACK_ALIGNMENT, 1 );
  glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT );
  glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT );
  glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST );
  glTexParameteri( GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST );
  glTexEnvf( GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_MODULATE );

  glTexImage2D( GL_TEXTURE_2D, 0, GL_RGB, info.width, info.height, 0,
	        GL_RGB, GL_UNSIGNED_BYTE, image );

  delete image;

  return list;
}

void gluWireCube( const double size, const Position& center )
{
  const Position min = center - 0.5*Position(size,size,size);
  const Position max = center + 0.5*Position(size,size,size);

  glBegin( GL_LINES );
  glVertex3d( min.x, min.y, min.z );
  glVertex3d( min.x, min.y, max.z );
  glVertex3d( min.x, max.y, min.z );
  glVertex3d( min.x, max.y, max.z );
  glVertex3d( max.x, min.y, min.z );
  glVertex3d( max.x, min.y, max.z );
  glVertex3d( max.x, max.y, min.z );
  glVertex3d( max.x, max.y, max.z );
  glEnd();
  glBegin( GL_LINE_LOOP );
  glVertex3d( min.x, min.y, min.z );
  glVertex3d( min.x, max.y, min.z );
  glVertex3d( max.x, max.y, min.z );
  glVertex3d( max.x, min.y, min.z );
  glEnd();
  glBegin( GL_LINE_LOOP );
  glVertex3d( min.x, min.y, max.z );
  glVertex3d( min.x, max.y, max.z );
  glVertex3d( max.x, max.y, max.z );
  glVertex3d( max.x, min.y, max.z );
  glEnd();
}

void gluSolidCube( const double size, const Position& center )
{
  const Position min = center - 0.5*Position(size,size,size);
  const Position max = center + 0.5*Position(size,size,size);

  glEnable( GL_CULL_FACE );
  //glCullFace( GL_FRONT );
  glCullFace( GL_BACK );

  glBegin( GL_QUADS );
  glNormal3d( -1.0, 0.0, 0.0 );
  glVertex3d( min.x, min.y, min.z );
  glVertex3d( min.x, min.y, max.z );
  glVertex3d( min.x, max.y, max.z );
  glVertex3d( min.x, max.y, min.z );

  glNormal3d( +1.0, 0.0, 0.0 );
  glVertex3d( max.x, min.y, min.z );
  glVertex3d( max.x, max.y, min.z );
  glVertex3d( max.x, max.y, max.z );
  glVertex3d( max.x, min.y, max.z );

  glNormal3d( 0.0, -1.0, 0.0 );
  glVertex3d( min.x, min.y, min.z );
  glVertex3d( max.x, min.y, min.z );
  glVertex3d( max.x, min.y, max.z );
  glVertex3d( min.x, min.y, max.z );

  glNormal3d( 0.0, +1.0, 0.0 );
  glVertex3d( max.x, max.y, min.z );
  glVertex3d( min.x, max.y, min.z );
  glVertex3d( min.x, max.y, max.z );
  glVertex3d( max.x, max.y, max.z );

  glNormal3d( 0.0, 0.0, -1.0 );
  glVertex3d( min.x, min.y, min.z );
  glVertex3d( min.x, max.y, min.z );
  glVertex3d( max.x, max.y, min.z );
  glVertex3d( max.x, min.y, min.z );

  glNormal3d( 0.0, 0.0, +1.0 );
  glVertex3d( max.x, min.y, max.z );
  glVertex3d( max.x, max.y, max.z );
  glVertex3d( min.x, max.y, max.z );
  glVertex3d( min.x, min.y, max.z );

  glEnd();

  glDisable( GL_CULL_FACE );
}

void gluSolidBall( const double radius,
		  const Position& center, const int slices )
{
  static GLUquadricObj* q = gluNewQuadric();

  glPushMatrix();      
  glTranslated( center.x, center.y, center.z );
  gluSphere( q, radius, slices*2, slices );
  glPopMatrix();
}

void gluSolidStick( const double radius,
		   const Position& r1, const Position& r2, const int slices )
{
  static GLUquadricObj* q = gluNewQuadric();
  
  const double xo = r1.x;
  const double yo = r1.y;
  const double zo = r1.z;
  
  const double xa = r2.x - r1.x;
  const double ya = r2.y - r1.y;
  const double za = r2.z - r1.z;
  
  const double xa2 = xa*xa;
  const double ya2 = ya*ya;
  const double za2 = za*za;
  
  const int paxis = (xa2 > ya2 ? (xa2 > za2 ? 1 : 3) : (ya2 > za2 ? 2 : 3));
  
  double xb, yb, zb;
  if( false );
  else if( paxis == 1 ){
    xb = -ya, yb = xa, zb = 0.0;
  }
  else if( paxis == 2 ){
      xb = 0.0, yb = -za, zb = ya;
  }
  else{ // paxis == 3
    xb = za, yb = 0.0, zb = -xa;
  }
  double xc, yc, zc;
  
  xc = ya*zb - yb*za;
  yc = za*xb - zb*xa;
  zc = xa*yb - xb*ya;
  
  double irb  = 1.0/sqrt( xb*xb + yb*yb + zb*zb );
  double irc  = 1.0/sqrt( xc*xc + yc*yc + zc*zc );
  
  xb *= irb, yb *= irb, zb *= irb;
  xc *= irc, yc *= irc, zc *= irc;
  
  double mat[4][4];
  
  mat[0][0] = xb, mat[0][1] = yb, mat[0][2] = zb, mat[0][3] = 0.0;
  mat[1][0] = xc, mat[1][1] = yc, mat[1][2] = zc, mat[1][3] = 0.0;
  mat[2][0] = xa, mat[2][1] = ya, mat[2][2] = za, mat[2][3] = 0.0;
  mat[3][0] = xo, mat[3][1] = yo, mat[3][2] = zo, mat[3][3] = 1.0;
  
  glPushMatrix();{
    glMultMatrixd((double*)mat); 
    gluCylinder( q, radius, radius, 1.0, slices, 1 );
  }glPopMatrix();
}

void gluSolidCone( const double radius,
		   const Position& r1, const Position& r2, const int slices )
{
  static GLUquadricObj* q = gluNewQuadric();
  
  const double xo = r1.x;
  const double yo = r1.y;
  const double zo = r1.z;
  
  const double xa = r2.x - r1.x;
  const double ya = r2.y - r1.y;
  const double za = r2.z - r1.z;
  
  const double xa2 = xa*xa;
  const double ya2 = ya*ya;
  const double za2 = za*za;
  
  const int paxis = (xa2 > ya2 ? (xa2 > za2 ? 1 : 3) : (ya2 > za2 ? 2 : 3));
  
  double xb, yb, zb;
  if( false );
  else if( paxis == 1 ){
    xb = -ya, yb = xa, zb = 0.0;
  }
  else if( paxis == 2 ){
      xb = 0.0, yb = -za, zb = ya;
  }
  else{ // paxis == 3
    xb = za, yb = 0.0, zb = -xa;
  }
  double xc, yc, zc;
  
  xc = ya*zb - yb*za;
  yc = za*xb - zb*xa;
  zc = xa*yb - xb*ya;
  
  double irb  = 1.0/sqrt( xb*xb + yb*yb + zb*zb );
  double irc  = 1.0/sqrt( xc*xc + yc*yc + zc*zc );
  
  xb *= irb, yb *= irb, zb *= irb;
  xc *= irc, yc *= irc, zc *= irc;
  
  double mat[4][4];
  
  mat[0][0] = xb, mat[0][1] = yb, mat[0][2] = zb, mat[0][3] = 0.0;
  mat[1][0] = xc, mat[1][1] = yc, mat[1][2] = zc, mat[1][3] = 0.0;
  mat[2][0] = xa, mat[2][1] = ya, mat[2][2] = za, mat[2][3] = 0.0;
  mat[3][0] = xo, mat[3][1] = yo, mat[3][2] = zo, mat[3][3] = 1.0;
  
  glPushMatrix();{
    glMultMatrixd((double*)mat); 
    gluCylinder( q, radius, 0.0, 1.0, slices, 1 );
  }glPopMatrix();
}

void glRasterPos3dv( const Position& pos )
{
  glRasterPos3d( pos.x, pos.y, pos.z );
}

void glutBitmapString( const char* str )
{
}

int glGetWindowWidth( void )
{
  struct{
    GLint xo, yo, width, height;
  } view;

  glGetIntegerv( GL_VIEWPORT, (GLint*)&view );

  return view.width;
}

int glGetWindowHeight( void )
{
  struct{
    GLint xo, yo, width, height;
  } view;

  glGetIntegerv( GL_VIEWPORT, (GLint*)&view );

  return view.height;
}
