/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

/*!
 \file glmisc.h
 \brief GL̂̑̏
*/

#ifndef __GLMISC_H_INCLUDED
#define __GLMISC_H_INCLUDED

#include <GL/gl.h>
#include <GL/glu.h>

#include <vector>
#include "position.h"

// GLp̐F̍\
struct GLcolor
{
  GLdouble R, G, B, A;

  GLcolor( void ){
  }

  GLcolor( const GLdouble _r, const GLdouble _g,
	   const GLdouble _b, const GLdouble _a = 1.0 ) :
    R(_r), G(_g), B(_b), A(_a)
  {
  }

  GLcolor( const GLcolor& _c, const GLdouble _a ) :
    R(_c.R), G(_c.G), B(_c.B), A(_a)
  {
  }

  operator const GLdouble* () const {
    return &R;
  }
  operator const GLfloat* () const {
    static GLfloat c[4];
    c[0] = GLfloat(R);
    c[1] = GLfloat(G);
    c[2] = GLfloat(B);
    c[3] = GLfloat(A);
    return c;
  }
  const GLdouble& operator [] ( unsigned int i ) const {
    return (&R)[i];
  }
  GLdouble& operator [] ( unsigned int i ) {
    return (&R)[i];
  }

  GLcolor operator * ( const double f ) const {
    return GLcolor( R*f, G*f, B*f, A );
  }
  GLcolor operator + ( const GLcolor& c ) const {
    return GLcolor( R+c.R, G+c.G, B+c.B, A );
  }

  // {F̕ϐ̒`
public:
  static const GLcolor black;
  static const GLcolor white;

  static const GLcolor dimblack;
  static const GLcolor dimgray;
  static const GLcolor gray;
  static const GLcolor lightgray;

  static const GLcolor red;
  static const GLcolor green;
  static const GLcolor blue;

  static const GLcolor yellow;
  static const GLcolor cyan;
  static const GLcolor magenta;

  static const GLcolor orange;
};

inline void glClearColor( const GLdouble* color )
{
  glClearColor( (GLfloat)color[0], (GLfloat)color[1],
		(GLfloat)color[2], (GLfloat)color[3] );
}

class Gradation
{
public:
  std::vector<GLcolor> vcolor;
  GLdouble         alpha;
public:
  Gradation( void ){}
  Gradation( const GLcolor& c0, const GLdouble alpha=1.0 ){
    vcolor.push_back( GLcolor(c0,alpha) );
    this->alpha = alpha;
  }
  Gradation( const GLcolor& c0, const GLcolor& c1, const GLdouble alpha=1.0 ){
    vcolor.push_back( GLcolor(c0,alpha) );
    vcolor.push_back( GLcolor(c1,alpha) );
    this->alpha = alpha;
  }
  Gradation( const GLcolor& c0, const GLcolor& c1, const GLcolor& c2,
	      const GLdouble alpha=1.0 ){
    vcolor.push_back( GLcolor(c0,alpha) );
    vcolor.push_back( GLcolor(c1,alpha) );
    vcolor.push_back( GLcolor(c2,alpha) );
    this->alpha = alpha;
  }
  Gradation( const GLcolor& c0, const GLcolor& c1,
	     const GLcolor& c2, const GLcolor& c3,
	     const GLdouble alpha=1.0 ){
    vcolor.push_back( GLcolor(c0,alpha) );
    vcolor.push_back( GLcolor(c1,alpha) );
    vcolor.push_back( GLcolor(c2,alpha) );
    vcolor.push_back( GLcolor(c3,alpha) );
    this->alpha = alpha;
  }
  Gradation( const GLcolor& c0, const GLcolor& c1,
	     const GLcolor& c2, const GLcolor& c3,
	     const GLcolor& c4,
	     const GLdouble alpha=1.0 ){
    vcolor.push_back( GLcolor(c0,alpha) );
    vcolor.push_back( GLcolor(c1,alpha) );
    vcolor.push_back( GLcolor(c2,alpha) );
    vcolor.push_back( GLcolor(c3,alpha) );
    vcolor.push_back( GLcolor(c4,alpha) );
    this->alpha = alpha;
  }
  void clear( void ){
    vcolor.clear();
  }
  void push( const GLcolor& color ){
    vcolor.push_back( color );
    this->alpha = color.A;
  }

  void setAlpha( const double alpha ){
    for( int c=0; c<(int)vcolor.size(); c++ ){
      vcolor[c].A = alpha;
    }
    this->alpha = alpha;
  }

  GLcolor operator () ( const double value ) const {
    if( vcolor.size() == 1 ) return vcolor.front();

    if( value <= 0.0 ){
      double f = 1.0 + 8.0*value;
      if( f < 0.0 ) f = 0.0;
      return GLcolor( vcolor.front(), vcolor.front().A*f );
    }
    if( value >= 1.0 ){
      double f = 1.0 - 8.0*(value-1.0);
      if( f < 0.0 ) f = 0.0;
      return GLcolor( vcolor.back(), vcolor.back().A*f );
    }

    const double dv = 1.0/(vcolor.size()-1);
    for( int i=0; i+1<(int)vcolor.size(); i++ ){
      if( dv*i <= value && value <= dv*(i+1) ){
	const double f = (value-dv*i)/dv;
	return vcolor[i] * (1.0-f) + vcolor[i+1] * (f);
      }
    }

    return vcolor.front();
  }

  GLcolor& operator [] ( const int i ){
    return vcolor[i];
  }
  const GLcolor& operator [] ( const int i ) const{
    return vcolor[i];
  }

  GLcolor bottom( void ) const {
    return vcolor.front();
  }
  GLcolor middle( void ) const {
    return (vcolor.front() + vcolor.back())*0.5;
  }
  GLcolor top( void ) const {
    return vcolor.back();
  }
};

/// GL֐ glLightf()  double^ւ̓K
inline void glLightd( GLenum light, GLenum pname, GLdouble param ) 
{
  glLightf( light, pname, (GLfloat)param );
}

/// GL֐ glLightfv()  double^ւ̓K
inline void glLightdv( GLenum light, GLenum pname, const GLdouble *dparam ) 
{
  GLfloat fparam[4];
  fparam[0] = (GLfloat)dparam[0];
  fparam[1] = (GLfloat)dparam[1];
  fparam[2] = (GLfloat)dparam[2];
  fparam[3] = (GLfloat)dparam[3];
  glLightfv( light, pname, fparam );
}

/// GL֐ glLightModelf()  double^ւ̓K
inline void glLightModeld( GLenum pname, GLdouble param )
{
  glLightModelf( pname, (GLfloat)param );
}

/// GL֐ glLightModelfv()  double^ւ̓K
inline void glLightModeldv( GLenum pname, const GLdouble *dparam )
{
  GLfloat fparam[4];
  fparam[0] = (GLfloat)dparam[0];
  fparam[1] = (GLfloat)dparam[1];
  fparam[2] = (GLfloat)dparam[2];
  fparam[3] = (GLfloat)dparam[3];
  glLightModelfv( pname, fparam );
}

/// GL֐ glMaterialf()  double^ւ̓K
inline void glMateriald
( GLenum face, GLenum pname, GLdouble param )
{
  glMaterialf( face, pname, (GLfloat)param );
}

/// GL֐ glMaterialfv()  double^ւ̓K
inline void glMaterialdv
( GLenum face, GLenum pname, const GLdouble* dparam )
{
  GLfloat fparam[4];
  fparam[0] = (GLfloat)dparam[0];
  fparam[1] = (GLfloat)dparam[1];
  fparam[2] = (GLfloat)dparam[2];
  fparam[3] = (GLfloat)dparam[3];
  glMaterialfv( face, pname, fparam );
}

/// GLU֐ gluLookAt()  double[]^ւ̓K
inline void gluLookAt
( const GLdouble eye[], const GLdouble gaze[], const GLdouble up[] )
{
  gluLookAt( eye[0], eye[1], eye[2],
	     gaze[0], gaze[1], gaze[2],
	     up[0], up[1], up[2] );
}

inline void glTranslatedv( const Position& t )
{
  glTranslated( t.x, t.y, t.z);
}

void glTranslateR( const double dx, const double dy, const double dz );
void glRotateR( const double theta, const double x, const double y, const double z );
bool glSavePixels( const char* fname );

GLuint glLoadTexture2D( const char* fname );

void glRasterPos3dv( const Position& pos );
void glutBitmapString( const char* str );

void gluWireCube( const double size, const Position& center );
void gluSolidCube( const double size, const Position& center );
void gluSolidBall( const double radius, const Position& center, const int slices );
void gluSolidStick( const double radius,
		   const Position& r1, const Position& r2, const int slices );
void gluSolidCone( const double radius,
		   const Position& r1, const Position& r2, const int slices );

int glGetWindowWidth( void );
int glGetWindowHeight( void );

#endif // __GLMISC_H_INCLUDED
