/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

/*!
 \file glview.cc
 \brief GLr[ÃNX
*/

#include "dtmodel.h"
#include "glview.h"
#include "qtexception.h"


void GLView::drawRegion( void )
{
  const int win_width  = glGetWindowWidth();
  const int win_height = glGetWindowHeight();

  glDisable( GL_DEPTH_TEST );
  glDisable( GL_LIGHTING );
  glColor4dv( GLcolor::white );

  glMatrixMode( GL_MODELVIEW );
  glPushMatrix();
  glLoadIdentity();
  
  glMatrixMode( GL_PROJECTION );
  glPushMatrix();
  glLoadIdentity();
  gluOrtho2D( 0.0, (GLfloat)win_width, (GLfloat)win_height, 0.0 );

  glColor4dv( GLcolor::yellow );
  glBegin( GL_LINE_LOOP );
  glVertex2d( x_down, y_down );
  glVertex2d( x_down, y_drag );
  glVertex2d( x_drag, y_drag );
  glVertex2d( x_drag, y_down );
  glEnd();

  glPopMatrix();
  glMatrixMode( GL_MODELVIEW );
  glPopMatrix();
  
  glEnable( GL_LIGHTING );
  glEnable( GL_DEPTH_TEST );
}

GLView::GLView( DTModel& _model, QWidget *parent ) :
  QGLWidget(parent),
  model(_model),
  lattice(_model,this),
  field(_model,this)
{
  setFocusPolicy( Qt::ClickFocus );
  setSizePolicy( QSizePolicy::Expanding, QSizePolicy::Expanding);

  connect( &model, SIGNAL(changed()),   this, SLOT(update()) );
  connect(   this, SIGNAL(changed()), &model, SLOT(update()) );

  //|bvAbvj[̐ݒ
  popupmenu = new QMenu(this);
  popupmenu->addAction( new QAction(tr("reset to xy-plane"),this) );
  popupmenu->addAction( new QAction(tr("reset to yz-plane"),this) );
  popupmenu->addAction( new QAction(tr("reset to zx-plane"),this) );
  popupmenu->addAction( new QAction(tr("reset to ab-plane"),this) );
  popupmenu->addAction( new QAction(tr("reset to bc-plane"),this) );
  popupmenu->addAction( new QAction(tr("reset to ca-plane"),this) );
  popupmenu->addSeparator();
  popupmenu->addAction( new QAction(tr("change perspective/orthogonal"),this) );
  popupmenu->addSeparator();
  popupmenu->addAction( new QAction(tr("change fullscreen/window"),this) );
  popupmenu->addSeparator();
  popupmenu->addAction( new QAction(tr("save this image"),this) );
  popupmenu->addSeparator();
  popupmenu->addAction( new QAction(tr("reload default.glml"),this) );
  popupmenu->addSeparator();
  popupmenu->addAction( new QAction(tr("quit"),this) );
  connect(popupmenu, SIGNAL(triggered(QAction*)), this, SLOT(menuEvent(QAction*)));

  shift_dragging = false;
}

void GLView::update( void )
{
  if( model.gloption.lattice.show ){
    lattice.update();
  }
  if( model.gloption.field.show ){
    field.update();
  }

  updateGL();
}

void GLView::initializeGL( void )
{
  glClearColor( model.gloption.window.background );

  glMatrixMode( GL_MODELVIEW );
  glLoadIdentity();
  gluLookAt( model.gloption.location.eye,
	     model.gloption.location.gaze,
	     model.gloption.location.up );

  glEnable( GL_DEPTH_TEST );
  glEnable( GL_NORMALIZE );

  glShadeModel( GL_SMOOTH );
  glLightModeli(  GL_LIGHT_MODEL_TWO_SIDE, 1 );
  glLightModeli(  GL_LIGHT_MODEL_LOCAL_VIEWER, 1 );
  glLightModeldv( GL_LIGHT_MODEL_AMBIENT, model.gloption.light.ambient );
  glEnable( GL_LIGHTING );

  glLightdv( GL_LIGHT0, GL_POSITION, model.gloption.light.direction );
  glLightdv( GL_LIGHT0, GL_DIFFUSE,  model.gloption.light.diffuse );
  glLightdv( GL_LIGHT0, GL_SPECULAR, model.gloption.light.specular );
  glEnable( GL_LIGHT0 );

  glLightdv( GL_LIGHT1, GL_POSITION, model.gloption.light.second_direction );

  glLightdv( GL_LIGHT1, GL_DIFFUSE,  model.gloption.light.diffuse );
  glLightdv( GL_LIGHT1, GL_SPECULAR, model.gloption.light.specular );
  if( model.gloption.light.second ){
    glEnable( GL_LIGHT1 );
  }

  glFogf( GL_FOG_MODE, GL_LINEAR );
  glFogf( GL_FOG_START,   model.gloption.location.world );
  glFogf( GL_FOG_END, 1.6*model.gloption.location.world );
  glFogfv( GL_FOG_COLOR,  model.gloption.window.background );
  if( model.gloption.light.fog ){
    glEnable( GL_FOG );
  }

  glMaterialdv( GL_FRONT_AND_BACK, GL_SPECULAR,  model.gloption.light.specular  );
  glMateriald ( GL_FRONT_AND_BACK, GL_SHININESS, model.gloption.light.shininess );

  glClearColor( model.gloption.window.background );
  glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
}


void GLView::gluProjection( int width, int height, bool perspective )
{
  if( perspective ){
    const double aspect = (double)width/height;
    gluPerspective( model.gloption.location.pangle,
		    aspect,
		    model.gloption.location.pnear,
		    model.gloption.location.pfar );
  }else{
    const double t = tan(model.gloption.location.pangle*0.5*M_PI/180.0);
    const double aspect = (double)width/height;
    const double top    = 0.65*sqrt(model.gloption.location.pnear*model.gloption.location.pfar) * t;
    const double bottom = -top;
    const double right  = top*aspect;
    const double left   = -right;

    glOrtho( left, right, bottom, top, 
	     model.gloption.location.pnear,
	     model.gloption.location.pfar );
  }
}

void GLView::resizeGL(int width, int height)
{
  glViewport( 0, 0, width, height );

  glMatrixMode( GL_PROJECTION );
  glLoadIdentity();

  gluProjection( width, height, model.gloption.location.perspective );

  glMatrixMode( GL_MODELVIEW );
  
  glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );
}

void GLView::paintGL( void )
{
  if( model.gloption.light.second ){
    glEnable( GL_LIGHT1 );
  }else{
    glDisable( GL_LIGHT1 );
  }
  if( model.gloption.light.fog ){
    glFogfv( GL_FOG_COLOR, model.gloption.window.background );
    glEnable( GL_FOG );
  }else{
    glDisable( GL_FOG );
  }
  glClearColor( model.gloption.window.background );  
  glClear( GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT );

  draw();

  if( shift_dragging ){
    drawRegion();
  }
}

// mouse drag begin
void GLView::mousePressEvent( QMouseEvent* ev )
{
  if( !(ev->buttons()&Qt::LeftButton) ) return;

  int x = ev->x(), y = ev->y();

  rotate_view = true;
  if( model.lattice.getData().isAtomSelected() ){
    vector<int> vselected = select( ev );
    if( vselected.size() == 1 &&
	vselected[0] == model.lattice.getData().vselected.back() ){
      rotate_view = false;
    }
  }


  x_down = x, y_down = y;
}

// mouse selection
void GLView::mouseDoubleClickEvent( QMouseEvent* ev )
{
  vector<int> vselected = select( ev );

  if( vselected.empty() ){
    model.lattice.getData().unselectAtom();
    emit changed();
    makeCurrent();
  }
  else if( ev->modifiers() & Qt::ShiftModifier ){
    bool status=false;
    for( int i=0; i<(int)vselected.size(); i++ ){
	status |= model.lattice.addselectAtom(vselected[i]);
    }
    if( status ){
      emit changed();
      makeCurrent();
    }
  }
  else{
    if( ev->modifiers() & Qt::ControlModifier ){
      if( model.lattice.addselectAtom(vselected[0]) ){
	emit changed();
	makeCurrent();
      }
      else{
	model.lattice.getData().unselectAtom();
	emit changed();
	makeCurrent();
      }
    }
    else{
      if( model.lattice.selectAtom(vselected[0]) ){
	emit changed();
	makeCurrent();
      }
      else{
	model.lattice.getData().unselectAtom();
	emit changed();
	makeCurrent();
      }
    }
  }

  x_down = ev->x(), y_down = ev->y();
}


vector<int> GLView::select(  QMouseEvent* ev )
{
  vector<int> vselected;

  static GLuint data[1024];
  glSelectBuffer( sizeof(data)/sizeof(GLuint), data );

  glRenderMode(GL_SELECT);
  glInitNames();
  glPushName( GLuint(-1) );

  glMatrixMode(GL_PROJECTION);
  glPushMatrix();
  glLoadIdentity();

  struct { GLint xo, yo, width, height; } viewport;
  glGetIntegerv( GL_VIEWPORT, (GLint*)&viewport );

  int xo, yo;
  int xw, yw;

  if( ev->modifiers() & Qt::ShiftModifier ){
    // ͈͓I[h
    xo = (ev->x() + x_down )/2;
    yo = (ev->y() + y_down )/2;
    xw = ev->x() - x_down;
    yw = ev->y() - y_down;
    if( xw<0 ) xw = -xw;
    if( yw<0 ) yw = -yw;
  }else{
    // _ӑI[h
    xo = ev->x();
    yo = ev->y();
    xw = 4;
    yw = 4;
  }

  gluPickMatrix( xo, viewport.height-1 - yo,
		  xw, yw, (GLint*)&viewport );

  gluProjection( viewport.width, viewport.height,
		 model.gloption.location.perspective );
  glMatrixMode(GL_MODELVIEW);

  draw();

  glMatrixMode(GL_PROJECTION);
  glPopMatrix();
  glMatrixMode(GL_MODELVIEW);

  GLuint hits = glRenderMode(GL_RENDER);
  GLuint selected = GLuint(-1);
  GLuint zminmin  = GLuint(-1);

  if( hits == 0 ){
    return vselected;
  }

  if( ev->modifiers() & Qt::ShiftModifier ){
    // qI[h
    for( GLuint i=0,j=0; i<hits; i++ ){
      GLuint n    = data[j++];
      j++;  // zmin
      j++;  // zmax
      if( n==1 ){
	selected = data[j++];
	vselected.push_back( (int)selected );
      }
    }
  }
  else{
    // PqI[h
    for( GLuint i=0,j=0; i<hits; i++ ){
      GLuint n    = data[j++];
      GLuint zmin = data[j++];
      j++; // zmax
      if( n==1 ){
	// ԋ߂̂I
	if( zminmin>zmin ){
	  zminmin = zmin;
	  selected = data[j++];
	}else{
	  j++;
	}
      }
    }
    vselected.push_back( (int)selected );
  }

  return vselected;
}

// mouse drag end
void GLView::mouseReleaseEvent( QMouseEvent* ev )
{
  // ͈͓I[h
  if( ev->modifiers() & Qt::ShiftModifier ){
    mouseDoubleClickEvent(ev);
    shift_dragging = false;
  }
  else if( model.lattice.getData().isAtomSelected() ){
    emit changed(); // }EXƂɕωʒmB
    makeCurrent();
  }
}

// mouse drag
void GLView::mouseMoveEvent( QMouseEvent* ev )
{
  if( !(ev->buttons()&Qt::LeftButton) ) return;

  // ͈͓I[h ł͉Ȃ
  if( ev->modifiers() & Qt::ShiftModifier ){
    x_drag = ev->x();
    y_drag = ev->y();
    shift_dragging = true;
    updateGL();
    return;
  }
  else{
    shift_dragging = false;
  }

  int x = ev->x(), y = ev->y();

  const double dx = (1.0/4)*(x-x_down);
  const double dy = (1.0/4)*(y-y_down);

  if( fabs(dx) < 1.0 && fabs(dy) < 1.0 ) return;

  if( !rotate_view ){
    //  if( model.lattice.getData().isAtomSelected() ){
    // ݂3D\̕ϊs肷
    double mat[4][4]; // 4x4zœ肷B
    glGetDoublev( GL_MODELVIEW_MATRIX, (GLdouble*)mat );
    // 3x3sɕϊ
    Matrix M;
    for( int i=0; i<3; i++ ){
      for( int j=0; j<3; j++ ){
	M(i,j) = mat[j][i];
      }
    }

    // }EX̓3DԂ̍Wnł̕ɕϊ
    const Position t = M.getInverse() * ((1.0/16) * Position(dx,-dy,0.0));
    // 3DԂŌqW𕽍sړB
    //    const Coordinates c = model.lattice.cell.getCoordinatesLocal(t);

    if( ev->modifiers() & Qt::ControlModifier ){
      if( model.lattice.moveAtomCoords(t) ){
	lattice.update();
      }
      else{
	if( MyException::question( "motion for this direction is restricted by symmetry.\nupdate symmetry?" ) ){
	  model.lattice.symmetry.clear();
	  model.lattice.updateAtoms();
	  model.lattice.moveAtomCoords(t);
	  lattice.update();
	}
      }
    }
    else{
      if( model.lattice.moveAtomsCoords(t) ){
	lattice.update();
      }
      else{
	if( MyException::question( "motion for this direction is restricted by symmetry.\nupdate symmetry?" ) ){
	  model.lattice.symmetry.clear();
	  model.lattice.updateAtoms();
	  model.lattice.moveAtomsCoords(t);
	  lattice.update();
	}
      }

    }
  }
  else{
    glRotateR( dy, 1.0, 0.0, 0.0 );
    glRotateR( dx, 0.0, 1.0, 0.0 );
  }

  x_down = x, y_down = y;

  updateGL();
}

void GLView::wheelEvent( QWheelEvent* ev )
{
  if( ev->orientation() == Qt::Horizontal ) return;

  const double dy = (-2.0) * (double)ev->delta() / 120.0;

  {
    const double f = 1.0 + (1.0/256)*dy;
    glScaled(f,f,f);
  }

  updateGL();
}

void GLView::keyPressEvent( QKeyEvent* ev )
{
  double dx=0.0, dy=0.0, dz=0.0;
  switch( ev->key() ){
  case Qt::Key_Left     : dx=-1.0; break;
  case Qt::Key_Right    : dx=+1.0; break;
  case Qt::Key_Up       : dy=-1.0; break;
  case Qt::Key_Down     : dy=+1.0; break;
  case Qt::Key_PageDown : dz=-1.0; break;
  case Qt::Key_PageUp   : dz=+1.0; break;
  default : break;
  }

  if( ev->modifiers() & Qt::ControlModifier ) {
    glRotateR( 15*dy, 1.0, 0.0, 0.0 );
    glRotateR( 15*dx, 0.0, 1.0, 0.0 );
    glRotateR( 15*dz, 0.0, 0.0, 1.0 );
  }
  else if( model.lattice.getData().isAtomSelected() ){
    double mat[4][4];
    glGetDoublev( GL_MODELVIEW_MATRIX, (GLdouble*)mat );
    
    Matrix M;
    for( int i=0; i<3; i++ ){
      for( int j=0; j<3; j++ ){
	M(i,j) = mat[j][i];
      }
    }

    const Position t = M.getInverse() * ((1.0/16) * Position(dx,-dy,0.0));

    //    const Coordinates c = model.lattice.cell.getCoordinatesLocal(t);
    model.lattice.moveAtomCoords(t);
    emit changed();
    makeCurrent();
  }
  else{
    switch( ev->key() ){
    case Qt::Key_PageDown :  {
      const double f = 1.0 + (1.0/8);
      glScaled(f,f,f);
    } break; 
    case Qt::Key_PageUp :  {
      const double f = 1.0 - (1.0/8);
      glScaled(f,f,f);
    } break;
    case Qt::Key_Down     : {
      const double dy=-15.0;
      glTranslateR( 0.0, (1.0/4)*(dy), 0.0 );
    } break;
    case Qt::Key_Up       : {
      const double dy=+15.0;
      glTranslateR( 0.0, (1.0/4)*(dy), 0.0 );
    } break;
    case Qt::Key_Left     : {
      const double dx=-15.0;
      glTranslateR( (1.0/4)*(dx), 0.0, 0.0 );
    } break;
    case Qt::Key_Right    : {
      const double dx=+15.0;
      glTranslateR( (1.0/4)*(dx), 0.0, 0.0 );
    } break;

    default : break;
    }
  }

  // ̑̑
  switch( ev->key() ){

  case Qt::Key_F1 : {
    model.gloption.window.fullscreen =
      !model.gloption.window.fullscreen;

    if( model.gloption.window.fullscreen ){
      ((QGLWidget*)parent())->showFullScreen();
    }else{
      ((QGLWidget*)parent())->showNormal();
    }
    updateGL();
  } break;

  default             : break;
  }

  updateGL();
}


void GLView::keyReleaseEvent( QKeyEvent* ev )
{
  if( ev->isAutoRepeat() ) return;

  switch( ev->key() ){
  case Qt::Key_Left   : break;
  case Qt::Key_Right  : break;
  case Qt::Key_Down   : break;
  case Qt::Key_Up     : break;
  case Qt::Key_PageDown : break;
  case Qt::Key_PageUp   : break;
  default             : return;
  }
}

void GLView::contextMenuEvent( QContextMenuEvent* ev )
{
  popupmenu->exec(ev->globalPos());
}

void GLView::menuEvent( QAction* action )
{
  if( false ){
  }
  else if( action->text() == "reset to ab-plane" ||
	   action->text() == "reset to bc-plane" ||
	   action->text() == "reset to ca-plane" ){
    glLoadIdentity();
    gluLookAt( model.gloption.location.eye,
	       model.gloption.location.gaze,
	       model.gloption.location.up );
    model.gloption.light.direction =
      model.gloption.light.direction_orig;
    glLightdv( GL_LIGHT0, GL_POSITION,
	       model.gloption.light.direction );

    Position Ea(0.0,0.0,0.0);
    Position Eb(0.0,0.0,0.0);
    Position Ec(0.0,0.0,0.0);

    if( action->text() == "reset to ab-plane" ){
      Ea = model.lattice.cell.Ea.normal();
      Eb = Position::outer_product(model.lattice.cell.Ec,Ea).normal();
      Ec = Position::outer_product(Ea,Eb).normal();
    }
    if( action->text() == "reset to bc-plane" ){
      Ea = model.lattice.cell.Eb.normal();
      Eb = Position::outer_product(model.lattice.cell.Ea,Ea).normal();
      Ec = Position::outer_product(Ea,Eb).normal();
    }
    if( action->text() == "reset to ca-plane" ){
      Ea = model.lattice.cell.Ec.normal();
      Eb = Position::outer_product(model.lattice.cell.Eb,Ea).normal();
      Ec = Position::outer_product(Ea,Eb).normal();
    }

    Matrix invM = Matrix( Ea.x, Ea.y, Ea.z,
			  Eb.x, Eb.y, Eb.z,
			  Ec.x, Ec.y, Ec.z ).getInverse();
    double mat[4][4];

    for( int i=0; i<4; i++ ){
      for( int j=0; j<4; j++ ){
	if( i==3 || j==3 ){
	  if( i==3 && j==3 ){
	    mat[i][j] = 1.0;
	  }
	  else{
	    mat[i][j] = 0.0;
	  }
	}
	else{
	  mat[i][j] = invM.M[i][j];
	}
      }
    }
    glMultMatrixd((double*)mat);
  }
  else if( action->text() == "reset to xy-plane" ||
	   action->text() == "reset to yz-plane" ||
	   action->text() == "reset to zx-plane" ){
    glLoadIdentity();
    gluLookAt( model.gloption.location.eye,
	       model.gloption.location.gaze,
	       model.gloption.location.up );
    model.gloption.light.direction =
      model.gloption.light.direction_orig;
    glLightdv( GL_LIGHT0, GL_POSITION,
	       model.gloption.light.direction );

    if( action->text() == "reset to xy-plane" ){
    }
    if( action->text() == "reset to yz-plane" ){
      glRotateR( -90.0, 0.0, 0.0, 1.0 );
      glRotateR( -90.0, 1.0, 0.0, 0.0 );
    }
    if( action->text() == "reset to zx-plane" ){
      glRotateR( 90.0, 0.0, 1.0, 0.0 );
      glRotateR( 90.0, 1.0, 0.0, 0.0 );
    }
  }
  else if( action->text() == "change perspective/orthogonal" ){
    model.gloption.location.perspective =
      !model.gloption.location.perspective;

    resizeGL( glGetWindowWidth(), glGetWindowHeight() );
  }

  else if( action->text() == "change fullscreen/window" ){
    model.gloption.window.fullscreen =
      !model.gloption.window.fullscreen;

    if( model.gloption.window.fullscreen ){
      ((QGLWidget*)parent())->showFullScreen();
    }else{
      ((QGLWidget*)parent())->showNormal();
    }
    updateGL();
  }

  else if( action->text() == "save this image" ){
    save();
  }
  else if( action->text() == "reload default.glml" ){
    model.gloption.loadDefaults();
    emit changed();
    makeCurrent();
  }
  else if( action->text() == "quit" ){
    exit(0);
  }
}


void GLView::draw( void )
{
  for( int p=0; p<6; p++ ){
    glDisable( GL_CLIP_PLANE0+p );
  }

  const double L = model.getScale(); // n̂܂ȑ傫
  const double f = model.gloption.location.world/L*(0.5);

  glPushMatrix();
  glScaled( f, f, f );

  int value;
  glGetIntegerv( GL_RENDER_MODE, &value );

  if( value == GL_RENDER ){
    if( model.gloption.location.wigner ){
      drawWigner();
    }
    else if( model.gloption.location.extendA > 0 ||
	     model.gloption.location.extendB > 0 ||
	     model.gloption.location.extendC > 0 ){
      drawPeriodic();
    }
    else{
      drawCell();
    }
  }
  else{
    drawCell();
  }

  glPopMatrix();
}

void GLView::drawCell( void )
{
  if( model.gloption.lattice.show ){
    lattice.draw();
  }

  if( model.gloption.field.show ){
    if( model.field.isset() ) field.draw();
  }
}


void GLView::drawPeriodic( void )
{
  const Position& La = model.lattice.cell.La;
  const Position& Lb = model.lattice.cell.Lb;
  const Position& Lc = model.lattice.cell.Lc;
  const Position& Lo = model.lattice.cell.Lo;
  const Position& Ka = model.lattice.cell.Ka;
  const Position& Kb = model.lattice.cell.Kb;
  const Position& Kc = model.lattice.cell.Kc;

  // note: clip plane equation:
  // eq[0]*x + eq[1]*y + eq[2]*z + eq[3] > 0 : visible
  double eq[4]; // NbsO
  const double delta = +1.0/1024; // NbsO̗V

  int p=0;
  if( model.gloption.location.extendA > 0 &&
      model.gloption.location.extendA < 100 ){
    // note: clip plane equation:
    // eq[0]*x + eq[1]*y + eq[2]*z + eq[3] > 0 : visible
    eq[0] = (+1.0)*Ka.x, eq[1] = (+1.0)*Ka.y, eq[2] = (+1.0)*Ka.z;
    eq[3] = (-1.0)*(Ka*Lo) +
      double(model.gloption.location.extendA)/100.0+delta;
    glClipPlane( GL_CLIP_PLANE0+p, eq );
    glEnable   ( GL_CLIP_PLANE0+p );
    p++;

    eq[0] = (-1.0)*Ka.x, eq[1] = (-1.0)*Ka.y, eq[2] = (-1.0)*Ka.z;
    eq[3] = (+1.0)*(Ka*Lo) +
      double(model.gloption.location.extendA)/100.0+1.0+delta;
    glClipPlane( GL_CLIP_PLANE0+p, eq );
    glEnable   ( GL_CLIP_PLANE0+p );
    p++;
  }

  if( model.gloption.location.extendB > 0 &&
      model.gloption.location.extendB < 100 ){
    eq[0] = (+1.0)*Kb.x, eq[1] = (+1.0)*Kb.y, eq[2] = (+1.0)*Kb.z;
    eq[3] = (-1.0)*(Kb*Lo) +
      double(model.gloption.location.extendB)/100.0+delta;
    glClipPlane( GL_CLIP_PLANE0+p, eq );
    glEnable   ( GL_CLIP_PLANE0+p );
    p++;

    eq[0] = (-1.0)*Kb.x, eq[1] = (-1.0)*Kb.y, eq[2] = (-1.0)*Kb.z;
    eq[3] = (+1.0)*(Kb*Lo) +
      double(model.gloption.location.extendB)/100.0+1.0+delta;
    glClipPlane( GL_CLIP_PLANE0+p, eq );
    glEnable   ( GL_CLIP_PLANE0+p );
    p++;
  }

  if( model.gloption.location.extendC > 0 &&
      model.gloption.location.extendC < 100 ){
    eq[0] = (+1.0)*Kc.x, eq[1] = (+1.0)*Kc.y, eq[2] = (+1.0)*Kc.z;
    eq[3] = (-1.0)*(Kc*Lo) +
      double(model.gloption.location.extendC)/100.0+delta;
    glClipPlane( GL_CLIP_PLANE0+p, eq );
    glEnable   ( GL_CLIP_PLANE0+p );
    p++;

    eq[0] = (-1.0)*Kc.x, eq[1] = (-1.0)*Kc.y, eq[2] = (-1.0)*Kc.z;
    eq[3] = (+1.0)*(Kc*Lo) +
      double(model.gloption.location.extendC)/100.0+1.0+delta;
    glClipPlane( GL_CLIP_PLANE0+p, eq );
    glEnable   ( GL_CLIP_PLANE0+p );
    p++;
  }

  int pxs, pxe;
  int pys, pye;
  int pzs, pze;

  if( model.gloption.location.extendA > 0 ){
    pxs = -1, pxe = +1;
  }else{
    pxs = pxe = 0;
  }
  if( model.gloption.location.extendB > 0 ){
    pys = -1, pye = +1;
  }else{
    pys = pye = 0;
  }
  if( model.gloption.location.extendC > 0 ){
    pzs = -1, pze = +1;
  }else{
    pzs = pze = 0;
  }

  for( int px=pxs; px<=+pxe; px++ ){
    for( int py=pys; py<=+pye; py++ ){
      for( int pz=pzs; pz<=+pze; pz++ ){
	const Position t = La*px + Lb*py + Lc*pz;

	glPushMatrix();
	glTranslated( t.x, t.y, t.z );

	if( model.gloption.lattice.show ){
	  lattice.draw();
	}
	  
	if( model.gloption.field.show ){
	  if( model.field.isset() ) field.draw();
	}
	  
	glPopMatrix();
      } // pz
    } // py
  } // px

}



void GLView::drawWigner( void )
{
  const Position& Lo = model.lattice.cell.Lo;

  // note: clip plane equation:
  // eq[0]*x + eq[1]*y + eq[2]*z + eq[3] > 0 : visible
  double eq[4]; // NbsO
  const double delta = +1.0/1024; // NbsO̗V

  // Wigner`掞͒PʖE̘g\Ȃ
  bool cell_frame_show_bak = model.gloption.lattice.cell.show;
  bool field_frame_show_bak = model.gloption.field.frame.show;
  model.gloption.lattice.cell.show = false;
  model.gloption.field.frame.show  = false;

  glPushMatrix();
  glTranslatedv( (-1.0)*Lo ); // PʖE̒Sւƌ_ύX

  // +-X,+-Y,+-Z8ӏɒPʖE`
  // ꂼ6ʂŃNbsOč邱ƂWigner`
  for( int c=0; c<8; c++ ){
    const Position Lt =
      - double(c/1%2) * model.lattice.cell.La
      - double(c/2%2) * model.lattice.cell.Lb
      - double(c/4%2) * model.lattice.cell.Lc;
    const Position La = model.lattice.cell.La * (c/1%2==0 ? 1.0 : -1.0);
    const Position Lb = model.lattice.cell.Lb * (c/2%2==0 ? 1.0 : -1.0);
    const Position Lc = model.lattice.cell.Lc * (c/4%2==0 ? 1.0 : -1.0);

    // 6{xNg La,Lb,Lc,La+Lb,Lb+Lc,Lc+La
    // {xNg̒_ʂAxNgɐȖʂ̕
    {
      const Position E = La;

      eq[0] = -E.x, eq[1] = -E.y, eq[2] = -E.z;
      eq[3] = E*(Lo+0.5*E) + delta;
      glClipPlane( GL_CLIP_PLANE0, eq );
      glEnable   ( GL_CLIP_PLANE0 );
    }
    {
      const Position E = Lb;

      eq[0] = -E.x, eq[1] = -E.y, eq[2] = -E.z;
      eq[3] = E*(Lo+0.5*E) + delta;
      glClipPlane( GL_CLIP_PLANE1, eq );
      glEnable   ( GL_CLIP_PLANE1 );
    }
    {
      const Position E = Lc;

      eq[0] = -E.x, eq[1] = -E.y, eq[2] = -E.z;
      eq[3] = E*(Lo+0.5*E) + delta;
      glClipPlane( GL_CLIP_PLANE2, eq );
      glEnable   ( GL_CLIP_PLANE2 );
    }
    {
      const Position E = La+Lb;

      eq[0] = -E.x, eq[1] = -E.y, eq[2] = -E.z;
      eq[3] = E*(Lo+0.5*E) + delta;
      glClipPlane( GL_CLIP_PLANE3, eq );
      glEnable   ( GL_CLIP_PLANE3 );
    }
    {
      const Position E = Lb+Lc;

      eq[0] = -E.x, eq[1] = -E.y, eq[2] = -E.z;
      eq[3] = E*(Lo+0.5*E) + delta;
      glClipPlane( GL_CLIP_PLANE4, eq );
      glEnable   ( GL_CLIP_PLANE4 );
    }
    {
      const Position E = Lc+La;

      eq[0] = -E.x, eq[1] = -E.y, eq[2] = -E.z;
      eq[3] = E*(Lo+0.5*E) + delta;
      glClipPlane( GL_CLIP_PLANE5, eq );
      glEnable   ( GL_CLIP_PLANE5 );
    }

    // eʓɓKȎlp``ANbsOɂĐؒf\
    {
      glDisable( GL_LIGHTING );
      glPolygonMode( GL_FRONT_AND_BACK, GL_LINE );

      glColor4dv( model.gloption.field.frame.color );
      {
	const Position E = La;
	const Position Ra = Lo + 0.5*E;
	const Position Rab = (Lb - (E*Lb)/(E*E)*E);
	const Position Rac = (Lc - (E*Lc)/(E*E)*E);
	const Position Rb = Ra + Rab;
	const Position Rc = Ra + Rac;
	const Position Rd = Ra + Rab + Rac;

	glBegin( GL_QUADS );
	glEdgeFlag( GL_FALSE );
	glVertex3dv(Ra);
	glEdgeFlag( GL_FALSE );
	glVertex3dv(Rb);
	glEdgeFlag( GL_TRUE );
	glVertex3dv(Rd);
	glEdgeFlag( GL_FALSE );
	glVertex3dv(Rc);
	glEnd();
      }
      {
	const Position E = Lb;
	const Position Rb = Lo + 0.5*E;
	const Position Rbc = (Lc - (E*Lc)/(E*E)*E);
	const Position Rba = (La - (E*La)/(E*E)*E);
	const Position Rc = Rb + Rbc;
	const Position Ra = Rb + Rba;
	const Position Rd = Rb + Rbc + Rba;

	glBegin( GL_QUADS );
	glEdgeFlag( GL_FALSE );
	glVertex3dv(Rb);
	glEdgeFlag( GL_FALSE );
	glVertex3dv(Rc);
	glEdgeFlag( GL_TRUE );
	glVertex3dv(Rd);
	glEdgeFlag( GL_FALSE );
	glVertex3dv(Ra);
	glEnd();
      }
      {
	const Position E = Lc;
	const Position Rc = Lo + 0.5*Lc;
	const Position Rca = La - (E*La)/(E*E)*E;
	const Position Rcb = Lb - (E*Lb)/(E*E)*E;
	const Position Ra = Rc + Rca;
	const Position Rb = Rc + Rcb;
	const Position Rd = Rc + Rca + Rcb;

	glBegin( GL_QUADS );
	glEdgeFlag( GL_FALSE );
	glVertex3dv(Rc);
	glEdgeFlag( GL_FALSE );
	glVertex3dv(Ra);
	glEdgeFlag( GL_TRUE );
	glVertex3dv(Rd);
	glEdgeFlag( GL_FALSE );
	glVertex3dv(Rb);
	glEnd();
      }

      {
	const Position E = La+Lb;
	const Position Ra  = Lo + 0.5*(E*E)/(E*La)* La;
	const Position Rb  = Lo + 0.5*(E*E)/(E*Lb)* Lb;
	const Position Rac = (Lc - (E*Lc)/(E*La)* La);
	const Position Rbd = (Lc - (E*Lc)/(E*Lb)* Lb);
	const Position Rc  = Ra + Rac;
	const Position Rd  = Rb + Rbd;
      
	glBegin( GL_QUADS );
	glEdgeFlag( GL_FALSE );
	glVertex3dv(Ra);
	glEdgeFlag( GL_FALSE );
	glVertex3dv(Rb);
	glEdgeFlag( GL_TRUE );
	glVertex3dv(Rd);
	glEdgeFlag( GL_TRUE );
	glVertex3dv(Rc);
	glEnd();
      }

      {
	const Position E = Lb+Lc;
	const Position Rb  = Lo + 0.5*(E*E)/(E*Lb)* Lb;
	const Position Rc  = Lo + 0.5*(E*E)/(E*Lc)* Lc;
	const Position Rba = (La - (E*La)/(E*Lb)* Lb);
	const Position Rcd = (La - (E*La)/(E*Lc)* Lc);
	const Position Ra = Rb + Rba;
	const Position Rd = Rc + Rcd;

	glBegin( GL_QUADS );
	glEdgeFlag( GL_FALSE );
	glVertex3dv(Rb);
	glEdgeFlag( GL_FALSE );
	glVertex3dv(Rc);
	glEdgeFlag( GL_TRUE );
	glVertex3dv(Rd);
	glEdgeFlag( GL_TRUE );
	glVertex3dv(Ra);
	glEnd();
      }

      {
	const Position E = Lc+La;
	const Position Rc  = Lo + 0.5*(E*E)/(E*Lc)* Lc;
	const Position Ra  = Lo + 0.5*(E*E)/(E*La)* La;
	const Position Rcb = (Lb - (E*Lb)/(E*Lc)* Lc);
	const Position Rad = (Lb - (E*Lb)/(E*La)* La);
	const Position Rb  = Rc + Rcb;
	const Position Rd  = Ra + Rad;
      
	glBegin( GL_QUADS );
	glEdgeFlag( GL_FALSE );
	glVertex3dv(Rc);
	glEdgeFlag( GL_FALSE );
	glVertex3dv(Ra);
	glEdgeFlag( GL_TRUE );
	glVertex3dv(Rd);
	glEdgeFlag( GL_TRUE );
	glVertex3dv(Rb);
	glEnd();
      }
      glPolygonMode( GL_FRONT_AND_BACK, GL_FILL );

      glEnable( GL_LIGHTING );
      glDisable( GL_BLEND );
    }

    glPushMatrix();

    // IɕsړĒPʖE`
    glTranslatedv( Lt );

    if( model.gloption.lattice.show ){
      lattice.draw();
    }
	  
    if( model.gloption.field.show ){
      if( model.field.isset() ) field.draw();
    }
	  
    glPopMatrix();
  }
  glPopMatrix();

  model.gloption.lattice.cell.show = cell_frame_show_bak;
  model.gloption.field.frame.show  = field_frame_show_bak;
}

bool GLView::save( void )
{
  static int cell=0;
  static char fname[256];

  snprintf( fname, sizeof(fname), "image%02d.bmp", cell++ );

  return glSavePixels( fname );
}

