/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

/*!
 \file imatrix.cc
 \brief 3x3s+3LxNgNX
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include "imatrix.h"

IMatrix::IMatrix( void )
{
  for( int i=0; i<3; i++ ){
    for( int j=0; j<3; j++ ){
      M[i][j] = (i==j) ? 1 : 0;
    }
    T[i] = 0;
  }
}

IMatrix::IMatrix( const int Mxx, const int Mxy, const int Mxz,
		  const int Myx, const int Myy, const int Myz,
		  const int Mzx, const int Mzy, const int Mzz )
{
  this->M[0][0] = Mxx, this->M[0][1] = Mxy, this->M[0][2] = Mxz;
  this->M[1][0] = Myx, this->M[1][1] = Myy, this->M[1][2] = Myz;
  this->M[2][0] = Mzx, this->M[2][1] = Mzy, this->M[2][2] = Mzz;
  this->T[0] = 0;
  this->T[1] = 0;
  this->T[2] = 0;
}

IMatrix::IMatrix( const int M[3][3], const fraction T[3] )
{
  for( int i=0; i<3; i++ ){
    for( int j=0; j<3; j++ ){
      this->M[i][j] = M[i][j];
    }
    this->T[i] = T[i];
  }
}

IMatrix::IMatrix( const char* symmetry )
{
  char sym[3][32];
  char axis[32];
  int n, m;

  if( false );
  else if( 3 == sscanf( symmetry,
		   "(%[-+abcxyzABCXYZ /0-9],"
		   "%[-+abcxyzABCXYZ /0-9],"
		   "%[-+abcxyzABCXYZ /0-9])", sym[0], sym[1], sym[2] )){
  }
  else if( 3 == sscanf( symmetry,
		   "%[-+abcxyzABCXYZ /0-9],"
		   "%[-+abcxyzABCXYZ /0-9],"
		   "%[-+abcxyzABCXYZ /0-9]", sym[0], sym[1], sym[2] )){
  }
  else{
    fprintf(stderr,"Error!: IMatrix::IMatrix(string) [%s]\n", symmetry );
    exit(0);
  }

  for( int i=0; i<3; i++ ){
    for( int j=0; j<3; j++ ){
      M[i][j] = 0;
    }
    T[i] = 0;
  }

  for( int c=0; c<3; c++ ){
    parse(sym[c],axis,n,m);

    if( strstr( axis, "+x" ) ){
      M[c][0] = +1;
    }else if( strstr( axis, "-x" ) ){
      M[c][0] = -1;
    }else{
      M[c][0] =  0;
    }

    if( strstr( axis, "+y" ) ){
      M[c][1] = +1;
    }else if( strstr( axis, "-y" ) ){
      M[c][1] = -1;
    }else{
      M[c][1] =  0;
    }

    if( strstr( axis, "+z" ) ){
      M[c][2] = +1;
    }else if( strstr( axis, "-z" ) ){
      M[c][2] = -1;
    }else{
      M[c][2] =  0;
    }

    if( n != 0 && m != 0 ){
      T[c] = fraction(n,m);
    }
    else{
      T[c] = 0;
    }
  }
}


bool IMatrix::operator == ( const IMatrix& that ) const
{
  for( int i=0; i<3; i++ ){
    for( int j=0; j<3; j++ ){
      if( this->M[i][j] != that.M[i][j] ) return false;
    }
    if( this->T[i] != that.T[i] ) return false;
  }

  return true;
}

bool IMatrix::operator != ( const IMatrix& that ) const
{
  for( int i=0; i<3; i++ ){
    for( int j=0; j<3; j++ ){
      if( this->M[i][j] != that.M[i][j] ) return true;
    }
    if( this->T[i] != that.T[i] ) return true;
  }

  return false;
}

IMatrix& IMatrix::operator += ( const IMatrix& that )
{
  for( int i=0; i<3; i++ ){
    for( int j=0; j<3; j++ ){
      this->M[i][j] += that.M[i][j];
    }
  }
  return *this;
}

IMatrix& IMatrix::operator *= ( const double d )
{
  for( int i=0; i<3; i++ ){
    for( int j=0; j<3; j++ ){
      this->M[i][j] = int((double)this->M[i][j]*d);
    }
  }
  return *this;
}

IMatrix IMatrix::operator * ( const IMatrix& that ) const
{
  IMatrix work;

  for( int i=0; i<3; i++ ){
    work.T[i] = 0;
    for( int j=0; j<3; j++ ){
      work.M[i][j] = 0;
      for( int k=0; k<3; k++ ){
	work.M[i][j] += this->M[i][k] * that.M[k][j];
      }
    }
    for( int k=0; k<3; k++ ){
      work.T[i] += this->M[i][k] * that.T[k];
    }
    work.T[i] += this->T[i];
    // sړ[0,1)ɋKi

    IMatrix::normalize(work.T[i]);
  }

  return work;
}


IMatrix IMatrix::getTransposed( void ) const
{
  IMatrix tra;

  for( int i=0; i<3; i++ ){
    for( int j=0; j<3; j++ ){
      tra.M[i][j] = this->M[j][i];
    }
  }

  tra.T[0] = 0;
  tra.T[1] = 0;
  tra.T[2] = 0;

  return tra;
}

IMatrix IMatrix::getTransInverse( void ) const
{
  IMatrix tin;

  const int det =
    + M[0][0] * M[1][1] * M[2][2]
    + M[0][1] * M[1][2] * M[2][0]
    + M[0][2] * M[1][0] * M[2][1]
    - M[0][2] * M[1][1] * M[2][0]
    - M[0][0] * M[1][2] * M[2][1]
    - M[0][1] * M[1][0] * M[2][2];
  const fraction idet = fraction( 1, det );

  tin.M[0][0] = ((M[1][1]*M[2][2] - M[1][2]*M[2][1])*idet).toInteger();
  tin.M[1][0] = ((M[2][1]*M[0][2] - M[2][2]*M[0][1])*idet).toInteger();
  tin.M[2][0] = ((M[0][1]*M[1][2] - M[0][2]*M[1][1])*idet).toInteger();
  tin.M[0][1] = ((M[1][2]*M[2][0] - M[1][0]*M[2][2])*idet).toInteger();
  tin.M[1][1] = ((M[2][2]*M[0][0] - M[2][0]*M[0][2])*idet).toInteger();
  tin.M[2][1] = ((M[0][2]*M[1][0] - M[0][0]*M[1][2])*idet).toInteger();
  tin.M[0][2] = ((M[1][0]*M[2][1] - M[1][1]*M[2][0])*idet).toInteger();
  tin.M[1][2] = ((M[2][0]*M[0][1] - M[2][1]*M[0][0])*idet).toInteger();
  tin.M[2][2] = ((M[0][0]*M[1][1] - M[0][1]*M[1][0])*idet).toInteger();

  tin.T[0] = T[0];
  tin.T[1] = T[1];
  tin.T[2] = T[2];

  return tin;
}

Coordinates IMatrix::operator * ( const Coordinates& p ) const
{
  Coordinates Tp;

  for( int i=0; i<3; i++ ){
    Tp[i] = 0.0;
    for( int j=0; j<3; j++ ){
      Tp[i] += this->M[i][j] * p[j];
    }
    Tp[i] += this->T[i].toDouble();
  }

  // sړ[0,1)ɋKi
  //  Position::normalize(Tp);

  return Tp;
}

/*
Matrix IMatrix::getMatrix( void ) const
{
  Matrix mat;

  for( int i=0; i<3; i++ ){
    for( int j=0; j<3; j++ ){
      mat.M[i][j] = (double)this->M[i][j];
    }
  }

  return mat;
}
*/

IMatrix IMatrix::getIdentity( void )
{
  IMatrix mat;

  for( int i=0; i<3; i++ ){
    for( int j=0; j<3; j++ ){
      mat.M[i][j] = (i==j) ? 1 : 0;
    }
    mat.T[i] = 0;
  }

  return mat;
}

IMatrix IMatrix::getConstraint( void ) const
{
  const IMatrix I = IMatrix::getIdentity();

  int n;
  IMatrix C = I;
  const IMatrix M = *this;
  IMatrix Mn = M;
  for( n=1; Mn!=I; Mn=Mn*M, n++ ){
    if( n>10 ){
      fprintf(stderr,"Error in getConstraint\n");
      exit(0);
    }
    C += Mn;
  }
  if( n>1 ){
    C *= 1.0/n;
  }

  return C;
}


int IMatrix::determinant( void ) const
{
  return
    + M[0][0] * (M[1][1] * M[2][2] - M[2][1] * M[1][2] )
    - M[1][0] * (M[0][1] * M[2][2] - M[2][1] * M[0][2] )
    + M[2][0] * (M[0][1] * M[1][2] - M[1][1] * M[0][2] );
}

void IMatrix::initCanditate( void )
{
  for( int i=0; i<3; i++ ){
    T[i] = 0;
  }
  for( int i=0; i<3; i++ ){
    for( int j=0; j<3; j++ ){
      if( i==j ){
	M[i][j] = 1;
      }
      else{
	M[i][j] = 0;
      }
    }
  }
}

bool IMatrix::nextCanditate( void )
{
  for( int i=0; i<3; i++ ){
    T[i] = 0;
  }
  for( int i=0; i<3; i++ ){
    for( int j=0; j<3; j++ ){
      if( i==j ){
	if( false );
	else if( M[i][j] == +1 ){
	  M[i][j] =  0;
	  return true;
	}
	else if( M[i][j] ==  0 ){
	  M[i][j] = -1;
	  return true;
	}
	else if( M[i][j] == -1 ){
	  M[i][j] = +1;
	  // to the next iteration
	}
      }
      else{
	if( false );
	else if( M[i][j] ==  0 ){
	  M[i][j] = -1;
	  return true;
	}
	else if( M[i][j] == -1 ){
	  M[i][j] = +1;
	  return true;
	}
	else if( M[i][j] == +1 ){
	  M[i][j] =  0;
	  // to the next iteration
	}
      }
    }
  }
  return false; // end of the candidates
}


void IMatrix::transformUnitary
( double matrix_transformed[3][3],
  const double matrix[3][3] ) const
{
  double matrix_work[3][3];

  for( int i=0; i<3; i++ ){
    for( int j=0; j<3; j++ ){
      matrix_work[i][j] = 0.0;
      for( int k=0; k<3; k++ ){
	matrix_work[i][j] += matrix[i][k] * M[k][j];
      }
    }
  }

  for( int i=0; i<3; i++ ){
    for( int j=0; j<3; j++ ){
      matrix_transformed[i][j] = 0.0;
      for( int l=0; l<3; l++ ){
	matrix_transformed[i][j] += M[l][i] * matrix_work[l][j];
      }
    }
  }
}

char* IMatrix::name( void ) const
{
  static char symmetry[36];
  char sym[3][16];
  char work[8];

  for( int c=0; c<3; c++ ){
    strcpy( sym[c], "" );
    if( M[c][0] == +1 ){
      strcat( sym[c], "+a" );
    }else if( M[c][0] == -1 ){
      strcat( sym[c], "-a" );
    }else if( M[c][0] == 0 ){
    }else{
      sprintf( work, "%+da", M[c][0] );
      strcat( sym[c], work );
    }
    if( M[c][1] == +1 ){
      strcat( sym[c], "+b" );
    }else if( M[c][1] == -1 ){
      strcat( sym[c], "-b" );
    }else if( M[c][1] == 0 ){
    }else{
      sprintf( work, "%+db", M[c][1] );
      strcat( sym[c], work );
    }
    if( M[c][2] == +1 ){
      strcat( sym[c], "+c" );
    }else if( M[c][2] == -1 ){
      strcat( sym[c], "-c" );
    }else if( M[c][2] == 0 ){
    }else{
      sprintf( work, "%+dc", M[c][2] );
      strcat( sym[c], work );
    }

    if( T[c].numerator != 0 ){
      sprintf( work, "%+d/%d", T[c].numerator, T[c].denominator );
      strcat( sym[c], work );
    }
  }

  sprintf( symmetry, "(%s,%s,%s)", sym[0], sym[1], sym[2] );

  return symmetry;
}


int IMatrix::parse( char* str, char* axis, int& n, int& m )
{
  char work[32];

  // erase whitespace.
  // add '+' in case not specified explicitly
  int i, j;
  for( i=0,j=0; str[i]; i++ ){
    if( str[i] != ' ' ){
      if( j==0 && !(str[i]=='+' || str[i]=='-') ){
	work[j++] = '+';
      }
      work[j++] = str[i];
    }
  }
  work[j++] = 0;

  // determine axis
  strcpy( axis, "" );
  if( strstr( work, "+x" ) || strstr( work, "+X" ) ||
      strstr( work, "+a" ) || strstr( work, "+A" ) ){
    strcat( axis, "+x" );
  }
  else if( strstr( work, "-x" ) || strstr( work, "-X" ) ||
	   strstr( work, "-a" ) || strstr( work, "-A" ) ){
    strcat( axis, "-x" );
  }

  if( strstr( work, "+y" ) || strstr( work, "+Y" ) ||
      strstr( work, "+b" ) || strstr( work, "+B" ) ){
    strcat( axis, "+y" );
  }
  else if( strstr( work, "-y" ) || strstr( work, "-Y" ) ||
	   strstr( work, "-b" ) || strstr( work, "-B" ) ){
    strcat( axis, "-y" );
  }

  if( strstr( work, "+z" ) || strstr( work, "+Z" ) ||
      strstr( work, "+c" ) || strstr( work, "+C" ) ){
    strcat( axis, "+z" );
  }
  else if( strstr( work, "-z" ) || strstr( work, "-Z" ) ||
	   strstr( work, "-c" ) || strstr( work, "-C" ) ){
    strcat( axis, "-z" );
  }

  if( false );
  else if( strstr( work, "+1/2" ) ){
    n=1, m=2;
  }
  else if( strstr( work, "+1/3" ) ){
    n=1, m=3;
  }
  else if( strstr( work, "+2/3" ) ){
    n=2, m=3;
  }
  else if( strstr( work, "+1/4" ) ){
    n=1, m=4;
  }
  else if( strstr( work, "+2/4" ) ){
    n=2, m=4;
  }
  else if( strstr( work, "+3/4" ) ){
    n=3, m=4;
  }
  else if( strstr( work, "+1/5" ) ){
    n=1, m=5;
  }
  else if( strstr( work, "+2/5" ) ){
    n=2, m=5;
  }
  else if( strstr( work, "+3/5" ) ){
    n=3, m=5;
  }
  else if( strstr( work, "+4/5" ) ){
    n=4, m=5;
  }
  else if( strstr( work, "+1/6" ) ){
    n=1, m=6;
  }
  else if( strstr( work, "+2/6" ) ){
    n=2, m=6;
  }
  else if( strstr( work, "+3/6" ) ){
    n=3, m=6;
  }
  else if( strstr( work, "+4/6" ) ){
    n=4, m=6;
  }
  else if( strstr( work, "+5/6" ) ){
    n=5, m=6;
  }
  else{
    n=0, m=0;
  }
  return true;
}

void IMatrix::normalize( fraction& f )
{
  while( f.numerator >= f.denominator ){
    f.numerator -= f.denominator;
  }
  while( f.numerator < 0 ){
    f.numerator += f.denominator;
  }
}
