/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

#include <stdio.h>
#include <math.h>
#include <algorithm>
#include "vector3d.h"
#include "position.h"

struct Cell
{
  Position La, Lb, Lc;
  double   V;
  Position Ka, Kb, Kc;
  int      Nka, Nkb, Nkc;
  Position dKa, dKb, dKc;
  Position Ko;

  double fKoa, fKob, fKoc;
  double dfKa, dfKb, dfKc;

  int indexA( double fka ) const {
    return int((fka-fKoa)/dfKa);
  }
  int indexB( double fkb ) const {
    return int((fkb-fKob)/dfKb);
  }
  int indexC( double fkc ) const {
    return int((fkc-fKoc)/dfKc);
  }
  Position K( int ika, int ikb, int ikc ) const {
    return Ko + dKa*ika + dKb*ikb + dKc*ikc;
  }
  Position fK( int ika, int ikb, int ikc ) const {
    double fka = ika*dfKa + fKoa;
    double fkb = ikb*dfKb + fKob;
    double fkc = ikc*dfKc + fKoc;

    return Position(fka,fkb,fkc);
  }

};

struct Sampling
{
  vector<double> venergy;
};

Cell cell;
vector3d<Sampling> vsample;
int nsample, nenergy;

FILE* fptr_in;
FILE* fptr_out;

void sortuniq( vector<double>& v )
{
  sort( v.begin(), v.end() );
  vector<double>::iterator end = unique( v.begin(), v.end() );
  v.erase( end, v.end() );
}
bool absless( const double& a, const double& b ){
  return fabs(a) < fabs(b);
}

void readCell( void )
{
  char buf[1024];
  double L;

  rewind(fptr_in);
  while( fgets(buf,sizeof(buf),fptr_in) ){
    if( strncmp(buf,"### CELL INFORMATION ###",
		sizeof("### CELL INFORMATION ###")-1) != 0 ) continue;

    fgets(buf,sizeof(buf),fptr_in); // L
    sscanf(buf,"%lf",&L);
    fgets(buf,sizeof(buf),fptr_in); // aa(:,1)
    sscanf(buf,"%lf %lf %lf", &cell.La.x, &cell.La.y, &cell.La.z );
    fgets(buf,sizeof(buf),fptr_in); // aa(:,2)
    sscanf(buf,"%lf %lf %lf", &cell.Lb.x, &cell.Lb.y, &cell.Lb.z );
    fgets(buf,sizeof(buf),fptr_in); // aa(:,3)
    sscanf(buf,"%lf %lf %lf", &cell.Lc.x, &cell.Lc.y, &cell.Lc.z );

    cell.La *= L;
    cell.Lb *= L;
    cell.Lc *= L;
    break;
  }

  cell.V  = cell.La * (cell.Lb % cell.Lc);
  cell.Ka = (cell.Lb % cell.Lc) * (2.0*M_PI/cell.V);
  cell.Kb = (cell.Lc % cell.La) * (2.0*M_PI/cell.V);
  cell.Kc = (cell.La % cell.Lb) * (2.0*M_PI/cell.V);

  printf("L\n");
  printf("%f %f %f\n", cell.La.x, cell.La.y, cell.La.z );
  printf("%f %f %f\n", cell.Lb.x, cell.Lb.y, cell.Lb.z );
  printf("%f %f %f\n", cell.Lc.x, cell.Lc.y, cell.Lc.z );
  printf("V\n%f\n", cell.V );
  printf("K\n");
  printf("%f %f %f\n", cell.Ka.x, cell.Ka.y, cell.Ka.z );
  printf("%f %f %f\n", cell.Kb.x, cell.Kb.y, cell.Kb.z );
  printf("%f %f %f\n", cell.Kc.x, cell.Kc.y, cell.Kc.z );
}

void readKvec( void )
{
  char buf[1024];
  double fka, fkb, fkc;
  double fka_step, fkb_step, fkc_step;
  double fka_max, fkb_max, fkc_max;

  vector<double> vfka, vfkb, vfkc;

  rewind(fptr_in);
  while( fgets(buf,sizeof(buf),fptr_in) ){
    if( 3 != sscanf( buf, "%*d-TH SK=  %lf %lf %lf", &fka, &fkb, &fkc ) ) continue;
    vfka.push_back(fka);
    vfkb.push_back(fkb);
    vfkc.push_back(fkc);
  }

  sortuniq( vfka );
  sortuniq( vfkb );
  sortuniq( vfkc );

  printf("fka\n");
  for( int i=0; i<vfka.size(); i++ ){
    printf("%f\n", vfka[i] );
  }
  printf("fkb\n");
  for( int i=0; i<vfkb.size(); i++ ){
    printf("%f\n", vfkb[i] );
  }
  printf("fkc\n");
  for( int i=0; i<vfkc.size(); i++ ){
    printf("%f\n", vfkc[i] );
  }

  vector<double> vstep;

  vstep.clear();
  for( int i=0; i+1<vfka.size(); i++ ){
    vstep.push_back( vfka[i+1] - vfka[i] );
  }
  fka_step = *min_element(vstep.begin(),vstep.end());

  for( int n=1; n<=5; n++ ){
    bool ok = true;
    double f = fka_step/n;
    for( int i=0; i<vstep.size(); i++ ){
      if( fmod( vstep[i]/f, 1.0 ) != 0.0 ){
	ok = false;
	break;
      }
    }
    if(ok){
      fka_step = f;
      break;
    }
  }


  vstep.clear();
  for( int i=0; i+1<vfkb.size(); i++ ){
    vstep.push_back( vfkb[i+1] - vfkb[i] );
  }
  fkb_step = *min_element(vstep.begin(),vstep.end());


  for( int n=1; n<=5; n++ ){
    bool ok = true;
    double f = fkb_step/n;
    for( int i=0; i<vstep.size(); i++ ){
      if( fmod( vstep[i]/f, 1.0 ) != 0.0 ){
	ok = false;
	break;
      }
    }
    if(ok){
      fkb_step = f;
      break;
    }
  }

  vstep.clear();
  for( int i=0; i+1<vfkc.size(); i++ ){
    vstep.push_back( vfkc[i+1] - vfkc[i] );
  }
  fkc_step = *min_element(vstep.begin(),vstep.end());

  for( int n=1; n<=5; n++ ){
    bool ok = true;
    double f = fkc_step/n;
    for( int i=0; i<vstep.size(); i++ ){
      if( fmod( vstep[i]/f, 1.0 ) != 0.0 ){
	ok = false;
	break;
      }
    }
    if(ok){
      fkc_step = f;
      break;
    }
  }


  printf("fk_step\n");
  printf("%f %f %f\n", fka_step, fkb_step, fkc_step );

  fka_max = fabs(*max_element(vfka.begin(),vfka.end(),absless));
  fkb_max = fabs(*max_element(vfkb.begin(),vfkb.end(),absless));
  fkc_max = fabs(*max_element(vfkc.begin(),vfkc.end(),absless));

  printf("fk_max\n");
  printf("%f %f %f\n", fka_max, fkb_max, fkc_max );

  cell.Nka = int(2.0*fka_max/fka_step)+1;
  cell.Nkb = int(2.0*fkb_max/fkb_step)+1;
  cell.Nkc = int(2.0*fkc_max/fkc_step)+1;

  cell.dKa = cell.Ka * fka_step;
  cell.dKb = cell.Kb * fkb_step;
  cell.dKc = cell.Kc * fkc_step;

  cell.Ko  =
    + cell.dKa * (-(cell.Nka-1)/2)
    + cell.dKb * (-(cell.Nkb-1)/2)
    + cell.dKc * (-(cell.Nkc-1)/2);

  printf("Ko\n");
  printf("%3d %+f %+f %+f\n", 0, cell.Ko.x, cell.Ko.y, cell.Ko.z );
  printf("dKa\n");
  printf("%3d %+f %+f %+f\n", cell.Nka, cell.dKa.x, cell.dKa.y, cell.dKa.z );
  printf("dKb\n");
  printf("%3d %+f %+f %+f\n", cell.Nkb, cell.dKb.x, cell.dKb.y, cell.dKb.z );
  printf("dKc\n");
  printf("%3d %+f %+f %+f\n", cell.Nkc, cell.dKc.x, cell.dKc.y, cell.dKc.z );


  cell.fKoa = -fka_max;
  cell.dfKa =  fka_step;
  cell.fKob = -fkb_max;
  cell.dfKb =  fkb_step;
  cell.fKoc = -fkc_max;
  cell.dfKc =  fkc_step;

  for( int i=0; i<vfka.size(); i++ ){
    printf("fka %d: %d\n", i, cell.indexA(vfka[i]) );
  }
  for( int i=0; i<vfkb.size(); i++ ){
    printf("fkb %d: %d\n", i, cell.indexB(vfkb[i]) );
  }
  for( int i=0; i<vfkc.size(); i++ ){
    printf("fkc %d: %d\n", i, cell.indexC(vfkc[i]) );
  }
}

void readEnergy( void )
{
  char buf[1024];
  double fka, fkb, fkc;

  vsample.resize(cell.Nka,cell.Nkb,cell.Nkc);

  rewind(fptr_in);
  while( fgets(buf,sizeof(buf),fptr_in) ){
    if( 3 != sscanf( buf, "%*d-TH SK=  %lf %lf %lf", &fka, &fkb, &fkc ) ) continue;

    int ika = cell.indexA(fka);
    int ikb = cell.indexB(fkb);
    int ikc = cell.indexC(fkc);

    int line;
    double v[5+1];
    while( fgets(buf,sizeof(buf),fptr_in) ){
      int status = sscanf(buf,"%d %lf %lf %lf %lf %lf",
			  &line, &v[1], &v[2], &v[3], &v[4], &v[5] );
      if( status < 2 ) break;

      for( int i=1; i<status; i++ ){
	vsample(ika,ikb,ikc).venergy.push_back(v[i]);
      }
    }
    if( vsample(ika,ikb,ikc).venergy.size()>0 ){
      nenergy = vsample(ika,ikb,ikc).venergy.size();
    }
  }

  nsample = vsample.size();
}

void writeDX( void )
{
  fprintf( fptr_out, "object 1 class gridpositions counts %d %d %d\n",
	   cell.Nka, cell.Nkb, cell.Nkc );
  fprintf( fptr_out, "origin   %+f %+f %+f\n",
	   cell.Ko.x, cell.Ko.y, cell.Ko.z );
  fprintf( fptr_out, "delta    %+f %+f %+f\n",
	   cell.dKa.x, cell.dKa.y, cell.dKa.z );
  fprintf( fptr_out, "delta    %+f %+f %+f\n",
	   cell.dKb.x, cell.dKb.y, cell.dKb.z );
  fprintf( fptr_out, "delta    %+f %+f %+f\n",
	   cell.dKc.x, cell.dKc.y, cell.dKc.z );
  fprintf( fptr_out, "object 2 class gridconnections counts %d %d %d\n",
	   cell.Nka, cell.Nkb, cell.Nkc );


  for( int m=0; m<nenergy; m++ ){
    fprintf( fptr_out, "object %d class array type double rank 0\n", m+3 );
    fprintf( fptr_out, "items %d data follows\n", 
	     cell.Nka*cell.Nkb*cell.Nkc );
    for( int i=0; i<nsample; i++ ){
      if( m<vsample(i).venergy.size() ){
	fprintf( fptr_out, "%f\n", vsample(i).venergy[m] );
      }
      else{
	fprintf( fptr_out, "%f\n", 0.0 );
      }
    }
    fprintf( fptr_out, "attribute \"dep\" string \"positions\"\n");
  }

  fprintf( fptr_out, "object \"TAPP band\" class field\n");
  fprintf( fptr_out, "component \"positions\" value 1\n");
  fprintf( fptr_out, "component \"connections\" value 2\n");
  for( int m=0; m<nenergy; m++ ){
    fprintf( fptr_out, "component \"band%d\" value %d\n", m, m+3);
  }
  fprintf( fptr_out, "end\n");

  fclose(fptr_out);
}

int main( int argc, char* argv[] )
{
  fptr_in  = fopen(argv[1],"r");
  fptr_out = fopen(argv[2],"w");

  readCell();
  readKvec();
  readEnergy();
  writeDX();

}
