/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

/*!
 \file matrix.cc
 \brief 3x3sNX
*/

#include "matrix.h"

Matrix::Matrix( void )
{
  for( int i=0; i<3; i++ ){
    for( int j=0; j<3; j++ ){
      M[i][j] = (i==j) ? 1.0 : 0.0;
    }
  }
}

Matrix::Matrix( const Position M[3] )
{
  for( int i=0; i<3; i++ ){
    for( int j=0; j<3; j++ ){
      this->M[i][j] = M[j][i];
    }
  }
}

Matrix::Matrix
( const double Mxx, const double Mxy, const double Mxz,
  const double Myx, const double Myy, const double Myz,
  const double Mzx, const double Mzy, const double Mzz )
{
  this->M[0][0] = Mxx, this->M[0][1] = Mxy, this->M[0][2] = Mxz;
  this->M[1][0] = Myx, this->M[1][1] = Myy, this->M[1][2] = Myz;
  this->M[2][0] = Mzx, this->M[2][1] = Mzy, this->M[2][2] = Mzz;
}

bool Matrix::operator == ( const Matrix& that ) const
{
  for( int i=0; i<3; i++ ){
    for( int j=0; j<3; j++ ){
      if( this->M[i][j] != that.M[i][j] ) return false;
    }
  }

  return true;
}

bool Matrix::operator != ( const Matrix& that ) const
{
  for( int i=0; i<3; i++ ){
    for( int j=0; j<3; j++ ){
      if( this->M[i][j] != that.M[i][j] ) return true;
    }
  }

  return false;
}

Matrix& Matrix::operator += ( const Matrix& that )
{
  for( int i=0; i<3; i++ ){
    for( int j=0; j<3; j++ ){
      this->M[i][j] += that.M[i][j];
    }
  }
  return *this;
}

Matrix& Matrix::operator *= ( const double d )
{
  for( int i=0; i<3; i++ ){
    for( int j=0; j<3; j++ ){
      this->M[i][j] *= d;
    }
  }
  return *this;
}


Matrix Matrix::operator * ( const Matrix& that ) const
{
  Matrix work;

  for( int i=0; i<3; i++ ){
    for( int j=0; j<3; j++ ){
      work.M[i][j] = 0;
      for( int k=0; k<3; k++ ){
	work.M[i][j] += this->M[i][k] * that.M[k][j];
      }
    }
  }

  return work;
}

Position Matrix::operator * ( const Position& p ) const
{
  Position Tp;
  for( int i=0; i<3; i++ ){
    Tp[i] = 0.0;
    for( int j=0; j<3; j++ ){
      Tp[i] += this->M[i][j] * p[j];
    }
  }
  return Tp;
}


// tsԂ֐B
Matrix Matrix::getInverse( void ) const
{
  Matrix inv;

  const double det =
    + M[0][0] * M[1][1] * M[2][2]
    + M[0][1] * M[1][2] * M[2][0]
    + M[0][2] * M[1][0] * M[2][1]
    - M[0][2] * M[1][1] * M[2][0]
    - M[0][0] * M[1][2] * M[2][1]
    - M[0][1] * M[1][0] * M[2][2];
  const double idet = 1.0/det;

  inv.M[0][0] = (M[1][1] * M[2][2] - M[1][2] * M[2][1])*idet;
  inv.M[0][1] = (M[2][1] * M[0][2] - M[2][2] * M[0][1])*idet;
  inv.M[0][2] = (M[0][1] * M[1][2] - M[0][2] * M[1][1])*idet;
  inv.M[1][0] = (M[1][2] * M[2][0] - M[1][0] * M[2][2])*idet;
  inv.M[1][1] = (M[2][2] * M[0][0] - M[2][0] * M[0][2])*idet;
  inv.M[1][2] = (M[0][2] * M[1][0] - M[0][0] * M[1][2])*idet;
  inv.M[2][0] = (M[1][0] * M[2][1] - M[1][1] * M[2][0])*idet;
  inv.M[2][1] = (M[2][0] * M[0][1] - M[2][1] * M[0][0])*idet;
  inv.M[2][2] = (M[0][0] * M[1][1] - M[0][1] * M[1][0])*idet;

  return inv;
}

const Matrix& Matrix::getIdentity( void )
{
  static Matrix mat;
  static bool first=true;

  if( first ){
    for( int i=0; i<3; i++ ){
      for( int j=0; j<3; j++ ){
	mat.M[i][j] = (i==j) ? 1.0 : 0.0;
      }
    }
    first=false;
  }
  return mat;
}

const Matrix& Matrix::getZero( void )
{
  static Matrix mat;
  static bool first=true;

  if( first ){
    for( int i=0; i<3; i++ ){
      for( int j=0; j<3; j++ ){
	mat.M[i][j] = 0.0;
      }
    }
    first=false;
  }
  return mat;
}

