/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

/*!
 \file position.h
 \brief 3WNX
*/

#ifndef __POSITION_H_INCLUDED
#define __POSITION_H_INCLUDED

#include <math.h>

// 3W̍\
struct Position
{
  union { double x, a; };
  union { double y, b; };
  union { double z, c; };

  inline Position( void ){
  }
  inline Position( const double& _x, const double& _y, const double& _z ) : 
  x(_x), y(_y), z(_z) {
  }

  operator const double* () const {
    return &x;
  }
  operator double* ()  {
    return &x;
  }
  inline double&  operator () ( const int i ){
    return (&x)[i];
  }
  inline const double&  operator () ( const int i ) const {
    return (&x)[i];
  }
  inline double&  operator [] ( const int i ){
    return (&x)[i];
  }
  inline const double&  operator [] ( const int i ) const {
    return (&x)[i];
  }

  inline const Position& operator + ( void ) const {
    return *this;
  }
  inline const Position operator - ( void ) const {
    return Position(-x,-y,-z);
  }

  inline Position& operator *= ( const double& d ){
    x *= d; y *= d; z *= d;
    return *this;
  }
  inline Position& operator += ( const Position& v ){
    x += v.x; y += v.y; z += v.z;
    return *this;
  }
  inline Position& operator -= ( const Position& v ){
    x -= v.x; y -= v.y; z -= v.z;
    return *this;
  }
  bool operator == ( const Position& p ) const {
    return x == p.x && y == p.y && z == p.z;
  }
  bool operator != ( const Position& p ) const {
    return x != p.x || y != p.y || z != p.z;
  }
  static bool neareq( const Position& p1, const Position& p2 ) {
    const double dx=fabs(p1.x-p2.x);
    if(dx>1e-4) return false;
    const double dy=fabs(p1.y-p2.y);
    if(dy>1e-4) return false;
    const double dz=fabs(p1.z-p2.z);
    if(dz>1e-4) return false;
    return true;
  }

  bool operator <  ( const Position& p ) const {
    return
      (x<p.x) || (x==p.x && y<p.y) || (x==p.x && y==p.y && z<p.z );
  }

  inline Position normal( void ) const {
    const double sn = sqrt(norm());
    const double inorm = sn == 0.0 ? 0.0 : 1.0/sn;
    return Position( x*inorm, y*inorm, z*inorm );
  }

  inline double norm( void ) const {
    return x*x + y*y + z*z;
  }

  static inline bool match( const Position& p1, const Position& p2 ){
    const double eps=1.0e-5;
    return
      fabs(p1[0]-p2[0])<eps &&
      fabs(p1[1]-p2[1])<eps &&
      fabs(p1[2]-p2[2])<eps;
  }
  static inline bool match( const Position& p1, const Position& p2, const double eps ){
    return
      fabs(p1[0]-p2[0])<eps &&
      fabs(p1[1]-p2[1])<eps &&
      fabs(p1[2]-p2[2])<eps;
  }

  static inline Position outer_product( const Position& p1, const Position& p2 ){
    Position p;
    p.x = p1.y*p2.z - p2.y*p1.z;
    p.y = p1.z*p2.x - p2.z*p1.x;
    p.z = p1.x*p2.y - p2.x*p1.y;
    return p;
  }

  static inline double inner_product( const Position& p1, const Position& p2 ){
    return p1.x*p2.x + p1.y*p2.y + p1.z*p2.z;
  }

  // Position^̒2vZ֐
  static inline double length2( const Position& p ){
    return p.x*p.x + p.y*p.y + p.z*p.z;
  }
  // Position^̒vZ֐
  static inline double length( const Position& p ){
    return sqrt( p.x*p.x + p.y*p.y + p.z*p.z );
  }
  // Position^̋2vZ֐
  static inline double length2( const Position& p1, const Position& p2 ){
    const double dx = p1.x - p2.x;
    const double dy = p1.y - p2.y;
    const double dz = p1.z - p2.z;
    return dx*dx + dy*dy + dz*dz;
  }
  static inline double angle( const Position& p1, const Position& p2 ){
    const double len1 = length(p1);
    const double len2 = length(p2);
    const double prod = inner_product(p1,p2);

    return acos(prod/(len1*len2))*180.0/M_PI;
  }

  static inline Position translate( const double vecs[3][3], const Position& p_org )
  {
    Position p_new;
    for( int i=0; i<3; i++ ){
      p_new[i] = 0.0;
      for( int j=0; j<3; j++ ){
	p_new[i] += vecs[j][i]*p_org[j];
      }
    }
    return p_new;
  }

  static inline Position translate( const Position vecs[3], const double p_org[3] )
  {
    Position p_new;
    for( int i=0; i<3; i++ ){
      p_new[i] = 0.0;
      for( int j=0; j<3; j++ ){
	p_new[i] += vecs[j][i]*p_org[j];
      }
    }
    return p_new;
  }

  static inline Position translate( const Position vecs[3], const Position& p_org )
  {
    Position p_new;
    for( int i=0; i<3; i++ ){
      p_new[i] = 0.0;
      for( int j=0; j<3; j++ ){
	p_new[i] += vecs[j][i]*p_org[j];
      }
    }
    return p_new;
  }

  static inline void fold( Position& p )
  {
    for( int i=0; i<3; i++ ){
      while( p[i]<0.0 ){
	p[i]+=1.0;
      }
      while( p[i]>=1.0 ){
	p[i]-=1.0;
      }
    }
  }



};

// 3W̉ZK
inline Position operator + ( const Position& v1, const Position& v2 )
{
  Position t;
  t.x = v1.x + v2.x;
  t.y = v1.y + v2.y;
  t.z = v1.z + v2.z;
  return t;
}

inline Position operator - ( const Position& v1, const Position& v2 )
{
  Position t;
  t.x = v1.x - v2.x;
  t.y = v1.y - v2.y;
  t.z = v1.z - v2.z;
  return t;
}

inline Position operator * ( const Position& v, const double& d )
{
  Position t;
  t.x = v.x*d;
  t.y = v.y*d;
  t.z = v.z*d;
  return t;
}

inline Position operator * ( const double& d, const Position& v )
{
  Position t;
  t.x = v.x*d;
  t.y = v.y*d;
  t.z = v.z*d;
  return t;
}

inline Position operator % ( const Position& v1, const Position& v2 )
{
  Position t;
  t.x = v1.y*v2.z - v2.y*v1.z;
  t.y = v1.z*v2.x - v2.z*v1.x;
  t.z = v1.x*v2.y - v2.x*v1.y;
  return t;
}

inline double operator * ( const Position& v1, const Position& v2 )
{
  return v1.x*v2.x + v1.y*v2.y + v1.z*v2.z;
}


struct Coordinates : public Position
{
  Coordinates( void ) : Position() {
  }
  Coordinates( const double a, const double b, const double c ) :
    Position(a,b,c) {
  }
  Coordinates( const Position& p ) :
    Position(p.a,p.b,p.c) {
  }
  /*
  operator Position(){
    return Position(a,b,c);
  }
  */

  static inline Coordinates& normalize( Coordinates& p ){
    if( p.a >= 1.0 ) p.a = fmod( p.a, 1.0 );
    else if( p.a < 0.0 ) p.a = 1.0 + fmod( p.a, 1.0 );
    if( p.b >= 1.0 ) p.b = fmod( p.b, 1.0 );
    else if( p.b < 0.0 ) p.b = 1.0 + fmod( p.b, 1.0 );
    if( p.c >= 1.0 ) p.c = fmod( p.c, 1.0 );
    else if( p.c < 0.0 ) p.c = 1.0 + fmod( p.c, 1.0 );

    const double eps = 1.0e-8;
    if( p.a>=1.0-eps ) p.a = 0.0;
    if( p.b>=1.0-eps ) p.b = 0.0;
    if( p.c>=1.0-eps ) p.c = 0.0;

    return p;
  }
  static inline Coordinates normalize( const Coordinates& q ){
    Coordinates p=q;
    if( p.a >= 1.0 ) p.a = fmod( p.a, 1.0 );
    else if( p.a < 0.0 ) p.a = 1.0 + fmod( p.a, 1.0 );
    if( p.b >= 1.0 ) p.b = fmod( p.b, 1.0 );
    else if( p.b < 0.0 ) p.b = 1.0 + fmod( p.b, 1.0 );
    if( p.c >= 1.0 ) p.c = fmod( p.c, 1.0 );
    else if( p.c < 0.0 ) p.c = 1.0 + fmod( p.c, 1.0 );
    return p;
  }

  static inline Coordinates& normalize5( Coordinates& p ){
    while( p.a>=+0.5 ) p.a -= 1.0;
    while( p.a< -0.5 ) p.a += 1.0;
    while( p.b>=+0.5 ) p.b -= 1.0;
    while( p.b< -0.5 ) p.b += 1.0;
    while( p.c>=+0.5 ) p.c -= 1.0;
    while( p.c< -0.5 ) p.c += 1.0;
    return p;
  }

  static inline Coordinates normalize5( const Coordinates& q ){
    Coordinates p=q;
    while( p.a>=+0.5 ) p.a -= 1.0;
    while( p.a< -0.5 ) p.a += 1.0;
    while( p.b>=+0.5 ) p.b -= 1.0;
    while( p.b< -0.5 ) p.b += 1.0;
    while( p.c>=+0.5 ) p.c -= 1.0;
    while( p.c< -0.5 ) p.c += 1.0;
    return p;
  }

};

#endif // __POSITION_H_INCLUDED
