/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

/*!
 \file qtatoms.cc
 \brief iq̌qzuGUI\̃NX
*/

#include "dtmodel.h"
#include "qtatoms.h"
#include "qtmisc.h"
#include <algorithm>
#include "qtexception.h"

QTAtoms::QTAtoms( DTModel& _model, const int _id ) :
  model(_model), id(_id)
{
  move(0,0);
  setFixedWidth (600);
  setMinimumSize(600,500);
  setFocusPolicy( Qt::NoFocus );

  setWindowTitle("Atoms settings");

  QHBoxLayout* layout = new QHBoxLayout(this);
  QVBoxLayout* layout1 = new QVBoxLayout;

  {
    QGroupBox* group = new QGroupBox("atom coordinates");
    QVBoxLayout* vbox  = new QVBoxLayout;

    {
      // PP, A, B, C, motion
      table = new QTableWidget( 0, 1+3+1, this );
      table->setSelectionMode( QAbstractItemView::ExtendedSelection );
      table->setSelectionBehavior( QAbstractItemView::SelectRows );

      connect(table,SIGNAL(itemChanged(QTableWidgetItem*)),this,SLOT(itemEdited(QTableWidgetItem*)) );
      connect(table,SIGNAL(itemSelectionChanged()),this,SLOT(itemSelected()) );
      int row=0;
      table->setHorizontalHeaderItem(row,new QTableWidgetItem("PP"));
      table->setColumnWidth(row++,24);
      table->setHorizontalHeaderItem(row,new QTableWidgetItem("A"));
      table->setColumnWidth(row++,70);
      table->setHorizontalHeaderItem(row,new QTableWidgetItem("B"));
      table->setColumnWidth(row++,70);
      table->setHorizontalHeaderItem(row,new QTableWidgetItem("C"));
      table->setColumnWidth(row++,70);
      table->setHorizontalHeaderItem(row,new QTableWidgetItem("mkd"));
      table->setColumnWidth(row++,30);
      /*
	table->setHorizontalHeaderItem(row,new QTableWidgetItem("Vx"));
	table->setColumnWidth(row++,50);
	table->setHorizontalHeaderItem(row,new QTableWidgetItem("Vy"));
	table->setColumnWidth(row++,50);
	table->setHorizontalHeaderItem(row,new QTableWidgetItem("Vz"));
	table->setColumnWidth(row++,50);
      */
      vbox->addWidget(table);
    }

    group->setLayout(vbox);
    layout1->addWidget(group);
  }

  {
    QGroupBox*   group = new QGroupBox;
    QGridLayout* box = new QGridLayout;
    {
      MyQPushButton* widget = new MyQPushButton
	( "remove", this, SLOT(edit(const MyEvent&)), ID_REMOVE );
      box->addWidget( widget, 0, 0 );
    }
    {
      MyQPushButton* widget = new MyQPushButton
	( "add",    this, SLOT(edit(const MyEvent&)), ID_ADD );
      box->addWidget( widget, 0, 1 );
    }
    {
      MyQPushButton* widget = new MyQPushButton
	( "undo",    this, SLOT(edit(const MyEvent&)), ID_UNDO );
      box->addWidget( widget, 1, 0 );
    }
    {
      MyQPushButton* widget = new MyQPushButton
	( "close",  this, SLOT(edit(const MyEvent&)), ID_CLOSE );
      box->addWidget( widget, 1, 1 );
    }
    group->setLayout(box);
    layout1->addWidget(group);
  }
  layout->addLayout(layout1);

  QVBoxLayout* layout2 = new QVBoxLayout;
  {
    QGroupBox*   group = new QGroupBox("selected bond");
    group->setFixedHeight(120);
    QVBoxLayout* vbox  = new QVBoxLayout;
    {
      QGridLayout* grid = new QGridLayout;
      int row=0;
      grid->addWidget(new QLabel("length:"),row,0);
      {
	MyQLineEdit* widget = new MyQLineEdit
	  (
	   model.lattice.getData().bond_length,
	   0.0, 10.0,
	   this, SLOT(edit(const MyEvent&)), ID_BOND );
	widget->setFixedWidth(100);
	grid->addWidget(widget,row,1);
	lineedit_bond_length = widget;
      }
      grid->addWidget(new QLabel("angstrom"),row,2);
      row++;

      grid->addWidget(new QLabel("angle:"),row,0);
      {
	MyQLineEdit* widget = new MyQLineEdit
	  (
	   model.lattice.getData().bond_angle,
	   -180.0, 180.0,
	   this, SLOT(edit(const MyEvent&)), ID_BOND );
	widget->setFixedWidth(100);
	grid->addWidget(widget,row,1);
	lineedit_bond_angle = widget;
      }
      grid->addWidget(new QLabel("degree"),row,2);
      row++;

      grid->addWidget(new QLabel("dihedral:"),row,0);
      {
	MyQLineEdit* widget = new MyQLineEdit
	  (
	   model.lattice.getData().bond_dihedral,
	   -180.0, 180.0,
	   this, SLOT(edit(const MyEvent&)), ID_BOND );
	widget->setFixedWidth(100);
	grid->addWidget(widget,row,1);
	lineedit_bond_dihedral = widget;
      }
      grid->addWidget(new QLabel("degree"),row,2);
      row++;

      vbox->addLayout(grid);
    }

    group->setLayout(vbox);
    layout2->addWidget(group);
  }

  {
    QGroupBox*   group = new QGroupBox("element properties");
    group->setFixedHeight(250);
    QVBoxLayout* vbox  = new QVBoxLayout;
    group->setFixedWidth(280);
    {
      MyQSpinBoxGroup* widget = new MyQSpinBoxGroup
	("PP index to edit","No.",
	 model.lattice.getData().velement_index_shown, 0, 0,
	 this, SLOT(edit(const MyEvent&)), ID_PS_INDEX );
      //      widget->setFixedWidth(40);
      vbox->addWidget(widget);

      spinbox_element = widget;
    }

    {
      QGridLayout* grid = new QGridLayout;
      int row=0;
      grid->addWidget(new QLabel("name:"),row,0);
      {
	MyQLineEdit* widget = new MyQLineEdit
	  (
	   model.lattice.getData().velement_value_shown.name,
	   this, SLOT(edit(const MyEvent&)), ID_PS_VALUE );
	widget->setFixedWidth(50);
	grid->addWidget(widget,row,1);
	lineedit_element_name = widget;
      }
      row++;

      grid->addWidget(new QLabel("zo:"),row,0);
      {
	MyQLineEdit* widget = new MyQLineEdit
	  (
	   model.lattice.getData().velement_value_shown.zo,
	   -100.0, 100.0,
	   this, SLOT(edit(const MyEvent&)), ID_PS_VALUE );
	widget->setFixedWidth(80);
	grid->addWidget(widget,row,1);
	lineedit_element_zo = widget;
      }
      row++;

      grid->addWidget(new QLabel("zn:"),row,0);
      {
	MyQLineEdit* widget = new MyQLineEdit
	  (
	   model.lattice.getData().velement_value_shown.zn,
	   -100.0, 100.0,
	   this, SLOT(edit(const MyEvent&)), ID_PS_VALUE );
	widget->setFixedWidth(80);
	grid->addWidget(widget,row,1);
	lineedit_element_zn = widget;
      }
      row++;

      grid->addWidget(new QLabel("mass:"),row,0);
      {
	MyQLineEdit* widget = new MyQLineEdit
	  (
	   model.lattice.getData().velement_value_shown.mass,
	   0.0, 1000.0,
	   this, SLOT(edit(const MyEvent&)), ID_PS_VALUE );
	widget->setFixedWidth(80);
	grid->addWidget(widget,row,1);
	lineedit_element_mass = widget;
      }
      row++;

      {
	MyQPushButton* widget = new MyQPushButton
	  ( "add", this, SLOT(edit(const MyEvent&)), ID_PS_ADD );
	widget->setFixedWidth(60);
	grid->addWidget(widget,row,0);
      }
      {
	MyQPushButton* widget = new MyQPushButton
	  ( "del", this, SLOT(edit(const MyEvent&)), ID_PS_DEL );
	widget->setFixedWidth(60);
	grid->addWidget(widget,row,1);
      }

      vbox->addLayout(grid);
    }

    group->setLayout(vbox);
    layout2->addWidget(group);
  }
  layout->addLayout(layout2);

  setLayout(layout);

  update();
}


void QTAtoms::update( void )
{
  const vector<DTAtom>& vatom = model.lattice.getAtoms();
  int natom_show = 0;
  for( int n=0; n<(int)vatom.size(); n++ ){
    if( vatom[n].isCopy() && !model.gloption.lattice.atom.show_copy ){
    }
    else{
      natom_show++;
    }
  }
  
  table->blockSignals(true);

  if( table->rowCount() != natom_show ){
    table->clearContents();
    table->setRowCount(natom_show);

    for( int n=0; n<natom_show; n++ ){
      int row=0;
      table->setItem(n,row++,new QTableWidgetItem); // element
      table->setItem(n,row++,new QTableWidgetItem); // A
      table->setItem(n,row++,new QTableWidgetItem); // B
      table->setItem(n,row++,new QTableWidgetItem); // C
      table->setItem(n,row++,new QTableWidgetItem); // fix
      /*
      table->setItem(n,row++,new QTableWidgetItem); // Vx
      table->setItem(n,row++,new QTableWidgetItem); // Vy
      table->setItem(n,row++,new QTableWidgetItem); // Vz
      */
    }
  }

  for( int n=0; n<natom_show; n++ ){
    int row=0;

    if( vatom[n].isCopy() && !model.gloption.lattice.atom.show_copy ){
      continue;
    }

    {
      QTableWidgetItem* item = table->item(n,row++);
      item->setText( tr("%1").arg(vatom[n].element.name) );
    }
    {
      QTableWidgetItem* item = table->item(n,row++);
      item->setText( tr("%1").arg(vatom[n].coords.a) );
    }
    {
      QTableWidgetItem* item = table->item(n,row++);
      item->setText( tr("%1").arg(vatom[n].coords.b) );
    }
    {
      QTableWidgetItem* item = table->item(n,row++);
      item->setText( tr("%1").arg(vatom[n].coords.c) );
    }
    {
      QTableWidgetItem* item = table->item(n,row++);
      item->setText( tr("%1").arg(vatom[n].motion) );
    }
    /*
    {
      QTableWidgetItem* item = table->item(n,row++);
      item->setText( tr("%1").arg(vatom[n].velocity.x) );
    }
    {
      QTableWidgetItem* item = table->item(n,row++);
      item->setText( tr("%1").arg(vatom[n].velocity.y) );
    }
    {
      QTableWidgetItem* item = table->item(n,row++);
      item->setText( tr("%1").arg(vatom[n].velocity.z) );
    }
    */
  }

  if( model.lattice.isAtomSelected() ){
    table->clearSelection();
    const vector<int>& viatom = model.lattice.getData().getAtomsSelected();
    for( int i=0; i<(int)viatom.size(); i++ ){
      const int iatom = viatom[i];
      table->selectRow(iatom);
    }
  }
  else{
    table->clearSelection();
  }

  table->blockSignals(false);

  lineedit_bond_length->update();
  lineedit_bond_angle->update();
  lineedit_bond_dihedral->update();

  if( model.lattice.getData().getDragMode()>=2 ){
    lineedit_bond_length->setEnabled(true);
  }
  else{
    lineedit_bond_length->setEnabled(false);
  }
  if( model.lattice.getData().getDragMode()>=3 ){
    lineedit_bond_angle->setEnabled(true);
  }
  else{
    lineedit_bond_angle->setEnabled(false);
  }
  if( model.lattice.getData().getDragMode()>=4 ){
    lineedit_bond_dihedral->setEnabled(true);
  }
  else{
    lineedit_bond_dihedral->setEnabled(false);
  }

  model.lattice.getData().update_shown();
  spinbox_element->setRange( 1, model.lattice.getData().size_shown() );
  lineedit_element_name->update();
  lineedit_element_zo->update();
  lineedit_element_zn->update();
  lineedit_element_mass->update();

  if( model.lattice.isAtomSelected() ){
    show();
  }
}


void QTAtoms::edit( const MyEvent& ev )
{
  switch(ev.id){
  case ID_REMOVE : {
    model.lattice.removeAtoms();
    model.lattice.updateAtoms();
    emit changed();
  } break;
  case ID_ADD : {
    model.lattice.addAtoms();
    model.lattice.updateAtoms();
    emit changed();
  } break;
  case ID_UNDO : {
    model.undo();
  } break;
  case ID_CLOSE : {
    model.lattice.unselectAtom();
    emit changed();
    hide();
  } break;

  case ID_BOND : {
    if( model.lattice.getData().moveBond() ){
      emit changed();
    }
    else{
      if( MyException::question( "change is restricted by symmetry.\nupdate symmetry?" ) ){
	model.lattice.symmetry.clear();
	model.lattice.updateAtoms();
	model.lattice.getData().moveBond();
	emit changed();
      }
    }
  } break;
  case ID_PS_INDEX : {
    model.lattice.getData().update_shown();
    emit changed();
  } break;
  case ID_PS_VALUE : {
    model.lattice.getData().change_shown();
    emit changed();
  } break;
  case ID_PS_ADD : {
    model.lattice.getData().addElement();
    emit changed();
  } break;
  case ID_PS_DEL : {
    model.lattice.getData().delElement();
    emit changed();
  } break;
  default: break;
  }
}


void QTAtoms::itemSelected( void )
{
  const QList<QTableWidgetItem*> litem = table->selectedItems();

  vector<int> vrow;
  for( int i=0; i<(int)litem.size(); i++ ){
    if( std::find( vrow.begin(), vrow.end(), litem.at(i)->row() ) == vrow.end() ){ // not found
      vrow.push_back( (int)litem.at(i)->row() );
    }
  }

  if( model.lattice.getAtomsSelected() != vrow ){
    model.lattice.selectAtoms(vrow);
    emit changed();
  }
}

void QTAtoms::itemEdited( QTableWidgetItem* item )
{
  const vector<DTAtom>& vatom = model.lattice.getAtoms();
  const int n = item->row();

  table->blockSignals(true);
  switch( item->column() ){

  case 0 : { // element
    char element[8];
    if( 1 == sscanf( qPrintable(item->text()), "%s", element ) ){
      item->setText( tr("%1").arg(element) );
      if( model.lattice.changeAtomElement(element) ) emit changed();
    }else{
      item->setText( tr("%1").arg(vatom[n].element.name) );
    }
  } break;
  case 1 : { // A
    double a;
    if( 1 == sscanf( qPrintable(item->text()), "%lf", &a ) ){
      item->setText( tr("%1").arg(a) );

      if( model.lattice.changeAtomCoordA(a) ){
	emit changed();
      }
      else{
	//MyException::information("coord restricted by symmetry.");
	//item->setText( tr("%1").arg(vatom[n].coords.a) );
	if( MyException::question( "coord restricted by symmetry.\nupdate symmetry?" ) ){
	  model.lattice.symmetry.clear();
	  model.lattice.updateAtoms();
	  if( model.lattice.changeAtomCoordA(a) ){
	    emit changed();
	  }
	  else{
	    item->setText( tr("%1").arg(vatom[n].coords.a) );
	  }
	}
	else{
	  item->setText( tr("%1").arg(vatom[n].coords.a) );
	}
      }

    }else{
      item->setText( tr("%1").arg(vatom[n].coords.a) );
    }
  } break;
  case 2 : { // B
    double b;
    if( 1 == sscanf( qPrintable(item->text()), "%lf", &b ) ){
      item->setText( tr("%1").arg(b) );
      if( model.lattice.changeAtomCoordB(b) ){
	emit changed();
      }
      else{
	//MyException::information("coord restricted by symmetry.");
	//item->setText( tr("%1").arg(vatom[n].coords.b) );
	if( MyException::question( "coord restricted by symmetry.\nupdate symmetry?" ) ){
	  model.lattice.symmetry.clear();
	  model.lattice.updateAtoms();
	  if( model.lattice.changeAtomCoordB(b) ){
	    emit changed();
	  }
	  else{
	    item->setText( tr("%1").arg(vatom[n].coords.b) );
	  }
	}
	else{
	  item->setText( tr("%1").arg(vatom[n].coords.b) );
	}
      }
    }else{
      item->setText( tr("%1").arg(vatom[n].coords.b) );
    }
  } break;
  case 3 : { // C
    double c;
    if( 1 == sscanf( qPrintable(item->text()), "%lf", &c ) ){
      item->setText( tr("%1").arg(c) );
      if( model.lattice.changeAtomCoordC(c) ){
	emit changed();
      }
      else{
	//MyException::information("coord restricted by symmetry.");
	//item->setText( tr("%1").arg(vatom[n].coords.c) );
	if( MyException::question( "coord restricted by symmetry.\nupdate symmetry?" ) ){
	  model.lattice.symmetry.clear();
	  model.lattice.updateAtoms();
	  if( model.lattice.changeAtomCoordC(c) ){
	    emit changed();
	  }
	  else{
	    item->setText( tr("%1").arg(vatom[n].coords.c) );
	  }
	}
	else{
	  item->setText( tr("%1").arg(vatom[n].coords.c) );
	}
      }
    }else{
      item->setText( tr("%1").arg(vatom[n].coords.c) );
    }
  } break;

  case 4 : { // Motion
    int motion;
    if( 1 == sscanf( qPrintable(item->text()), "%d", &motion ) ){
      item->setText( tr("%1").arg(motion) );
      if( model.lattice.changeAtomMotion(motion) ) emit changed();
    }else{
      item->setText( tr("%1").arg(vatom[n].motion) );
    }
  } break;
    /*
  case 5 : { // Vx
    double vx;
    if( 1 == sscanf( qPrintable(item->text()), "%lf", &vx ) ){
      item->setText( tr("%1").arg(vx) );
      if( model.lattice.changeAtomVelocityX(vx) ) emit changed();
    }else{
      item->setText( tr("%1").arg(vatom[n].velocity.x) );
    }
  } break;
  case 6 : { // Vy
    double vy;
    if( 1 == sscanf( qPrintable(item->text()), "%lf", &vy ) ){
      item->setText( tr("%1").arg(vy) );
      if( model.lattice.changeAtomVelocityY(vy) ) emit changed();
    }else{
      item->setText( tr("%1").arg(vatom[n].velocity.y) );
    }
  } break;
  case 7 : { // Vz
    double vz;
    if( 1 == sscanf( qPrintable(item->text()), "%lf", &vz ) ){
      item->setText( tr("%1").arg(vz) );
      if( model.lattice.changeAtomVelocityZ(vz) ) emit changed();
    }else{
      item->setText( tr("%1").arg(vatom[n].velocity.z) );
    }
  } break;
    */

  }
  table->blockSignals(false);
}




