/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

/*!
 \file qttapp.cc
 \brief TAPP̐ݒGUI\̃NX
*/

#include "dtmodel.h"
#include "qttapp.h"

//------------------------------------------------------------

QTTappOption::QTTappOption( DTModel& _model, const int id ) : MyQTab(id), model(_model)
{
  QVBoxLayout* layout = new QVBoxLayout(this);
  layout->setSizeConstraint( QLayout::SetFixedSize );

  // Option
  {
    MyQGroupBox* group = new MyQGroupBox("Option");
    QVBoxLayout* box = new QVBoxLayout;

    {
      MyQCheckBox* widget = new MyQCheckBox
	( "wfn on disk", model.tapp.option.wfn_on_disk,
	  this, SLOT(edit(const MyEvent&)), ID_OPTION );
      box->addWidget(widget);
      vwidget.push_back(widget);
    }
    {
      MyQCheckBox* widget = new MyQCheckBox
	( "qg on disk", model.tapp.option.qg_on_disk,
	  this, SLOT(edit(const MyEvent&)), ID_OPTION );
      box->addWidget(widget);
      vwidget.push_back(widget);
    }

    // fsmear
    {
      QGroupBox* group = new QGroupBox("fsmear");
      QVBoxLayout* vbox = new QVBoxLayout;

      {
	MyQRadioButton* widget = new MyQRadioButton
	  ( "cosine", model.tapp.option.fsmear, 0,
	    this, SLOT(edit(const MyEvent&)), ID_OPTION );
	vbox->addWidget(widget);
	vwidget.push_back(widget);
      }
      {
	MyQRadioButton* widget = new MyQRadioButton
	  ( "fermi", model.tapp.option.fsmear, 1,
	    this, SLOT(edit(const MyEvent&)), ID_OPTION );
	vbox->addWidget(widget);
	vwidget.push_back(widget);
      }
      {
	MyQRadioButton* widget = new MyQRadioButton
	  ( "auto", model.tapp.option.fsmear, -1,
	    this, SLOT(edit(const MyEvent&)), ID_OPTION );
	vbox->addWidget(widget);
	vwidget.push_back(widget);
      }

      group->setLayout(vbox);
      box->addWidget(group);
    }

    // hpc
    {
      QGroupBox* group = new QGroupBox("hpc");
      QVBoxLayout* vbox = new QVBoxLayout;

      {
	MyQRadioButton* widget = new MyQRadioButton
	  ( "none", model.tapp.option.hpc, 0,
	    this, SLOT(edit(const MyEvent&)), ID_OPTION );
	vbox->addWidget(widget);
	vwidget.push_back(widget);
      }
      {
	MyQRadioButton* widget = new MyQRadioButton
	  ( "HPC", model.tapp.option.hpc, 1,
	    this, SLOT(edit(const MyEvent&)), ID_OPTION );
	vbox->addWidget(widget);
	vwidget.push_back(widget);
      }
      {
	MyQRadioButton* widget = new MyQRadioButton
	  ( "FI", model.tapp.option.hpc, 2,
	    this, SLOT(edit(const MyEvent&)), ID_OPTION );
	vbox->addWidget(widget);
	vwidget.push_back(widget);
      }

      group->setLayout(vbox);
      box->addWidget(group);
    }

    group->setLayout(box);
    layout->addWidget(group);

    group->setExpert(true);
    vgroup.push_back(group);
  }

  setLayout(layout);
  updateExpert(false);
}

void QTTappOption::edit( const MyEvent& ev )
{
  emit changed(MyEvent(id));
}

void QTTappOption::updateExpert( const bool expert )
{
  MyQTab::updateExpert(expert);
}

//------------------------------------------------------------

QTTappLSDA::QTTappLSDA( DTModel& _model, const int id ) : MyQTab(id), model(_model)
{
  QVBoxLayout* layout = new QVBoxLayout(this);
  layout->setSizeConstraint( QLayout::SetFixedSize );


  // xctype
  {
    MyQGroupBox* group = new MyQGroupBox("xctype");
    QHBoxLayout* box = new QHBoxLayout;

    box->addWidget(new QLabel("xctype:"));
    {
      MyQComboBox* widget = new MyQComboBox
	( model.tapp.lsda.ixctype, model.tapp.lsda.vxctype,
	  this, SLOT(edit(const MyEvent&)), ID_XCTYPE );
      box->addWidget(widget);
      vwidget.push_back(widget);
    }

    group->setLayout(box);
    layout->addWidget(group);

    group->setExpert(false);
    vgroup.push_back(group);
  }

  // nspin
  {
    MyQGroupBox* group = new MyQGroupBox("nspin");
    QVBoxLayout* box = new QVBoxLayout;

    {
      MyQRadioButton* widget = new MyQRadioButton
	( "LDA",
	  model.tapp.lsda.nspin, 1,
	  this, SLOT(edit(const MyEvent&)), ID_NSPIN );
      box->addWidget(widget);
      vwidget.push_back(widget);
    }

    {
      MyQRadioButton* widget = new MyQRadioButton
	( "LSD",
	  model.tapp.lsda.nspin, 2,
	  this, SLOT(edit(const MyEvent&)), ID_NSPIN );
      box->addWidget(widget);
      vwidget.push_back(widget);
    }

    group->setLayout(box);
    layout->addWidget(group);

    group->setExpert(true);
    vgroup.push_back(group);
  }

  // spinmode
  {
    MyQGroupBox* group = new MyQGroupBox("spinmode");
    QVBoxLayout* box = new QVBoxLayout;

    {
      MyQRadioButton* widget = new MyQRadioButton
	( "Free mode",
	  model.tapp.lsda.spinmode, 0,
	  this, SLOT(edit(const MyEvent&)), ID_SPINMODE );
      box->addWidget(widget);
      vwidget.push_back(widget);
    }

    {
      MyQRadioButton* widget = new MyQRadioButton
	( "Fixed mode",
	  model.tapp.lsda.spinmode, 1,
	  this, SLOT(edit(const MyEvent&)), ID_SPINMODE );
      box->addWidget(widget);
      vwidget.push_back(widget);
    }

    group->setLayout(box);
    layout->addWidget(group);

    group->setExpert(true);
    vgroup.push_back(group);
  }


  // ipmode
  {
    MyQGroupBox* group = new MyQGroupBox("ipmode");
    QVBoxLayout* box = new QVBoxLayout;

    {
      MyQRadioButton* widget = new MyQRadioButton
	( "Bare PP",
	  model.tapp.lsda.ipmode, 0,
	  this, SLOT(edit(const MyEvent&)), ID_IPMODE );
      box->addWidget(widget);
      vwidget.push_back(widget);
    }

    {
      MyQRadioButton* widget = new MyQRadioButton
	( "Neutralized",
	  model.tapp.lsda.ipmode, 1,
	  this, SLOT(edit(const MyEvent&)), ID_IPMODE );
      box->addWidget(widget);
      vwidget.push_back(widget);
    }

    {
      MyQRadioButton* widget = new MyQRadioButton
	( "Spin polarized",
	  model.tapp.lsda.ipmode, 2,
	  this, SLOT(edit(const MyEvent&)), ID_IPMODE );
      box->addWidget(widget);
      vwidget.push_back(widget);
    }

    group->setLayout(box);
    layout->addWidget(group);

    group->setExpert(true);
    vgroup.push_back(group);
  }

  // nfixed
  {
    MyQGroupBox* group = new MyQGroupBox("nfixed");
    QHBoxLayout* box = new QHBoxLayout;

    {
      box->addWidget(new QLabel("nfixed:") );
      {
	MyQSpinBox* widget = new MyQSpinBox
	  ( model.tapp.lsda.nfixed,
	    DTTapp::LSDA::nfixed_min, DTTapp::LSDA::nfixed_max,
	    this, SLOT(edit(const MyEvent&)), ID_NFIXED );
	box->addWidget(widget);

	vwidget.push_back(widget);
      }
      box->addWidget(new QLabel("steps") );
    }

    group->setLayout(box);
    layout->addWidget(group);

    group->setExpert(true);
    vgroup.push_back(group);
  }

  setLayout(layout);
  updateExpert(false);
}

void QTTappLSDA::edit( const MyEvent& ev )
{
  emit changed(MyEvent(id));
}

void QTTappLSDA::updateExpert( const bool expert )
{
  MyQTab::updateExpert(expert);
}

//------------------------------------------------------------

QTTappCutoff::QTTappCutoff( DTModel& _model, const int id ) : MyQTab(id), model(_model)
{
  QVBoxLayout* layout = new QVBoxLayout(this);
  layout->setSizeConstraint( QLayout::SetFixedSize );

  // qf
  {
    MyQGroupBox* group = new MyQGroupBox("cutoff for PP");
    QHBoxLayout* box = new QHBoxLayout;

    box->addWidget(new QLabel("qf:  "));
    {
      MyQLineEdit* widget = new MyQLineEdit
	(
	 model.tapp.cutoff.qf, DTTapp::Cutoff::qf_min, DTTapp::Cutoff::qf_max,
	 this, SLOT(edit(const MyEvent&)), ID_QF );
      widget->setFixedWidth(100);
      box->addWidget(widget);

      vwidget.push_back(widget);
    }
    box->addWidget(new QLabel("Rydberg"));

    group->setLayout(box);
    layout->addWidget(group);

    group->setExpert(false);
    vgroup.push_back(group);
  }

  // rf
  {
    MyQGroupBox* group = new MyQGroupBox("cutoff for grid");
    QHBoxLayout* box = new QHBoxLayout;

    box->addWidget(new QLabel("rf:  "));
    {
      MyQLineEdit* widget = new MyQLineEdit
	(
	 model.tapp.cutoff.rf, DTTapp::Cutoff::rf_min, DTTapp::Cutoff::rf_max,
	 this, SLOT(edit(const MyEvent&)), ID_RF );
      widget->setFixedWidth(100);
      box->addWidget(widget);

      vwidget.push_back(widget);
    }
    box->addWidget(new QLabel("Rydberg"));

    group->setLayout(box);
    layout->addWidget(group);

    group->setExpert(true);
    vgroup.push_back(group);
  }

  // qm
  {
    MyQGroupBox* group = new MyQGroupBox("cutoff for wavefunction");
    QHBoxLayout* box = new QHBoxLayout;

    box->addWidget(new QLabel("qm:  "));
    {
      MyQLineEdit* widget = new MyQLineEdit
	(
	 model.tapp.cutoff.qm, DTTapp::Cutoff::qm_min, DTTapp::Cutoff::qm_max,
	 this, SLOT(edit(const MyEvent&)), ID_QM );
      widget->setFixedWidth(100);
      box->addWidget(widget);

      vwidget.push_back(widget);
    }
    box->addWidget(new QLabel("Rydberg"));

    group->setLayout(box);
    layout->addWidget(group);

    group->setExpert(false);
    vgroup.push_back(group);
  }

  // qc
  {
    MyQGroupBox* group = new MyQGroupBox("cutoff for kspace");
    QHBoxLayout* box = new QHBoxLayout;

    box->addWidget(new QLabel("qc:  "));
    {
      MyQLineEdit* widget = new MyQLineEdit
	(
	 model.tapp.cutoff.qc, DTTapp::Cutoff::qc_min, DTTapp::Cutoff::qc_max,
	 this, SLOT(edit(const MyEvent&)), ID_QC );
      widget->setFixedWidth(100);
      box->addWidget(widget);

      vwidget.push_back(widget);
    }
    box->addWidget(new QLabel("Rydberg"));

    group->setLayout(box);
    layout->addWidget(group);

    group->setExpert(false);
    vgroup.push_back(group);
  }

  // beta
  {
    MyQGroupBox* group = new MyQGroupBox("mixing weight");
    QHBoxLayout* box = new QHBoxLayout;

    box->addWidget(new QLabel("beta:"));
    {
      MyQLineEdit* widget = new MyQLineEdit
	(
	 model.tapp.cutoff.beta, DTTapp::Cutoff::beta_min, DTTapp::Cutoff::beta_max,
	 this, SLOT(edit(const MyEvent&)), ID_BETA );
      widget->setFixedWidth(100);
      box->addWidget(widget);

      vwidget.push_back(widget);
    }
    box->addWidget(new QLabel("Rydberg"));

    group->setLayout(box);
    layout->addWidget(group);

    group->setExpert(true);
    vgroup.push_back(group);
  }

  setLayout(layout);
  updateExpert(false);
}

void QTTappCutoff::edit( const MyEvent& ev )
{
  emit changed(MyEvent(id));
}

void QTTappCutoff::updateExpert( const bool expert )
{
  MyQTab::updateExpert(expert);
}


//------------------------------------------------------------

QTTappMesh::QTTappMesh( DTModel& _model, const int id ) : MyQTab(id), model(_model)
{
  QVBoxLayout* layout = new QVBoxLayout(this);
  layout->setSizeConstraint( QLayout::SetFixedSize );

  // ndx,ndy,ndz
  {
    MyQGroupBox* group = new MyQGroupBox("K-mesh for DOS");
    QHBoxLayout* box = new QHBoxLayout;

    box->addWidget(new QLabel("ndx:"));
    {
      MyQSpinBox* widget = new MyQSpinBox
	( model.tapp.mesh.ndx, DTTapp::Mesh::nd_min, DTTapp::Mesh::nd_max,
	  this, SLOT(edit(const MyEvent&)), ID_ND );
      box->addWidget(widget);
      vwidget.push_back(widget);
    }
    box->addWidget(new QLabel("ndy:"));
    {
      MyQSpinBox* widget = new MyQSpinBox
	( model.tapp.mesh.ndy, DTTapp::Mesh::nd_min, DTTapp::Mesh::nd_max,
	  this, SLOT(edit(const MyEvent&)), ID_ND );
      box->addWidget(widget);
      vwidget.push_back(widget);
    }
    box->addWidget(new QLabel("ndz:"));
    {
      MyQSpinBox* widget = new MyQSpinBox
	( model.tapp.mesh.ndz, DTTapp::Mesh::nd_min, DTTapp::Mesh::nd_max,
	  this, SLOT(edit(const MyEvent&)), ID_ND );
      box->addWidget(widget);
      vwidget.push_back(widget);
    }

    group->setLayout(box);
    layout->addWidget(group);

    group->setExpert(true);
    vgroup.push_back(group);
  }


  // nw
  {
    MyQGroupBox* group = new MyQGroupBox("number of band");
    QHBoxLayout* box = new QHBoxLayout;

    box->addWidget(new QLabel("nw:"), Qt::AlignRight );
    {
      MyQSpinBox* widget = new MyQSpinBox
	( 
	 model.tapp.mesh.nw, DTTapp::Mesh::nw_min, DTTapp::Mesh::nw_max,
	  this, SLOT(edit(const MyEvent&)), ID_NW );
      widget->setFixedWidth(60);
      box->addWidget(widget, Qt::AlignLeft);
      vwidget.push_back(widget);
    }

    group->setLayout(box);
    layout->addWidget(group);

    group->setExpert(false);
    vgroup.push_back(group);
  }

  // nb1,nb2
  {
    MyQGroupBox* group = new MyQGroupBox("range of band");
    QHBoxLayout* box = new QHBoxLayout;

    box->addWidget(new QLabel("nb1:"), Qt::AlignRight);
    {
      MyQSpinBox* widget = new MyQSpinBox
	( 
	 model.tapp.mesh.nb1, DTTapp::Mesh::nb_min, DTTapp::Mesh::nb_max,
	  this, SLOT(edit(const MyEvent&)), ID_NB );
      box->addWidget(widget, Qt::AlignLeft);
      vwidget.push_back(widget);
    }

    box->addWidget(new QLabel("nb2:"), Qt::AlignRight);
    {
      MyQSpinBox* widget = new MyQSpinBox
	( 
	 model.tapp.mesh.nb2, DTTapp::Mesh::nb_min, DTTapp::Mesh::nb_max,
	  this, SLOT(edit(const MyEvent&)), ID_NB );
      box->addWidget(widget, Qt::AlignLeft);
      vwidget.push_back(widget);
    }

    group->setLayout(box);
    layout->addWidget(group);

    group->setExpert(false);
    vgroup.push_back(group);
  }

  // znext
  {
    MyQGroupBox* group = new MyQGroupBox("extra electrons");
    QHBoxLayout* box = new QHBoxLayout;

    box->addWidget(new QLabel("znext:"), Qt::AlignRight);
    {
      MyQLineEdit* widget = new MyQLineEdit
	(
	 model.tapp.mesh.znext, DTTapp::Mesh::znext_min, DTTapp::Mesh::znext_max,
	 this, SLOT(edit(const MyEvent&)), ID_ZNEXT );
      widget->setFixedWidth(100);
      box->addWidget(widget, Qt::AlignLeft);
      vwidget.push_back(widget);
    }

    group->setLayout(box);
    layout->addWidget(group);

    group->setExpert(true);
    vgroup.push_back(group);
  }

  // dpsin
  {
    MyQGroupBox* group = new MyQGroupBox("extra spins");
    QHBoxLayout* box = new QHBoxLayout;

    box->addWidget(new QLabel("dspin:"), Qt::AlignRight);
    {
      MyQLineEdit* widget = new MyQLineEdit
	(
	 model.tapp.mesh.dspin, DTTapp::Mesh::dspin_min, DTTapp::Mesh::dspin_max,
	 this, SLOT(edit(const MyEvent&)), ID_DSPIN );
      widget->setFixedWidth(100);
      box->addWidget(widget, Qt::AlignLeft);
      vwidget.push_back(widget);
    }

    group->setLayout(box);
    layout->addWidget(group);

    group->setExpert(true);
    vgroup.push_back(group);
  }

  setLayout(layout);
  updateExpert(false);
}

void QTTappMesh::edit( const MyEvent& ev )
{
  emit changed(MyEvent(id));
}

void QTTappMesh::updateExpert( const bool expert )
{
  MyQTab::updateExpert(expert);
}

//------------------------------------------------------------

QTTappEwald::QTTappEwald( DTModel& _model, const int id ) : MyQTab(id), model(_model)
{
  QVBoxLayout* layout = new QVBoxLayout(this);
  layout->setSizeConstraint( QLayout::SetFixedSize );

  // qgb
  {
    MyQGroupBox* group = new MyQGroupBox("size in G-space");
    QHBoxLayout* box = new QHBoxLayout;

    box->addWidget(new QLabel("qgb:"));
    {
      MyQLineEdit* widget = new MyQLineEdit
	(
	 model.tapp.ewald.qgb, DTTapp::Ewald::qgb_min, DTTapp::Ewald::qgb_max,
	 this, SLOT(edit(const MyEvent&)), ID_QGB );
      widget->setFixedWidth(100);
      box->addWidget(widget);
      vwidget.push_back(widget);
    }

    group->setLayout(box);
    layout->addWidget(group);

    group->setExpert(true);
    vgroup.push_back(group);
  }

  // qrb
  {
    MyQGroupBox* group = new MyQGroupBox("size in R-space");
    QHBoxLayout* box = new QHBoxLayout;

    box->addWidget(new QLabel("qrb:"));
    {
      MyQLineEdit* widget = new MyQLineEdit
	(
	 model.tapp.ewald.qrb, DTTapp::Ewald::qrb_min, DTTapp::Ewald::qrb_max,
	 this, SLOT(edit(const MyEvent&)), ID_QRB );
      widget->setFixedWidth(100);
      box->addWidget(widget);
      vwidget.push_back(widget);
    }

    group->setLayout(box);
    layout->addWidget(group);

    group->setExpert(true);
    vgroup.push_back(group);
  }

  // g
  {
    MyQGroupBox* group = new MyQGroupBox("cutoff");
    QHBoxLayout* box = new QHBoxLayout;

    box->addWidget(new QLabel("g:  "));
    {
      MyQLineEdit* widget = new MyQLineEdit
	(
	 model.tapp.ewald.g, DTTapp::Ewald::g_min, DTTapp::Ewald::g_max,
	 this, SLOT(edit(const MyEvent&)), ID_G );
      widget->setFixedWidth(100);
      box->addWidget(widget);
      vwidget.push_back(widget);
    }

    group->setLayout(box);
    layout->addWidget(group);

    group->setExpert(true);
    vgroup.push_back(group);
  }

  setLayout(layout);
  updateExpert(false);
}

void QTTappEwald::edit( const MyEvent& ev )
{
  emit changed(MyEvent(id));
}

void QTTappEwald::updateExpert( const bool expert )
{
  MyQTab::updateExpert(expert);
}


//------------------------------------------------------------

QTTappSampling::QTTappSampling( DTModel& _model, const int id ) : MyQTab(id), model(_model)
{
  QVBoxLayout* layout = new QVBoxLayout(this);
  layout->setSizeConstraint( QLayout::SetFixedSize );

  // outline
  {
    MyQGroupBox* group = new MyQGroupBox("outline setting");
    QVBoxLayout* box = new QVBoxLayout;

    {
      QGroupBox* sgroup = new QGroupBox;
      QHBoxLayout* hbox = new QHBoxLayout;
      hbox->addWidget(new QLabel("mesh:"), Qt::AlignRight);
      {
	MyQComboBox* widget = new MyQComboBox
	  ( model.tapp.sampling.imesh, model.tapp.sampling.vmesh,
	    this, SLOT(edit(const MyEvent&)), ID_OUTLINE );
	hbox->addWidget(widget, Qt::AlignLeft);
	vwidget.push_back(widget);
      }
      sgroup->setLayout(hbox);
      box->addWidget(sgroup);
    }

    {
      QGroupBox* sgroup = new QGroupBox;
      QHBoxLayout* hbox = new QHBoxLayout;

      hbox->addWidget(new QLabel("mesh:"));
      for( int j=0; j<3; j++ ){
	MyQSpinBox* widget = new MyQSpinBox
	  ( model.tapp.sampling.mesh[j],
	    DTTapp::Mesh::nk_min, DTTapp::Mesh::nk_max,
	    this, SLOT(edit(const MyEvent&)), ID_OUTLINE );
	hbox->addWidget(widget);
	vwidget.push_back(widget);
      }
      sgroup->setLayout(hbox);
      box->addWidget(sgroup);
    }

    {
      MyQCheckBox* widget = new MyQCheckBox
	( "gamma point in mesh", model.tapp.sampling.gamma,
	  this, SLOT(edit(const MyEvent&)), ID_OUTLINE );
      box->addWidget(widget);
      vwidget.push_back(widget);
    }

    group->setLayout(box);
    layout->addWidget(group);

    group->setExpert(false);
    vgroup.push_back(group);
  }

  // detail 
  {
    MyQGroupBox* group = new MyQGroupBox("detail setting");
    QVBoxLayout* box = new QVBoxLayout;

    {
      QGroupBox* sgroup = new QGroupBox;
      QHBoxLayout* hbox = new QHBoxLayout;

      hbox->addWidget(new QLabel("nk:"), Qt::AlignRight);
      {
	MyQSpinBox* widget = new MyQSpinBox
	  ( 
	   model.tapp.mesh.nk, DTTapp::Mesh::nk_min, DTTapp::Mesh::nk_max,
	   this, SLOT(edit(const MyEvent&)), ID_DETAIL_NK );
	widget->setSingleStep(2);
	widget->setFixedWidth(60);
	hbox->addWidget(widget, Qt::AlignLeft);
	vwidget.push_back(widget);
      }
      sgroup->setLayout(hbox);
      box->addWidget(sgroup);
    }

    {
      QGroupBox* sgroup = new QGroupBox("mmm");
      QGridLayout* grid = new QGridLayout;

      int row=0;

      {
	QGroupBox* group = new QGroupBox("mmm index to edit");
	QHBoxLayout* box = new QHBoxLayout;

	box->addWidget(new QLabel("No."));
	{
	  MyQSpinBox* widget = new MyQSpinBox
	    ( model.tapp.sampling.index_shown, 0, 0,
	      this, SLOT(edit(const MyEvent&)), ID_DETAIL_INDEX );
	  box->addWidget(widget);
	  vwidget.push_back(widget);
	  spinbox_index = widget;
	}
	{
	  MyQLabel* widget = new MyQLabel("of 0 elements");
	  box->addWidget(widget);
	  vwidget.push_back(widget);
	  label_size = widget;
	}
	group->setLayout(box);

	grid->addWidget(group,row,0,1,4);
      }
      row++;

      grid->addWidget(new QLabel("mmm1:"),row,0);
      for( int j=0; j<3; j++ ){
	MyQSpinBox* widget = new MyQSpinBox
	  ( model.tapp.sampling.mmm1_shown[j],
	    DTTapp::Mesh::nk_min, DTTapp::Mesh::nk_max,
	    this, SLOT(edit(const MyEvent&)), ID_DETAIL_MMM );
	grid->addWidget(widget,row,j+1);
	vwidget.push_back(widget);
      }
      row++;

      grid->addWidget(new QLabel("mmm2:"),row,0);
      for( int j=0; j<3; j++ ){
	MyQSpinBox* widget = new MyQSpinBox
	  ( model.tapp.sampling.mmm2_shown[j],
	    DTTapp::Mesh::nk_min, DTTapp::Mesh::nk_max,
	    this, SLOT(edit(const MyEvent&)), ID_DETAIL_MMM );
	grid->addWidget(widget,row,j+1);
	vwidget.push_back(widget);
      }
      row++;

      {
	MyQPushButton* widget = new MyQPushButton
	  ( "add", this, SLOT(edit(const MyEvent&)), ID_DETAIL_ADD );
	widget->setFixedWidth(60);
	grid->addWidget(widget,row,0,1,2);
	vwidget.push_back(widget);
      }
      {
	MyQPushButton* widget = new MyQPushButton
	  ( "del", this, SLOT(edit(const MyEvent&)), ID_DETAIL_DEL );
	widget->setFixedWidth(60);
	grid->addWidget(widget,row,2,1,2);
	vwidget.push_back(widget);
      }
      row++;

      sgroup->setLayout(grid);
      box->addWidget(sgroup);
    }

    {
      QGroupBox* sgroup = new QGroupBox("extra parameters");
      QGridLayout* grid = new QGridLayout;
      int row=0;

      grid->addWidget(new QLabel("nkrnew:"),row,0);
      {
	MyQSpinBox* widget = new MyQSpinBox
	  ( model.tapp.sampling.nkrnew, DTTapp::Sampling::nkrnew_min, DTTapp::Sampling::nkrnew_max,
	    this, SLOT(edit(const MyEvent&)), ID_DETAIL_EX );
	grid->addWidget(widget,row,1);
	vwidget.push_back(widget);
      }
      row++;

      grid->addWidget(new QLabel("ll:"),row,0);
      {
	MyQSpinBox* widget = new MyQSpinBox
	  ( model.tapp.sampling.ll1, DTTapp::Sampling::ll_min, DTTapp::Sampling::ll_max,
	    this, SLOT(edit(const MyEvent&)), ID_DETAIL_EX );
	grid->addWidget(widget,row,1);
	vwidget.push_back(widget);
      }
      {
	MyQSpinBox* widget = new MyQSpinBox
	  ( model.tapp.sampling.ll2, DTTapp::Sampling::ll_min, DTTapp::Sampling::ll_max,
	    this, SLOT(edit(const MyEvent&)), ID_DETAIL_EX );
	grid->addWidget(widget,row,2);
	vwidget.push_back(widget);
      }
      {
	MyQSpinBox* widget = new MyQSpinBox
	  ( model.tapp.sampling.ll3, DTTapp::Sampling::ll_min, DTTapp::Sampling::ll_max,
	    this, SLOT(edit(const MyEvent&)), ID_DETAIL_EX );
	grid->addWidget(widget,row,3);
	vwidget.push_back(widget);
      }
      sgroup->setLayout(grid);
      box->addWidget(sgroup);
    }

    group->setLayout(box);
    layout->addWidget(group);

    group->setExpert(true);
    vgroup.push_back(group);
  }

  setLayout(layout);
  updateExpert(false);
}

void QTTappSampling::edit( const MyEvent& ev )
{
  switch(ev.id){
  case ID_OUTLINE : {
    model.tapp.sampling.setByOutline( model.tapp.mesh.nk );
  } break;
  case ID_DETAIL_NK : {
    model.tapp.sampling.setByDetail( model.tapp.mesh.nk );
  } break;
  case ID_DETAIL_INDEX : {
    model.tapp.sampling.update_shown();
  } break;
  case ID_DETAIL_MMM : {
    model.tapp.sampling.change_shown();
    model.tapp.sampling.setByDetail( model.tapp.mesh.nk );
  } break;
  case ID_DETAIL_ADD : {
    model.tapp.sampling.addMMM();
    model.tapp.sampling.setByDetail( model.tapp.mesh.nk );
  } break;
  case ID_DETAIL_DEL : {
    model.tapp.sampling.delMMM();
    model.tapp.sampling.setByDetail( model.tapp.mesh.nk );
  } break;
  case ID_DETAIL_EX : {
    model.tapp.sampling.setByDetail( model.tapp.mesh.nk );
  } break;
  default : break;
  }

  emit changed(MyEvent(id));
}

void QTTappSampling::updateExpert( const bool expert )
{
  model.tapp.sampling.update_shown();
  spinbox_index->setRange( 1, model.tapp.sampling.size_shown() );
  {
    char text[32];
    snprintf( text, sizeof(text), "of %d elements",
	      model.tapp.sampling.size_shown() );
    label_size->setText(text);
  }

  MyQTab::updateExpert(expert);
}


//------------------------------------------------------------

QTTappFile::QTTappFile( DTModel& _model, const int id ) : MyQTab(id), model(_model)
{
  QVBoxLayout* layout = new QVBoxLayout(this);
  layout->setSizeConstraint( QLayout::SetFixedSize );

  // File
  {
    MyQGroupBox* group = new MyQGroupBox("File");
    QVBoxLayout* box = new QVBoxLayout;

    {
      MyQCheckBox* widget = new MyQCheckBox
	( "load restart", model.tapp.file.iopt1,
	  this, SLOT(edit(const MyEvent&)), ID_FILE );
      box->addWidget(widget);
    }

    {
      MyQCheckBox* widget = new MyQCheckBox
	( "load wavefunctions", model.tapp.file.iopt2,
	  this, SLOT(edit(const MyEvent&)), ID_FILE );
      box->addWidget(widget);
    }

    {
      MyQCheckBox* widget = new MyQCheckBox
	( "load potentials", model.tapp.file.iopt3,
	  this, SLOT(edit(const MyEvent&)), ID_FILE );
      box->addWidget(widget);
    }

    {
      MyQCheckBox* widget = new MyQCheckBox
	( "save wavefunctions", model.tapp.file.iopt4,
	  this, SLOT(edit(const MyEvent&)), ID_FILE );
      box->addWidget(widget);
    }

    group->setLayout(box);
    layout->addWidget(group);

    group->setExpert(true);
    vgroup.push_back(group);
  }

  setLayout(layout);
  updateExpert(false);
}

void QTTappFile::edit( const MyEvent& ev )
{
  emit changed(MyEvent(id));
}

void QTTappFile::updateExpert( const bool expert )
{
  MyQTab::updateExpert(expert);
}


//------------------------------------------------------------

QTTappMass::QTTappMass( DTModel& _model, const int id ) : MyQTab(id), model(_model)
{
  QVBoxLayout* layout = new QVBoxLayout(this);
  layout->setSizeConstraint( QLayout::SetFixedSize );

  // 
  {
    MyQGroupBox* group = new MyQGroupBox("Inverse mass tensor");
    QGridLayout* grid = new QGridLayout;
    int row = 0;

    {
      QGroupBox* group = new QGroupBox("tim index to edit");
      QHBoxLayout* box = new QHBoxLayout;

      box->addWidget(new QLabel("No."));
      {
	MyQSpinBox* widget = new MyQSpinBox
	  ( model.tapp.mass.index_shown, 0, 0,
	    this, SLOT(edit(const MyEvent&)), ID_INDEX );
	box->addWidget(widget);
	vwidget.push_back(widget);
	spinbox_index = widget;
      }
      {
	MyQLabel* widget = new MyQLabel("of 0 elements");
	box->addWidget(widget);
	vwidget.push_back(widget);
	label_size = widget;
      }
      group->setLayout(box);

      grid->addWidget(group,row,0,1,4);
    }
    row++;

    grid->addWidget(new QLabel("tim:"),row,0);
    for( int i=0; i<3; i++ ){
      for( int j=0; j<3; j++ ){
	MyQLineEdit* widget = new MyQLineEdit
	  (
	   model.tapp.mass.tim_shown(i,j), 0.0, 128.0,
	   this, SLOT(edit(const MyEvent&)), ID_TENSOR );
	widget->setFixedWidth(80);
	grid->addWidget(widget,row,j+1);
	vwidget.push_back(widget);
      }
      row++;
    }

    {
      MyQPushButton* widget = new MyQPushButton
	( "add", this, SLOT(edit(const MyEvent&)), ID_ADD );
      widget->setFixedWidth(60);
      grid->addWidget(widget,row,0,1,2,Qt::AlignCenter);
      vwidget.push_back(widget);
    }
    {
      MyQPushButton* widget = new MyQPushButton
	( "del", this, SLOT(edit(const MyEvent&)), ID_DEL );
      widget->setFixedWidth(60);
      grid->addWidget(widget,row,2,1,2,Qt::AlignCenter);
      vwidget.push_back(widget);
    }
    row++;

    group->setLayout(grid);
    layout->addWidget(group);

    group_mass = group;

    group->setExpert(true);
    vgroup.push_back(group);
  }

  setLayout(layout);
  updateExpert(false);
}

void QTTappMass::edit( const MyEvent& ev )
{
  switch(ev.id){
  case ID_INDEX : {
    model.tapp.mass.update_shown();
  } break;
  case ID_TENSOR : {
    model.tapp.mass.change_shown();
  } break;
  case ID_ADD : {
    model.tapp.mass.add();
  } break;
  case ID_DEL : {
    model.tapp.mass.del();
  } break;

  default : break;
  }

  emit changed(MyEvent(id));
}

void QTTappMass::updateExpert( const bool expert )
{
  model.tapp.mass.update_shown();
  spinbox_index->setRange( 1, model.tapp.mass.size_shown() );
  {
    char text[32];
    snprintf( text, sizeof(text), "of %d elements",
	      model.tapp.mass.size_shown() );
    label_size->setText(text);
  }

  MyQTab::updateExpert(expert);
}


//------------------------------------------------------------

QTTappParameter::QTTappParameter( DTModel& _model, const int id ) : MyQTab(id), model(_model)
{
  QVBoxLayout* layout = new QVBoxLayout(this);
  layout->setSizeConstraint( QLayout::SetFixedSize );

  // parameters
  {
    MyQGroupBox* group = new MyQGroupBox("Parameters");
    QGridLayout* box = new QGridLayout;
    int row=0;

    // eps
    {
      MyQEditSlider* widget = new MyQEditSlider
	( "eps", "eps:", "", "high", "low",
	  model.tapp.parameter.eps, DTTapp::Parameter::eps_min, DTTapp::Parameter::eps_max,
	  this, SLOT(edit(const MyEvent&)), ID_EPS );
      widget->setFixedWidth(250);
      box->addWidget(widget,row,0);
      vwidget.push_back(widget);
    }

    // eepsa
    {
      MyQEditSlider* widget = new MyQEditSlider
	( "eepsa", "eepsa:", "", "high", "low",
	 model.tapp.parameter.eepsa, DTTapp::Parameter::eepsa_min, DTTapp::Parameter::eepsa_max,
	 this, SLOT(edit(const MyEvent&)), ID_EEPSA );
      widget->setFixedWidth(250);
      box->addWidget(widget,row,1);
      vwidget.push_back(widget);
    }
    row++;

    // feps
    {
      MyQEditSlider* widget = new MyQEditSlider
	( "feps", "feps:", "", "high", "low",
	 model.tapp.parameter.feps, DTTapp::Parameter::feps_min, DTTapp::Parameter::feps_max,
	 this, SLOT(edit(const MyEvent&)), ID_FEPS );
      widget->setFixedWidth(250);
      box->addWidget(widget,row,0);
      vwidget.push_back(widget);
    }

    // ekbt
    {
      MyQEditSlider* widget = new MyQEditSlider
	( "ekbt", "ekbt:", "", "high", "low",
	 model.tapp.parameter.ekbt, 1.0e-7, 1.0e-3,
	 this, SLOT(edit(const MyEvent&)), ID_EKBT );
      widget->setFixedWidth(250);
      box->addWidget(widget,row,1);
      vwidget.push_back(widget);
    }
    row++;

    // decr
    {
      MyQEditSlider* widget = new MyQEditSlider
	( "decr", "decr:", "", "high", "low",
	 model.tapp.parameter.decr, DTTapp::Parameter::decr_min, DTTapp::Parameter::decr_max,
	 this, SLOT(edit(const MyEvent&)), ID_DECR );
      widget->setFixedWidth(250);
      box->addWidget(widget,row,0);
      vwidget.push_back(widget);
    }

    // okatom
    {
      MyQEditSlider* widget = new MyQEditSlider
	( "okatom", "okatom:", "", "high", "low",
	 model.tapp.parameter.okatom, DTTapp::Parameter::okatom_min, DTTapp::Parameter::okatom_max,
	 this, SLOT(edit(const MyEvent&)), ID_OKATOM );
      widget->setFixedWidth(250);
      box->addWidget(widget,row,1);
      vwidget.push_back(widget);
    }
    row++;

    // uptime
    {
      MyQEditSlider* widget = new MyQEditSlider
	( "uptime", "uptime:", "", "short", "long",
	 model.tapp.parameter.uptime, DTTapp::Parameter::uptime_min, DTTapp::Parameter::uptime_max,
	 this, SLOT(edit(const MyEvent&)), ID_UPTIME );
      widget->setFixedWidth(250);
      box->addWidget(widget,row,0);
      vwidget.push_back(widget);
    }

    group->setLayout(box);
    layout->addWidget(group);

    group->setExpert(true);
    vgroup.push_back(group);
  }

  // misc parameters
  {
    MyQGroupBox* group = new MyQGroupBox("misc");
    QGridLayout* box = new QGridLayout;
    int row=0;

    box->addWidget(new QLabel("niter0:"),row,0,Qt::AlignRight );
    {
      MyQSpinBox* widget = new MyQSpinBox
	( model.tapp.parameter.niter0, DTTapp::Parameter::niter_min, DTTapp::Parameter::niter_max,
	  this, SLOT(edit(const MyEvent&)), ID_MISC );
      widget->setFixedWidth(40);
      box->addWidget(widget,row,1,Qt::AlignLeft);
      vwidget.push_back(widget);
    }

    box->addWidget(new QLabel("niter1:"),row,2,Qt::AlignRight );
    {
      MyQSpinBox* widget = new MyQSpinBox
	( model.tapp.parameter.niter1, DTTapp::Parameter::niter_min, DTTapp::Parameter::niter_max,
	  this, SLOT(edit(const MyEvent&)), ID_MISC );
      widget->setFixedWidth(40);
      box->addWidget(widget,row,3,Qt::AlignLeft);
      vwidget.push_back(widget);
    }
    row++;

    box->addWidget(new QLabel("itsb:"),row,0,Qt::AlignRight );
    {
      MyQSpinBox* widget = new MyQSpinBox
	( model.tapp.parameter.itsb, DTTapp::Parameter::itsb_min, DTTapp::Parameter::itsb_max,
	  this, SLOT(edit(const MyEvent&)), ID_MISC );
      widget->setFixedWidth(40);
      box->addWidget(widget,row,1,Qt::AlignLeft);
      vwidget.push_back(widget);
    }

    box->addWidget(new QLabel("ndiag0:"),row,2,Qt::AlignRight );
    {
      MyQSpinBox* widget = new MyQSpinBox
	( model.tapp.parameter.ndiag0, DTTapp::Parameter::ndiag_min, DTTapp::Parameter::ndiag_max,
	  this, SLOT(edit(const MyEvent&)), ID_MISC );
      widget->setFixedWidth(40);
      box->addWidget(widget,row,3,Qt::AlignLeft);
      vwidget.push_back(widget);
    }
    row++;

    box->addWidget(new QLabel("ndiag1:"),row,0,Qt::AlignRight );
    {
      MyQSpinBox* widget = new MyQSpinBox
	( model.tapp.parameter.ndiag1, DTTapp::Parameter::ndiag_min, DTTapp::Parameter::ndiag_max,
	  this, SLOT(edit(const MyEvent&)), ID_MISC );
      widget->setFixedWidth(40);
      box->addWidget(widget,row,1,Qt::AlignLeft);
      vwidget.push_back(widget);
    }

    box->addWidget(new QLabel("ncycl:"),row,2,Qt::AlignRight );
    {
      MyQSpinBox* widget = new MyQSpinBox
	( model.tapp.parameter.ncycl, DTTapp::Parameter::ncycl_min, DTTapp::Parameter::ncycl_max,
	  this, SLOT(edit(const MyEvent&)), ID_MISC );
      widget->setFixedWidth(40);
      box->addWidget(widget,row,3,Qt::AlignLeft);
      vwidget.push_back(widget);
    }
    row++;

    box->addWidget(new QLabel("mrfr:"),row,0,Qt::AlignRight );
    {
      MyQSpinBox* widget = new MyQSpinBox
	( model.tapp.parameter.mrfr, DTTapp::Parameter::mrfr_min, DTTapp::Parameter::mrfr_max,
	  this, SLOT(edit(const MyEvent&)), ID_MISC );
      widget->setFixedWidth(40);
      box->addWidget(widget,row,1,Qt::AlignLeft);
      vwidget.push_back(widget);
    }

    box->addWidget(new QLabel("most:"),row,2,Qt::AlignRight );
    {
      MyQSpinBox* widget = new MyQSpinBox
	( model.tapp.parameter.most, DTTapp::Parameter::most_min, DTTapp::Parameter::most_max,
	  this, SLOT(edit(const MyEvent&)), ID_MISC );
      widget->setFixedWidth(40);
      box->addWidget(widget,row,3,Qt::AlignLeft);
      vwidget.push_back(widget);
    }
    row++;

    group->setLayout(box);
    layout->addWidget(group);

    group->setExpert(true);
    vgroup.push_back(group);
  }

  setLayout(layout);
  updateExpert(false);
}

void QTTappParameter::edit( const MyEvent& ev )
{
  emit changed(MyEvent(id));
}

void QTTappParameter::updateExpert( const bool expert )
{
  MyQTab::updateExpert(expert);
}

//------------------------------------------------------------

QTTappPolar::QTTappPolar( DTModel& _model, const int id ) : MyQTab(id), model(_model)
{
  QVBoxLayout* layout = new QVBoxLayout(this);
  layout->setSizeConstraint( QLayout::SetFixedSize );

  // kd_pol
  {
    MyQGroupBox* group = new MyQGroupBox("Polarization of element");
    QGridLayout* grid = new QGridLayout;
    int row = 0;

    {
      QGroupBox* group = new QGroupBox("element index to edit");
      QHBoxLayout* box = new QHBoxLayout;

      box->addWidget(new QLabel("No."));
      {
	MyQSpinBox* widget = new MyQSpinBox
	  ( model.tapp.polar.index_kd_shown, 0, 0,
	    this, SLOT(edit(const MyEvent&)), ID_KD_INDEX );
	box->addWidget(widget);
	vwidget.push_back(widget);
	spinbox_index_kd = widget;
      }
      {
	MyQLabel* widget = new MyQLabel("of 0 elements");
	box->addWidget(widget);
	vwidget.push_back(widget);
	label_kd_size = widget;
      }
      group->setLayout(box);

      grid->addWidget(group,row,0,1,4);
    }
    row++;

    grid->addWidget(new QLabel("kd_pol:"),row,0);
    {
      MyQSpinBox* widget = new MyQSpinBox
	( model.tapp.polar.kd_shown.pol, 1, 10,
	  this, SLOT(edit(const MyEvent&)), ID_KD );
      grid->addWidget(widget,row,1);
      vwidget.push_back(widget);
    }
    row++;

    grid->addWidget(new QLabel("pol_kd:"),row,0);
    {
      MyQLineEdit* widget = new MyQLineEdit
	(
	 model.tapp.polar.kd_shown.pol1, 0.0, 1.0,
	 this, SLOT(edit(const MyEvent&)), ID_KD );
      widget->setFixedWidth(80);
      grid->addWidget(widget,row,1);
      vwidget.push_back(widget);
    }
    {
      MyQLineEdit* widget = new MyQLineEdit
	(
	 model.tapp.polar.kd_shown.pol2, 0.0, 1.0,
	 this, SLOT(edit(const MyEvent&)), ID_KD );
      widget->setFixedWidth(80);
      grid->addWidget(widget,row,2);
      vwidget.push_back(widget);
    }
    row++;

    {
      MyQPushButton* widget = new MyQPushButton
	( "add", this, SLOT(edit(const MyEvent&)), ID_KD_ADD );
      widget->setFixedWidth(60);
      grid->addWidget(widget,row,0,1,2);
      vwidget.push_back(widget);
    }
    {
      MyQPushButton* widget = new MyQPushButton
	( "del", this, SLOT(edit(const MyEvent&)), ID_KD_DEL );
      widget->setFixedWidth(60);
      grid->addWidget(widget,row,2,1,2);
      vwidget.push_back(widget);
    }
    row++;

    group->setLayout(grid);
    layout->addWidget(group);

    group_polar_kd = group;

    group->setExpert(true);
    vgroup.push_back(group);
  }


  // it_pol
  {
    MyQGroupBox* group = new MyQGroupBox("Polarization of atom");
    QGridLayout* grid = new QGridLayout;
    int row = 0;

    {
      QGroupBox* group = new QGroupBox("atom index to edit");
      QHBoxLayout* box = new QHBoxLayout;

      box->addWidget(new QLabel("No."));
      {
	MyQSpinBox* widget = new MyQSpinBox
	  ( model.tapp.polar.index_it_shown, 0, 0,
	    this, SLOT(edit(const MyEvent&)), ID_IT_INDEX );
	box->addWidget(widget);
	vwidget.push_back(widget);
	spinbox_index_it = widget;
      }
      {
	MyQLabel* widget = new MyQLabel("of 0 elements");
	box->addWidget(widget);
	vwidget.push_back(widget);
	label_it_size = widget;
      }
      group->setLayout(box);

      grid->addWidget(group,row,0,1,4);
    }
    row++;

    grid->addWidget(new QLabel("it_pol:"),row,0);
    {
      MyQSpinBox* widget = new MyQSpinBox
	( model.tapp.polar.it_shown.pol, 1, 10,
	  this, SLOT(edit(const MyEvent&)), ID_IT );
      grid->addWidget(widget,row,1);
      vwidget.push_back(widget);
    }
    row++;

    grid->addWidget(new QLabel("pol_asi:"),row,0);
    {
      MyQLineEdit* widget = new MyQLineEdit
	(
	 model.tapp.polar.it_shown.asi1, 0.0, 1.0,
	 this, SLOT(edit(const MyEvent&)), ID_IT );
      widget->setFixedWidth(80);
      grid->addWidget(widget,row,1);
      vwidget.push_back(widget);
    }
    {
      MyQLineEdit* widget = new MyQLineEdit
	(
	 model.tapp.polar.it_shown.asi2, 0.0, 1.0,
	 this, SLOT(edit(const MyEvent&)), ID_IT );
      widget->setFixedWidth(80);
      grid->addWidget(widget,row,2);
      vwidget.push_back(widget);
    }
    row++;

    {
      MyQPushButton* widget = new MyQPushButton
	( "add", this, SLOT(edit(const MyEvent&)), ID_IT_ADD );
      widget->setFixedWidth(60);
      grid->addWidget(widget,row,0,1,2);
      vwidget.push_back(widget);
    }
    {
      MyQPushButton* widget = new MyQPushButton
	( "del", this, SLOT(edit(const MyEvent&)), ID_IT_DEL );
      widget->setFixedWidth(60);
      grid->addWidget(widget,row,2,1,2);
      vwidget.push_back(widget);
    }
    row++;

    group->setLayout(grid);
    layout->addWidget(group);

    group_polar_it = group;

    group->setExpert(true);
    vgroup.push_back(group);
  }

  setLayout(layout);
  updateExpert(false);
}

void QTTappPolar::edit( const MyEvent& ev )
{
  switch(ev.id){
  case ID_KD_INDEX : {
    model.tapp.polar.update_shown();
  } break;
  case ID_KD : {
    model.tapp.polar.change_shown();
  } break;
  case ID_KD_ADD : {
    model.tapp.polar.addKD();
  } break;
  case ID_KD_DEL : {
    model.tapp.polar.delKD();
  } break;

  case ID_IT_INDEX : {
    model.tapp.polar.update_shown();
  } break;
  case ID_IT : {
    model.tapp.polar.change_shown();
  } break;
  case ID_IT_ADD : {
    model.tapp.polar.addIT();
  } break;
  case ID_IT_DEL : {
    model.tapp.polar.delIT();
  } break;

  default : break;
  }

  emit changed(MyEvent(id));
}

void QTTappPolar::updateExpert( const bool expert )
{
  model.tapp.polar.update_shown();
  spinbox_index_kd->setRange( 1, model.tapp.polar.size_shown_kd() );
  spinbox_index_it->setRange( 1, model.tapp.polar.size_shown_it() );
  {
    char text[32];
    snprintf( text, sizeof(text), "of %d elements",
	      model.tapp.polar.size_shown_kd() );
    label_kd_size->setText(text);
    snprintf( text, sizeof(text), "of %d elements",
	      model.tapp.polar.size_shown_it() );
    label_it_size->setText(text);
  }

  /*
  if( model.tapp.polar.size_shown_kd() == 0 ){
    group_polar_kd->setEnabled(false);
  }else{
    group_polar_kd->setEnabled(true);
  }
  if( model.tapp.polar.size_shown_it() == 0 ){
    group_polar_it->setEnabled(false);
  }else{
    group_polar_it->setEnabled(true);
  }
  */

  MyQTab::updateExpert(expert);
}


//------------------------------------------------------------

QTTappVBPEF::QTTappVBPEF( DTModel& _model, const int id ) : MyQTab(id), model(_model)
{
  QVBoxLayout* layout = new QVBoxLayout(this);
  layout->setSizeConstraint( QLayout::SetFixedSize );

  /*
  {
    MyQGroupBox* group = new MyQGroupBox("K-points");
    QGridLayout* grid = new QGridLayout;
    int row=0;

    {
      QGroupBox* group = new QGroupBox("kpoint index to edit");
      QHBoxLayout* box = new QHBoxLayout;

      box->addWidget(new QLabel("No."), Qt::AlignRight );
      {
	MyQSpinBox* widget = new MyQSpinBox
	  ( model.tapp.vbpef.index_shown, 0, 0,
	    this, SLOT(edit(const MyEvent&)), ID_KPOINT_INDEX );
	box->addWidget(widget);
	vwidget.push_back(widget);
	spinbox_index = widget;
      }
      {
	MyQLabel* widget = new MyQLabel("of 0 elements");
	box->addWidget(widget);
	vwidget.push_back(widget);
	label_size = widget;
      }
      group->setLayout(box);

      grid->addWidget(group,row,0,1,4);
    }
    row++;

    grid->addWidget(new QLabel("nkpt:"),row,0);
    {
      MyQLineEdit* widget = new MyQLineEdit
	(
	 model.tapp.vbpef.band_shown.nkpt,
	 this, SLOT(edit(const MyEvent&)), ID_KPOINT );
      widget->setFixedWidth(60);
      grid->addWidget(widget,row,1);
      vwidget.push_back(widget);
    }
    row++;

    grid->addWidget(new QLabel("ak:"),row,0);
    for( int j=0; j<3; j++ ){
      MyQLineEdit* widget = new MyQLineEdit
	(
	 model.tapp.vbpef.band_shown.ak(j), 0.0, 1.0,
	 this, SLOT(edit(const MyEvent&)), ID_KPOINT );
      widget->setFixedWidth(80);
      grid->addWidget(widget,row,j+1);
      vwidget.push_back(widget);
    }
    row++;

    grid->addWidget(new QLabel("nkfi:"),row,0);
    {
      MyQSpinBox* widget = new MyQSpinBox
	( model.tapp.vbpef.band_shown.nkfi, 0, 100,
	  this, SLOT(edit(const MyEvent&)), ID_KPOINT );
      widget->setFixedWidth(60);
      grid->addWidget(widget,row,1);
      vwidget.push_back(widget);
    }
    row++;

    {
      MyQPushButton* widget = new MyQPushButton
	( "add", this, SLOT(edit(const MyEvent&)), ID_KPOINT_ADD );
      widget->setFixedWidth(60);
      grid->addWidget(widget,row,1);
      vwidget.push_back(widget);
    }
    {
      MyQPushButton* widget = new MyQPushButton
	( "del", this, SLOT(edit(const MyEvent&)), ID_KPOINT_DEL );
      widget->setFixedWidth(60);
      grid->addWidget(widget,row,2);
      vwidget.push_back(widget);
    }
    row++;

    group->setLayout(grid);
    layout->addWidget(group);

    group_kpoint = group;

    group->setExpert(false);
    vgroup.push_back(group);
  }
  */

  // 
  {
    MyQGroupBox* group = new MyQGroupBox("VBPEF");
    QGridLayout* grid = new QGridLayout;
    int row=0;

    grid->addWidget(new QLabel("ifbunp:"),row,0, Qt::AlignRight );
    {
      MyQSpinBox* widget = new MyQSpinBox
	( model.tapp.vbpef.ifbunp, DTTapp::VBPEF::ifbunp_min, DTTapp::VBPEF::ifbunp_max,
	  this, SLOT(edit(const MyEvent&)), ID_VBPEF );
      widget->setFixedWidth(80);
      grid->addWidget(widget,row,1);
      vwidget.push_back(widget);
    }

    grid->addWidget(new QLabel("ifbfix:"),row,2, Qt::AlignRight );
    {
      MyQSpinBox* widget = new MyQSpinBox
	( model.tapp.vbpef.ifbfix, DTTapp::VBPEF::ifbfix_min, DTTapp::VBPEF::ifbfix_max,
	  this, SLOT(edit(const MyEvent&)), ID_VBPEF );
      widget->setFixedWidth(80);
      grid->addWidget(widget,row,3);
      vwidget.push_back(widget);
    }
    row++;


    grid->addWidget(new QLabel("nw:"),row,0, Qt::AlignRight );
    {
      MyQSpinBox* widget = new MyQSpinBox
	( model.tapp.vbpef.nw, DTTapp::VBPEF::nw_min, DTTapp::VBPEF::nw_max,
	  this, SLOT(edit(const MyEvent&)), ID_VBPEF );
      widget->setFixedWidth(80);
      grid->addWidget(widget,row,1);
      vwidget.push_back(widget);
    }

    grid->addWidget(new QLabel("nb2:"),row,2, Qt::AlignRight );
    {
      MyQSpinBox* widget = new MyQSpinBox
	( model.tapp.vbpef.nb2, DTTapp::VBPEF::nb2_min, DTTapp::VBPEF::nb2_max,
	  this, SLOT(edit(const MyEvent&)), ID_VBPEF );
      grid->addWidget(widget,row,3);
      vwidget.push_back(widget);
    }
    row++;

    grid->addWidget(new QLabel("itsb:"),row,0, Qt::AlignRight );
    {
      MyQSpinBox* widget = new MyQSpinBox
	( model.tapp.vbpef.itsb, DTTapp::VBPEF::itsb_min, DTTapp::VBPEF::itsb_max,
	  this, SLOT(edit(const MyEvent&)), ID_VBPEF );
      widget->setFixedWidth(80);
      grid->addWidget(widget,row,1);
      vwidget.push_back(widget);
    }

    grid->addWidget(new QLabel("ndiag:"),row,2, Qt::AlignRight );
    {
      MyQSpinBox* widget = new MyQSpinBox
	( model.tapp.vbpef.ndiag, DTTapp::VBPEF::ndiag_min, DTTapp::VBPEF::ndiag_max,
	  this, SLOT(edit(const MyEvent&)), ID_VBPEF );
      widget->setFixedWidth(80);
      grid->addWidget(widget,row,3);
      vwidget.push_back(widget);
    }
    row++;

    grid->addWidget(new QLabel("eps:"),row,0, Qt::AlignRight);
    {
      MyQLineEdit* widget = new MyQLineEdit
	(
	 model.tapp.vbpef.eps, DTTapp::VBPEF::eps_min, DTTapp::VBPEF::eps_max,
	 this, SLOT(edit(const MyEvent&)), ID_VBPEF );
      widget->setFixedWidth(80);
      grid->addWidget(widget,row,1);
      vwidget.push_back(widget);
    }

    grid->addWidget(new QLabel("ndiv:"),row,2, Qt::AlignRight );
    {
      MyQSpinBox* widget = new MyQSpinBox
	( model.tapp.vbpef.ndiv, DTTapp::VBPEF::ndiv_min, DTTapp::VBPEF::ndiv_max,
	  this, SLOT(edit(const MyEvent&)), ID_VBPEF );
      widget->setFixedWidth(80);
      grid->addWidget(widget,row,3);
      vwidget.push_back(widget);
    }
    row++;

    grid->addWidget(new QLabel("nbba1:"),row,0, Qt::AlignRight );
    {
      MyQSpinBox* widget = new MyQSpinBox
	( model.tapp.vbpef.nbba1, DTTapp::VBPEF::nbba_min, DTTapp::VBPEF::nbba_max,
	  this, SLOT(edit(const MyEvent&)), ID_VBPEF );
      widget->setFixedWidth(80);
      grid->addWidget(widget,row,1);
      vwidget.push_back(widget);
    }

    grid->addWidget(new QLabel("nbba2:"),row,2, Qt::AlignRight );
    {
      MyQSpinBox* widget = new MyQSpinBox
	( model.tapp.vbpef.nbba2, DTTapp::VBPEF::nbba_min, DTTapp::VBPEF::nbba_max,
	  this, SLOT(edit(const MyEvent&)), ID_VBPEF );
      widget->setFixedWidth(80);
      grid->addWidget(widget,row,3);
      vwidget.push_back(widget);
    }
    row++;

    grid->addWidget(new QLabel("nsph:"),row,0, Qt::AlignRight );
    {
      MyQSpinBox* widget = new MyQSpinBox
	( model.tapp.vbpef.nsph, DTTapp::VBPEF::nsph_min, DTTapp::VBPEF::nsph_max,
	  this, SLOT(edit(const MyEvent&)), ID_VBPEF );
      widget->setFixedWidth(80);
      grid->addWidget(widget,row,1);
      vwidget.push_back(widget);
    }

    grid->addWidget(new QLabel("iaxis:"),row,2, Qt::AlignRight );
    {
      MyQSpinBox* widget = new MyQSpinBox
	( model.tapp.vbpef.iaxis, DTTapp::VBPEF::iaxis_min, DTTapp::VBPEF::iaxis_max,
	  this, SLOT(edit(const MyEvent&)), ID_VBPEF );
      widget->setFixedWidth(80);
      grid->addWidget(widget,row,3);
      vwidget.push_back(widget);
    }
    row++;

    grid->addWidget(new QLabel("rmin:"),row,0, Qt::AlignRight);
    {
      MyQLineEdit* widget = new MyQLineEdit
	(
	 model.tapp.vbpef.rmin, DTTapp::VBPEF::rmin_min, DTTapp::VBPEF::rmin_max,
	 this, SLOT(edit(const MyEvent&)), ID_VBPEF );
      widget->setFixedWidth(80);
      grid->addWidget(widget,row,1);
      vwidget.push_back(widget);
    }

    grid->addWidget(new QLabel("rmax:"),row,2, Qt::AlignRight);
    {
      MyQLineEdit* widget = new MyQLineEdit
	(
	 model.tapp.vbpef.rmax, DTTapp::VBPEF::rmax_min, DTTapp::VBPEF::rmax_max,
	 this, SLOT(edit(const MyEvent&)), ID_VBPEF );
      widget->setFixedWidth(80);
      grid->addWidget(widget,row,3);
      vwidget.push_back(widget);
    }
    row++;

    grid->addWidget(new QLabel("rzero:"),row,0, Qt::AlignRight);
    for( int j=0; j<3; j++ ){
      MyQLineEdit* widget = new MyQLineEdit
	(
	 model.tapp.vbpef.rzero(j), DTTapp::VBPEF::rzero_min, DTTapp::VBPEF::rzero_max,
	 this, SLOT(edit(const MyEvent&)), ID_VBPEF );
      widget->setFixedWidth(80);
      grid->addWidget(widget,row,j+1);
      vwidget.push_back(widget);
    }
    row++;

    group->setLayout(grid);
    layout->addWidget(group);

    group->setExpert(true);
    vgroup.push_back(group);
  }

  setLayout(layout);
  updateExpert(false);
}

void QTTappVBPEF::edit( const MyEvent& ev )
{
  switch(ev.id){
  case ID_KPOINT_INDEX : {
    //    model.tapp.vbpef.update_shown();
  } break;
  case ID_KPOINT : {
    //    model.tapp.vbpef.change_shown();
  } break;
  case ID_KPOINT_ADD : {
    //    model.tapp.vbpef.add();
  } break;
  case ID_KPOINT_DEL : {
    //    model.tapp.vbpef.del();
  } break;

  case ID_VBPEF : {
    //    model.tapp.vbpef.change_shown();
  } break;

  default : break;
  }

  emit changed(MyEvent(id));
}

void QTTappVBPEF::updateExpert( const bool expert )
{
  /*
  model.tapp.vbpef.update_shown();
  spinbox_index->setRange( 1, model.tapp.vbpef.size_shown() );
  {
    char text[32];
    snprintf( text, sizeof(text), "of %d elements",
	      model.tapp.vbpef.size_shown() );
    label_size->setText(text);
  }
  */

  /*
  if( model.tapp.vbpef.size_shown() == 0 ){
    group_kpoint->setEnabled(false);
  }else{
    group_kpoint->setEnabled(true);
  }
  */

  MyQTab::updateExpert(expert);
}


//------------------------------------------------------------

QTTappVBWFN::QTTappVBWFN( DTModel& _model, const int id ) : MyQTab(id), model(_model)
{
  QVBoxLayout* layout = new QVBoxLayout(this);
  layout->setSizeConstraint( QLayout::SetFixedSize );

  // VBWFN
  {
    MyQGroupBox* group = new MyQGroupBox("VBWFN");
    QVBoxLayout* box = new QVBoxLayout;

    {
      QHBoxLayout* hbox = new QHBoxLayout;

      hbox->addWidget(new QLabel("nran:"),Qt::AlignRight);
      {
	MyQSpinBox* widget = new MyQSpinBox
	  ( model.tapp.vbwfn.nraxn, 0, 1000,
	    this, SLOT(edit(const MyEvent&)), ID_VBWFN );
	widget->setFixedWidth(80);
	hbox->addWidget(widget);
	vwidget.push_back(widget);
      }
      {
	MyQSpinBox* widget = new MyQSpinBox
	  ( model.tapp.vbwfn.nrayn, 0, 1000,
	    this, SLOT(edit(const MyEvent&)), ID_VBWFN );
	widget->setFixedWidth(80);
	hbox->addWidget(widget);
	vwidget.push_back(widget);
      }
      {
	MyQSpinBox* widget = new MyQSpinBox
	  ( model.tapp.vbwfn.nrazn, 0, 1000,
	    this, SLOT(edit(const MyEvent&)), ID_VBWFN );
	widget->setFixedWidth(80);
	hbox->addWidget(widget);
	vwidget.push_back(widget);
      }
      box->addLayout(hbox);
    }

    {
      MyQCheckBox* widget = new MyQCheckBox
	( "sqamp", model.tapp.vbwfn.sqamp,
	  this, SLOT(edit(const MyEvent&)), ID_VBWFN );
      box->addWidget(widget);
    }
    {
      MyQCheckBox* widget = new MyQCheckBox
	( "wfn", model.tapp.vbwfn.wfn,
	  this, SLOT(edit(const MyEvent&)), ID_VBWFN );
      box->addWidget(widget);
    }

    {
      QHBoxLayout* hbox = new QHBoxLayout;

      hbox->addWidget(new QLabel("sk:"));
      {
	MyQLineEdit* widget = new MyQLineEdit
	(
	 model.tapp.vbwfn.sk[0], -1.0, +1.0,
	 this, SLOT(edit(const MyEvent&)), ID_VBWFN );
	widget->setFixedWidth(80);
	hbox->addWidget(widget);
	vwidget.push_back(widget);
      }
      {
	MyQLineEdit* widget = new MyQLineEdit
	(
	 model.tapp.vbwfn.sk[1], -1.0, +1.0,
	 this, SLOT(edit(const MyEvent&)), ID_VBWFN );
	widget->setFixedWidth(80);
	hbox->addWidget(widget);
	vwidget.push_back(widget);
      }
      {
	MyQLineEdit* widget = new MyQLineEdit
	(
	 model.tapp.vbwfn.sk[2], -1.0, +1.0,
	 this, SLOT(edit(const MyEvent&)), ID_VBWFN );
	widget->setFixedWidth(80);
	hbox->addWidget(widget);
	vwidget.push_back(widget);
      }
      box->addLayout(hbox);
    }
    

    {
      QGroupBox* group = new QGroupBox("nbb array");
      QGridLayout* grid = new QGridLayout;
      int row=0;

      grid->addWidget(new QLabel("index:"),row,0);
      {
	MyQSpinBox* widget = new MyQSpinBox
	  ( model.tapp.vbwfn.index_shown, 0, 0,
	    this, SLOT(edit(const MyEvent&)), ID_NB_INDEX );
	grid->addWidget(widget,row,1);
	vwidget.push_back(widget);
	spinbox_index = widget;
      }
      {
	MyQLabel* widget = new MyQLabel("of 0 elements");
	grid->addWidget(widget,row,2);
	vwidget.push_back(widget);
	label_size = widget;
      }
      row++;

      grid->addWidget(new QLabel("nbb[]:"),row,0);

      {
	MyQSpinBox* widget = new MyQSpinBox
	  ( model.tapp.vbwfn.nb_shown.nbb1, 0, 100,
	  this, SLOT(edit(const MyEvent&)), ID_NB );
	grid->addWidget(widget,row,1);
	vwidget.push_back(widget);
      }
      {
	MyQSpinBox* widget = new MyQSpinBox
	  ( model.tapp.vbwfn.nb_shown.nbb2, 0, 100,
	  this, SLOT(edit(const MyEvent&)), ID_NB );
	grid->addWidget(widget,row,2);
	vwidget.push_back(widget);
      }
      row++;

      {
	MyQPushButton* widget = new MyQPushButton
	  ( "add", this, SLOT(edit(const MyEvent&)), ID_NB_ADD );
	widget->setFixedWidth(60);
	grid->addWidget(widget,row,1);
	vwidget.push_back(widget);
      }
      {
	MyQPushButton* widget = new MyQPushButton
	  ( "del", this, SLOT(edit(const MyEvent&)), ID_NB_DEL );
	widget->setFixedWidth(60);
	grid->addWidget(widget,row,2);
	vwidget.push_back(widget);
      }
      row++;

      group->setLayout(grid);
      box->addWidget(group);
    }


    {
      QHBoxLayout* hbox = new QHBoxLayout;

      hbox->addWidget(new QLabel("itsb:"),Qt::AlignRight);
      {
	MyQSpinBox* widget = new MyQSpinBox
	  ( model.tapp.vbwfn.itsb, 0, 10,
	    this, SLOT(edit(const MyEvent&)), ID_VBWFN );
	widget->setFixedWidth(80);
	hbox->addWidget(widget,Qt::AlignLeft);
	vwidget.push_back(widget);
      }
      box->addLayout(hbox);
    }
    {
      QHBoxLayout* hbox = new QHBoxLayout;
      hbox->addWidget(new QLabel("ndiag:"),Qt::AlignRight);
      {
	MyQSpinBox* widget = new MyQSpinBox
	  ( model.tapp.vbwfn.ndiag, 0, 100,
	    this, SLOT(edit(const MyEvent&)), ID_VBWFN );
	widget->setFixedWidth(80);
	hbox->addWidget(widget,Qt::AlignLeft);
	vwidget.push_back(widget);
      }
      box->addLayout(hbox);
    }

    {
      QHBoxLayout* hbox = new QHBoxLayout;

      hbox->addWidget(new QLabel("eps:"),Qt::AlignRight);
      {
	MyQLineEdit* widget = new MyQLineEdit
	(
	 model.tapp.vbwfn.eps, 0.0, 1.0,
	 this, SLOT(edit(const MyEvent&)), ID_VBWFN );
	widget->setFixedWidth(100);
	hbox->addWidget(widget,Qt::AlignLeft);
	vwidget.push_back(widget);
      }
      box->addLayout(hbox);
    }

    group->setLayout(box);
    layout->addWidget(group);

    group->setExpert(true);
    vgroup.push_back(group);
  }

  setLayout(layout);
  updateExpert(false);
}

void QTTappVBWFN::edit( const MyEvent& ev )
{
  switch(ev.id){
  case ID_NB_INDEX : {
    model.tapp.vbwfn.update_shown();
  } break;
  case ID_NB : {
    model.tapp.vbwfn.change_shown();
  } break;
  case ID_NB_ADD : {
    model.tapp.vbwfn.add();
  } break;
  case ID_NB_DEL : {
    model.tapp.vbwfn.del();
  } break;
  }
  emit changed(MyEvent(id));
}

void QTTappVBWFN::updateExpert( const bool expert )
{
  model.tapp.vbwfn.update_shown();
  spinbox_index->setRange( 1, model.tapp.vbwfn.size_shown() );
  {
    char text[32];
    snprintf( text, sizeof(text), "of %d elements",
	      model.tapp.vbwfn.size_shown() );
    label_size->setText(text);
  }

  MyQTab::updateExpert(expert);
}

//------------------------------------------------------------

QTTappTCHRPOT::QTTappTCHRPOT( DTModel& _model, const int id ) : MyQTab(id), model(_model)
{
  QVBoxLayout* layout = new QVBoxLayout(this);
  layout->setSizeConstraint( QLayout::SetFixedSize );

  // TCHRPOT
  {
    MyQGroupBox* group = new MyQGroupBox("TCHRPOT");
    QVBoxLayout* box = new QVBoxLayout;

    {
      QHBoxLayout* hbox = new QHBoxLayout;

      hbox->addWidget(new QLabel("nran:"));
      {
	MyQSpinBox* widget = new MyQSpinBox
	  ( model.tapp.tchrpot.nraxn, 0, 10000,
	    this, SLOT(edit(const MyEvent&)), ID_TCHRPOT );
	widget->setFixedWidth(80);
	hbox->addWidget(widget);
	vwidget.push_back(widget);
      }
      {
	MyQSpinBox* widget = new MyQSpinBox
	  ( model.tapp.tchrpot.nrayn, 0, 10000,
	    this, SLOT(edit(const MyEvent&)), ID_TCHRPOT );
	widget->setFixedWidth(80);
	hbox->addWidget(widget);
	vwidget.push_back(widget);
      }
      {
	MyQSpinBox* widget = new MyQSpinBox
	  ( model.tapp.tchrpot.nrazn, 0, 10000,
	    this, SLOT(edit(const MyEvent&)), ID_TCHRPOT );
	widget->setFixedWidth(80);
	hbox->addWidget(widget);
	vwidget.push_back(widget);
      }
      box->addLayout(hbox);
    }

    {
      MyQCheckBox* widget = new MyQCheckBox
	( "tchr", model.tapp.tchrpot.tchr,
	  this, SLOT(edit(const MyEvent&)), ID_TCHRPOT );
      box->addWidget(widget);
    }
    {
      MyQCheckBox* widget = new MyQCheckBox
	( "lpot", model.tapp.tchrpot.lpot,
	  this, SLOT(edit(const MyEvent&)), ID_TCHRPOT );
      box->addWidget(widget);
    }


    group->setLayout(box);
    layout->addWidget(group);

    group->setExpert(true);
    vgroup.push_back(group);
  }

  setLayout(layout);
  updateExpert(false);
}

void QTTappTCHRPOT::edit( const MyEvent& ev )
{
  emit changed(MyEvent(id));
}

void QTTappTCHRPOT::updateExpert( const bool expert )
{
  MyQTab::updateExpert(expert);
}


//--------------------------------------------------------

QTTapp::QTTapp( DTModel& _model, const int _id ) :
  model(_model), id(_id)
{
  QVBoxLayout* layout = new QVBoxLayout(this);
  layout->setSizeConstraint( QLayout::SetFixedSize );

  setFixedSize(380,500);
  setWindowTitle("TAPP settings");

  {
    QTabWidget* group = new QTabWidget;
    {
      QTTappOption* widget = new QTTappOption( model, ID_OPTION );
      group->addTab(widget, "Option");
      vwidget.push_back(widget);
      connect(widget, SIGNAL(changed(const MyEvent&)), this, SLOT(edit(const MyEvent&)) );
    }
    {
      QTTappLSDA* widget = new QTTappLSDA( model, ID_LSDA );
      group->addTab(widget, "LSDA");
      vwidget.push_back(widget);
      connect(widget, SIGNAL(changed(const MyEvent&)), this, SLOT(edit(const MyEvent&)) );
    }
    {
      QTTappCutoff* widget = new QTTappCutoff( model, ID_CUTOFF );
      group->addTab(widget, "Cutoff");
      vwidget.push_back(widget);
      connect(widget, SIGNAL(changed(const MyEvent&)), this, SLOT(edit(const MyEvent&)) );
    }
    {
      QTTappMesh* widget = new QTTappMesh( model, ID_MESH );
      group->addTab(widget, "Mesh");
      vwidget.push_back(widget);
      connect(widget, SIGNAL(changed(const MyEvent&)), this, SLOT(edit(const MyEvent&)) );
    }

    {
      QTTappEwald* widget = new QTTappEwald( model, ID_EWALD );
      group->addTab(widget, "Ewald");
      vwidget.push_back(widget);
      connect(widget, SIGNAL(changed(const MyEvent&)), this, SLOT(edit(const MyEvent&)) );
    }
    {
      QTTappSampling* widget = new QTTappSampling( model, ID_SAMPLING );
      group->addTab(widget, "Sampling");
      vwidget.push_back(widget);
      connect(widget, SIGNAL(changed(const MyEvent&)), this, SLOT(edit(const MyEvent&)) );
    }
    {
      QTTappFile* widget = new QTTappFile( model, ID_FILE );
      group->addTab(widget, "File");
      vwidget.push_back(widget);
      connect(widget, SIGNAL(changed(const MyEvent&)), this, SLOT(edit(const MyEvent&)) );
    }
    {
      QTTappMass* widget = new QTTappMass( model, ID_MASS );
      group->addTab(widget, "Mass");
      vwidget.push_back(widget);
      connect(widget, SIGNAL(changed(const MyEvent&)), this, SLOT(edit(const MyEvent&)) );
    }
    {
      QTTappParameter* widget = new QTTappParameter( model, ID_PARAMETER );
      group->addTab(widget, "Parameter");
      vwidget.push_back(widget);
      connect(widget, SIGNAL(changed(const MyEvent&)), this, SLOT(edit(const MyEvent&)) );
    }
    {
      QTTappPolar* widget = new QTTappPolar( model, ID_POLAR );
      group->addTab(widget, "Polar");
      vwidget.push_back(widget);
      connect(widget, SIGNAL(changed(const MyEvent&)), this, SLOT(edit(const MyEvent&)) );
    }
    {
      QTTappVBPEF* widget = new QTTappVBPEF( model, ID_VBPEF );
      group->addTab(widget, "VBPEF");
      vwidget.push_back(widget);
      connect(widget, SIGNAL(changed(const MyEvent&)), this, SLOT(edit(const MyEvent&)) );
    }
    {
      QTTappVBWFN* widget = new QTTappVBWFN( model, ID_VBWFN );
      group->addTab(widget, "VBWFN");
      vwidget.push_back(widget);
      connect(widget, SIGNAL(changed(const MyEvent&)), this, SLOT(edit(const MyEvent&)) );
    }
    {
      QTTappTCHRPOT* widget = new QTTappTCHRPOT( model, ID_TCHRPOT );
      group->addTab(widget, "TCHRPOT");
      vwidget.push_back(widget);
      connect(widget, SIGNAL(changed(const MyEvent&)), this, SLOT(edit(const MyEvent&)) );
    }

    layout->addWidget(group);
  }

  QHBoxLayout* box = new QHBoxLayout;
  {
    MyQPushButton* widget = new MyQPushButton
      ( "close", this, SLOT(edit(const MyEvent&)), ID_CLOSE );
    box->addWidget( widget );
  }
  {
    MyQPushButton* widget = new MyQPushButton
      ( "undo", this, SLOT(edit(const MyEvent&)), ID_UNDO );
    box->addWidget( widget );
  }
  {
    expert = false;
    MyQToggleButton* widget = new MyQToggleButton
      ( "expert", expert, this, SLOT(edit(const MyEvent&)), ID_EXPERT );
    box->addWidget( widget );
  }

  layout->addLayout(box);

  setLayout(layout);
}

void QTTapp::edit( const MyEvent& ev )
{
  switch(ev.id){
  case ID_OPTION :
  case ID_LSDA :
  case ID_CUTOFF :
  case ID_MESH :
  case ID_EWALD :
  case ID_SAMPLING :
  case ID_FILE :
  case ID_MASS :
  case ID_PARAMETER :
  case ID_POLAR :
  case ID_VBPEF : 
  case ID_VBWFN :
  case ID_TCHRPOT :{
    emit changed();
  } break;


  case ID_CLOSE : {
    hide();
  } break;
  case ID_UNDO : {
    model.undo();
  } break;
  case ID_EXPERT : {
    update();
  } break;
  default: break;
  }
}

void QTTapp::update( void )
{
  for( int i=0; i<(int)vwidget.size(); i++ ){
    vwidget[i]->updateExpert(expert);
  }
}
