/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

/*!
 \file qtwidgets.cc
 \brief t@\GUIiNX
*/

#include <math.h>
#include "qtwidgets.h"
#include "qtmisc.h"

MyQAction::MyQAction
( const char* text,
  QObject* target, const char* slot, const int _id ) :
  QAction(text,target),
  id(_id)
{
  connect(this, SIGNAL(triggered()), this, SLOT(convert()));
  connect(this, SIGNAL(valueChanged(const MyEvent&)), target, slot );
}

void MyQAction::convert( void )
{
  emit valueChanged(MyEvent(id));
}

MyQLabel::MyQLabel( const char* text ) :
  QLabel(text)
{
}

void MyQLabel::update( void )
{
  // nothing to do
}


MyQPushButton::MyQPushButton
( const char* text,
  QWidget* target, const char* slot, const int _id ) :
  id(_id)
{
  setText(text);
  connect(this, SIGNAL(clicked()), this, SLOT(convert()));
  connect(this, SIGNAL(valueChanged(const MyEvent&)), target, slot );

  update();
}

void MyQPushButton::update( void )
{
  // nothing to do
}

void MyQPushButton::convert( void )
{
  emit valueChanged(MyEvent(id));
}

MyQToggleButton::MyQToggleButton
( const char* text,
  bool& _value,
  QWidget* target, const char* slot, const int _id ) :
  value(_value), id(_id)
{
  setCheckable(true);
  setText(text);

  connect(this, SIGNAL(toggled(bool)), this, SLOT(convert(bool)));
  connect(this, SIGNAL(valueChanged(const MyEvent&)), target, slot );

  update();
}

void MyQToggleButton::convert( bool checked )
{
  value = checked;
  emit valueChanged(MyEvent(id));
}

void MyQToggleButton::update( void )
{
  blockSignals(true);
  setChecked( value );
  blockSignals(false);
}


MyQCheckBox::MyQCheckBox
( const char* text, bool& _value, 
  QWidget* target, const char* slot, const int _id ) :
  QCheckBox(target),
  value(_value), id(_id)
{
  setText(text);

  connect(this, SIGNAL(stateChanged(int)), this, SLOT(convert(int)));
  connect(this, SIGNAL(valueChanged(const MyEvent&)), target, slot );

  update();
}

void MyQCheckBox::convert( int i )
{
  value = (i==Qt::Checked);
  emit valueChanged(MyEvent(id));
}

void MyQCheckBox::update( void )
{
  blockSignals(true);
  setCheckState( value ? Qt::Checked : Qt::Unchecked );
  blockSignals(false);
}


MyQTriCheckBox::MyQTriCheckBox
( const char* text, int& _value, 
  QWidget* target, const char* slot, const int _id ) :
  QCheckBox(target),
  value(_value), id(_id)
{
  setTristate(true);
  setText(text);

  connect(this, SIGNAL(stateChanged(int)), this, SLOT(convert(int)));
  connect(this, SIGNAL(valueChanged(const MyEvent&)), target, slot );

  update();
}

void MyQTriCheckBox::convert( int i )
{
  value = i;
  emit valueChanged(MyEvent(id));
}

void MyQTriCheckBox::update( void )
{
  blockSignals(true);
  switch( value ){
  case 0 : setCheckState(Qt::Unchecked); break;
  case 1 : setCheckState(Qt::PartiallyChecked); break;
  case 2 : setCheckState(Qt::Checked); break;
  }
  blockSignals(false);
}



MyQSlider::MyQSlider
( double& _value, const double _min, const double _max,
  QWidget* target, const char* slot, const int _id ) :
  QSlider( Qt::Horizontal ),
  value(_value), min(_min), max(_max), id(_id)
{
  logscale = max>0.0 && min>0.0 && fabs(log10(max/min)) > 2.99;

  connect(this, SIGNAL(valueChanged(int)), this, SLOT(convert(int)));
  connect(this, SIGNAL(valueChanged(const MyEvent&)), target, slot );

  update();
}

void MyQSlider::setRange( const double min, const double max )
{
  this->min = min;
  this->max = max;
  if( value < min ){
    value = min;
    emit valueChanged(MyEvent(id));
  }
  if( value > max ){
    value = max;
    emit valueChanged(MyEvent(id));
  }
}


void MyQSlider::convert( int i )
{
  if( i==99 ) i++;

  if( logscale ){
    value = pow( 10.0, (double)i/100.0*log10(max/min) )*min;
  }else{
    value = (double)i/100.0 * (max-min) + min;
  }

  emit valueChanged(MyEvent(id));
}


void MyQSlider::update( void )
{
  blockSignals(true);

  if( logscale ){
    setValue( int(100.0*log10(value/min)/log10(max/min)) );
  }else{
    if( min==max ){
      setValue( 50 );
    }else{
      setValue( int(100.0*(value-min)/(max-min)) );
    }
  }
  blockSignals(false);
}

MyQRadioButton::MyQRadioButton
( const char* name, int& _target_ivalue, const int _this_ivalue,
  QWidget* target, const char* slot, const int _id ) :
  QRadioButton(name),
  target_ivalue(_target_ivalue), this_ivalue(_this_ivalue),
  sdummy(""), target_svalue(sdummy), this_svalue(sdummy),
  id(_id)
{
  connect(this, SIGNAL(clicked()), this, SLOT(convert()));
  connect(this, SIGNAL(valueChanged(const MyEvent&)), target, slot );

  update();
}

MyQRadioButton::MyQRadioButton
( const char* name, QString& _target_svalue,
  QWidget* target, const char* slot, const int _id ) :
  QRadioButton(name),
  idummy(0), target_ivalue(idummy), this_ivalue(idummy),
  target_svalue(_target_svalue), this_svalue(name),
  id(_id)
{
  connect(this, SIGNAL(clicked()), this, SLOT(convert()));
  connect(this, SIGNAL(valueChanged(const MyEvent&)), target, slot );

  update();
}

void MyQRadioButton::convert( void )
{
  if( target_svalue == sdummy ){
    if( target_ivalue != this_ivalue ){
      target_ivalue = this_ivalue;
      emit valueChanged(MyEvent(id));
    }
  }
  else{
    if( target_svalue != this_svalue ){
      target_svalue = this_svalue;
      emit valueChanged(MyEvent(id));
    }
  }
}

void MyQRadioButton::update( void )
{
  blockSignals(true);
  if( target_svalue == sdummy ){
    setChecked( target_ivalue == this_ivalue );
  }
  else{
    setChecked( target_svalue == this_svalue );
  }
  blockSignals(false);
}

MyQSpinBoxGroup::MyQSpinBoxGroup
( const char* text, 
  const char* name, int& value, const int min, const int max,
  QWidget* target, const char* slot, const int id )
  : MyQGroupBox(text)
{
  this->min    = min;
  this->max    = max;
  this->target = target;
  this->slot   = slot;
  this->id     = id;

  //  setFixedWidth(240);

  box = new QHBoxLayout;
  box->addWidget(new QLabel(name), Qt::AlignRight );

  spinbox = new MyQSpinBox( value, min, max, target, slot, id );
  vspinbox.clear();

  box->addWidget(spinbox);
  if( min==max ){
    char text[32];
    snprintf( text, sizeof(text), "/%d", max );
    label_range = new QLabel(text);
    box->addWidget(label_range, Qt::AlignRight );
  }
  else{
    label_range = NULL;
  }
  this->setLayout(box);
}

MyQSpinBoxGroup::MyQSpinBoxGroup
( const char* text, 
  const char* name, const int min, const int max,
  QWidget* target, const char* slot, const int id )
  : MyQGroupBox(text)
{
  this->min    = min;
  this->max    = max;
  this->target = target;
  this->slot   = slot;
  this->id     = id;

  //  setFixedWidth(240);

  box = new QHBoxLayout;
  box->addWidget(new QLabel(name), Qt::AlignRight );
  spinbox = NULL;
  vspinbox.clear();
  if( min==max ){
    char text[32];
    snprintf( text, sizeof(text), "/%d", max );
    label_range = new QLabel(text);
    box->addWidget(label_range, Qt::AlignRight );
  }
  else{
    label_range = NULL;
  }
  this->setLayout(box);
}

void MyQSpinBoxGroup::setCheckable( bool& checkable )
{
  if( checkable ){
    MyQGroupBox::setCheckable
      ( checkable, target, slot, id );

    MyQGroupBox::setChecked(checkable=false);
  }
  else{
    checkable = true;
  }
}

void MyQSpinBoxGroup::addSpinBox( int& value )
{
  vspinbox.push_back( new MyQSpinBox( value, min, max, target, slot, id ) );
  box->insertWidget(vspinbox.size(),vspinbox.back());
}

void MyQSpinBoxGroup::update( void )
{
  if( spinbox ) spinbox->update();
  if( !vspinbox.empty() ){
    for( int i=0; i<(int)vspinbox.size(); i++ ){
      vspinbox[i]->update();
    }
  }
}

void MyQSpinBoxGroup::setRange( const int min, const int max )
{
  if( spinbox ) spinbox->setRange(min,max);
  if( !vspinbox.empty() ){
    for( int i=0; i<(int)vspinbox.size(); i++ ){
      vspinbox[i]->setRange(min,max);
    }
  }

  {
    char text[32];
    snprintf( text, sizeof(text), "/%d", max );
    label_range->setText(text);
  }
}


//----
MyQDoubleSpinBoxGroup::MyQDoubleSpinBoxGroup
( const char* text, const char* name,
  double& value, const double min, const double max,
  QWidget* target, const char* slot, const int id ) : MyQGroupBox(text)
{
  this->min    = min;
  this->max    = max;
  this->target = target;
  this->slot   = slot;
  this->id     = id;

  //  setFixedWidth(240);

  box = new QHBoxLayout;
  box->addWidget(new QLabel(name), Qt::AlignRight );

  spinbox = new MyQDoubleSpinBox( value, min, max, target, slot, id );
  vspinbox.clear();

  box->addWidget(spinbox);
  label_range = NULL;
  this->setLayout(box);
}


MyQDoubleSpinBoxGroup::MyQDoubleSpinBoxGroup
( const char* text, const char* name,
  const double min, const double max,
  QWidget* target, const char* slot, const int id ) : MyQGroupBox(text)
{
  this->min    = min;
  this->max    = max;
  this->target = target;
  this->slot   = slot;
  this->id     = id;

  //  setFixedWidth(240);

  box = new QHBoxLayout;
  box->addWidget(new QLabel(name), Qt::AlignRight );
  spinbox = NULL;
  vspinbox.clear();
  label_range = NULL;
  this->setLayout(box);
}

void MyQDoubleSpinBoxGroup::setCheckable( bool& checkable )
{
  if( checkable ){
    MyQGroupBox::setCheckable
      ( checkable, target, slot, id );

    MyQGroupBox::setChecked(checkable=false);
  }
  else{
    checkable = true;
  }
}

void MyQDoubleSpinBoxGroup::addSpinBox( double& value )
{
  vspinbox.push_back( new MyQDoubleSpinBox( value, min, max, target, slot, id ) );
  box->insertWidget(vspinbox.size(),vspinbox.back());
}

void MyQDoubleSpinBoxGroup::update( void )
{
  if( spinbox ) spinbox->update();
  if( !vspinbox.empty() ){
    for( int i=0; i<(int)vspinbox.size(); i++ ){
      vspinbox[i]->update();
    }
  }
}

MyQSpinBox::MyQSpinBox
( int& _value, const int _min, const int _max,
  QWidget* target, const char* slot, const int _id ) :
  value(_value), min(_min), max(_max), id(_id)
{
  blockSignals(true);
  QSpinBox::setRange(min,max);
  QSpinBox::setSingleStep(1);
  blockSignals(false);

  connect(this, SIGNAL(valueChanged(int)), this, SLOT(convert(int)));
  connect(this, SIGNAL(valueChanged(const MyEvent&)), target, slot );

  setRange( min, max );
  update();
}

void MyQSpinBox::setRange( const int min, const int max )
{
  blockSignals(true);
  QSpinBox::setRange(min,max);
  blockSignals(false);

  // in case min==max, do not change value.
  // to make additional label next to this spinbox.
  if( min<max ){
    if( value < min ){
      value = min;
      update();
      emit valueChanged(MyEvent(id));
    }
    if( value > max ){
      value = max;
      update();
      emit valueChanged(MyEvent(id));
    }
  }
}


void MyQSpinBox::convert( int i )
{
  value = i;
  emit valueChanged(MyEvent(id));
}

void MyQSpinBox::update( void )
{
  blockSignals(true);
  setValue(value);
  blockSignals(false);
}

MyQDoubleSpinBox::MyQDoubleSpinBox
( double& _value, const double _min, const double _max,
  QWidget* target, const char* slot, const int _id ) :
  value(_value), min(_min), max(_max), id(_id)
{
  blockSignals(true);
  QDoubleSpinBox::setRange(min,max);
  QDoubleSpinBox::setSingleStep(1.0);
  blockSignals(false);

  connect(this, SIGNAL(valueChanged(double)), this, SLOT(convert(double)));
  connect(this, SIGNAL(valueChanged(const MyEvent&)), target, slot );

  update();
}

void MyQDoubleSpinBox::setRange( const int min, const int max )
{
  blockSignals(true);
  QDoubleSpinBox::setRange(min,max);
  blockSignals(false);
}

void MyQDoubleSpinBox::convert( double d )
{
  value = d;
  emit valueChanged(MyEvent(id));
}

void MyQDoubleSpinBox::update( void )
{
  blockSignals(true);
  setValue(value);
  blockSignals(false);
}


MyQComboBox::MyQComboBox
( int& _value, const vector<QString>& _vname,
  QWidget* target, const char* slot, const int _id ) :
  value(_value), vname(_vname), id(_id)
{
  connect(this, SIGNAL(activated(const QString&)), this, SLOT(convert(const QString&)));
  connect(this, SIGNAL(valueChanged(const MyEvent&)), target, slot );

  update();
}

void MyQComboBox::convert( const QString& str )
{
  for( int i=0; i<(int)vname.size(); i++ ){
    if( vname[i] == str && value != i ){
      value = i;
      emit valueChanged(MyEvent(id));
      break;
    }
  }
}

void MyQComboBox::update( void )
{
  QComboBox::clear();
  for( int i=0; i<(int)vname.size(); i++ ){
    QComboBox::addItem(vname[i]);
  }

  blockSignals(true);
  setCurrentIndex( value );
  blockSignals(false);
}


MyQLineEdit::MyQLineEdit
( int& _value, const int _min, const int _max,
  QWidget* target, const char* slot, const int _id ) :
  ivalue(_value), imin(_min), imax(_max),
  dvalue(ddummy), dmin(0.0), dmax(0.0),
  svalue(sdummy), id(_id)
{
  type = INTEGER;
  sprintf(format,"%%d");

  setValidator( new QIntValidator( imin, imax, this ) );
  setAlignment(Qt::AlignLeft);

  connect(this, SIGNAL(textChanged(const QString&)), this, SLOT(convert(const QString&)));
  connect(this, SIGNAL(editingFinished()), this, SLOT(convert()));
  connect(this, SIGNAL(valueChanged(const MyEvent&)), target, slot );

  update();
}

MyQLineEdit::MyQLineEdit
( double& _value, const double _min, const double _max,
  QWidget* target, const char* slot, const int _id ) :
  ivalue(idummy),  imin(0), imax(0),
  dvalue(_value), dmin(_min), dmax(_max),
  svalue(sdummy), id(_id)
{
  if( (dmin!=0.0 && fabs(dmin)<1e-6) || (dmax!=0.0 && fabs(dmax)<1e-6) ||
      fabs(dmin) >= 1e+5 || fabs(dmax) >= 1e5 ){
    type = DOUBLE_E;
    sprintf(format,"%%.3e");
  }
  else{
    type = DOUBLE_F;
    sprintf(format,"%%.6f");
  }

  setValidator( new QDoubleValidator
		( dmin>=0.0 ? 0.0 : -1.0e100,
		  dmax>=0.0 ? +1.0e100 : 0.0,
		  (dmin!=0.0 && fabs(dmin)<1e-6) ||
		  (dmax!=0.0 && fabs(dmax)<1e-6) ? 10 : 8, this) );
  setAlignment(Qt::AlignLeft);

  connect(this, SIGNAL(textChanged(const QString&)), this, SLOT(convert(const QString&)));
  connect(this, SIGNAL(editingFinished()), this, SLOT(convert()));
  connect(this, SIGNAL(valueChanged(const MyEvent&)), target, slot );

  update();
}

MyQLineEdit::MyQLineEdit
( QString& _value,
  QWidget* target, const char* slot, const int _id ) :
  ivalue(idummy), imin(0), imax(0),
  dvalue(ddummy), dmin(0), dmax(0),
  svalue(_value), id(_id)
{
  type = STRING;
  sprintf(format,"%%s");

  setAlignment(Qt::AlignLeft);

  connect(this, SIGNAL(textChanged(const QString&)), this, SLOT(convert(const QString&)));
  connect(this, SIGNAL(editingFinished()), this, SLOT(convert()));
  connect(this, SIGNAL(valueChanged(const MyEvent&)), target, slot );

  update();
}

void MyQLineEdit::setRange( const int    min, const int    max )
{
  this->imin = min;
  this->imax = max;
}
void MyQLineEdit::setRange( const double min, const double max )
{
  this->dmin = min;
  this->dmax = max;
}

void MyQLineEdit::convert( const QString& str )
{
  temp_str_changed = true;
  temp_str = str;
}

void MyQLineEdit::convert( void )
{
  int    itemp;
  double dtemp;

  if( type == INTEGER ){
    if( 1 != sscanf( qPrintable(temp_str), "%d", &itemp ) ||
	itemp<imin || imax<itemp ){
      update();
      return;
    }
    ivalue = itemp;
  }
      
  if( type == DOUBLE_E || type == DOUBLE_F ){
    if( 1 != sscanf( qPrintable(temp_str), "%lf", &dtemp ) ||
      dtemp<dmin || dmax<dtemp ){
      update();
      return;
    }
    dvalue = dtemp;
  }
      
  if( type == STRING ){
    svalue = temp_str;
  }

  if( temp_str_changed ){
    temp_str_changed = false;
    emit valueChanged(id);
  }
}

void MyQLineEdit::update( void )
{
  QString str;
  if( type == INTEGER ){
    str.sprintf(format, ivalue );
  }
  if( type == DOUBLE_F ){
    str.sprintf(format, dvalue );
  }
  if( type == DOUBLE_E ){
    str.sprintf(format, dvalue );
  }
  if( type == STRING ){
    str.sprintf(format,qPrintable(svalue) );
  }
  temp_str = str;
  temp_str_changed = false;

  blockSignals(true);
  setText(str);
  blockSignals(false);
}

void MyQLineEdit::setFormat( const char* format )
{
  sprintf(this->format,format);
  update();
}

MyQEditSliderGroup::MyQEditSliderGroup
( const char* text, const char* name, const char* unit,
  const char* low, const char* high,
  double& value, const double min, const double max,
  QWidget* target, const char* slot, const int id ) : MyQGroupBox(text)
{
  this->target = target;
  this->slot   = slot;
  this->id     = id;

  lineedit = new MyQLineEdit( value, min, max, target, slot, id );
  slider   = new MyQSlider  ( value, min, max, target, slot, id );

  connect(lineedit, SIGNAL(valueChanged(const MyEvent&)), slider,   SLOT(update()) );
  connect(slider,   SIGNAL(valueChanged(const MyEvent&)), lineedit, SLOT(update()) );

  QGridLayout* box = new QGridLayout;
  box->addWidget(new QLabel(name), 0, 0, Qt::AlignRight );
  box->addWidget(lineedit, 0, 1 );
  box->addWidget(new QLabel(unit), 0, 2, Qt::AlignLeft  );
  box->addWidget(new QLabel(low),  1, 0, Qt::AlignRight );
  box->addWidget(slider, 1, 1 );
  box->addWidget(new QLabel(high), 1, 2, Qt::AlignLeft  );

  this->setLayout(box);
}

void MyQEditSliderGroup::setFormat( const char* format )
{
  lineedit->setFormat(format);
  lineedit->update();
}


void MyQEditSliderGroup::setCheckable( bool& checkable )
{
  if( checkable ){
    MyQGroupBox::setCheckable
      ( checkable, target, slot, id );
    MyQGroupBox::setChecked(checkable=false);
  }
  else{
    checkable = true;
  }
}

void MyQEditSliderGroup::setRange( const double min, const double max )
{
  blockSignals(true);
  lineedit->setRange(min,max);
  slider->setRange(min,max);
  blockSignals(false);
}

void MyQEditSliderGroup::update( void )
{
  lineedit->update();
  slider->update();
}

MyQGroupBox::MyQGroupBox
( const char* text ) :
  QGroupBox(text), value_ptr(NULL), id(0)
{
  expert = false;
  QGroupBox::setCheckable(false);
}

MyQGroupBox::MyQGroupBox
( const char* text, bool& value ) :
  QGroupBox(text)
{
  this->value_ptr = &value;

  connect(this, SIGNAL(clicked(bool)), this, SLOT(convert(bool)));

  expert = false;
  QGroupBox::setCheckable(true);
  setChecked(*value_ptr);
}

MyQGroupBox::MyQGroupBox
( const char* text, bool& value,
  QWidget* target, const char* slot, const int id ) :
  QGroupBox(text)
{
  this->value_ptr = &value;
  this->id = id;

  connect(this, SIGNAL(clicked(bool)), this, SLOT(convert(bool)));
  connect(this, SIGNAL(valueChanged(const MyEvent&)), target, slot );

  expert = false;
  QGroupBox::setCheckable(true);
  setChecked(*value_ptr);
}

void MyQGroupBox::setCheckable( bool& value )
{
  if( value ){
    this->value_ptr = &value;
    QGroupBox::setCheckable(value);

    connect(this, SIGNAL(clicked(bool)), this, SLOT(convert(bool)));
    QGroupBox::setCheckable(true);
    setChecked(*value_ptr=false);
  }
  else{
    value = true;
  }
}

void MyQGroupBox::setCheckable
( bool& value, QWidget* target, const char* slot, const int id )
{
  if( value ){
    this->value_ptr = &value;
    this->id = id;
    QGroupBox::setCheckable(value);

    connect(this, SIGNAL(clicked(bool)), this, SLOT(convert(bool)));
    connect(this, SIGNAL(valueChanged(const MyEvent&)), target, slot );
    QGroupBox::setCheckable(true);
    setChecked(*value_ptr=false);
  }
  else{
    value = true;
  }
}

void MyQGroupBox::convert( bool checked )
{
  *value_ptr = checked;
  if(id) emit valueChanged(MyEvent(id));
}

void MyQGroupBox::setExpert( const bool expert )
{
  this->expert = expert;
}


void MyQGroupBox::update( void )
{
  blockSignals(true);
  if( isCheckable() ) setChecked( *value_ptr );
  blockSignals(false);
}

void MyQGroupBox::update( const bool expert )
{
  blockSignals(true);
  if( isCheckable() ) setChecked( *value_ptr );
  blockSignals(false);

  if( this->expert && !expert ){
    hide();
  }
  else{
    show();
  }
}

void MyQTab::updateExpert( const bool expert )
{
  for( int i=0; i<(int)vgroup.size(); i++ ){
    vgroup[i]->update(expert);
  }
  for( int i=0; i<(int)vwidget.size(); i++ ){
    vwidget[i]->update();
  }
}
    
void MyQTab::update( void )
{
  for( int i=0; i<(int)vwidget.size(); i++ ){
    vwidget[i]->update();
  }
}
