/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

/*!
 \file qtxml.cc
 \brief XMLf[^ǍNX
*/

#include <stdio.h>
#include "qtxml.h"
#include "qtmisc.h"

bool XML::load( QDomDocument& doc, const char* filename )
{
  QFile file( filename );
  if( !file.open(QIODevice::ReadOnly) ){
    fprintf(stderr,"not found XML file %s\n", filename );
    return false;
  }
  if( !doc.setContent(&file) ){
    fprintf(stderr,"something wrong XML file %s\n", filename );
    file.close();
    return false;
  }
  file.close();

  return true;
}

bool XML::save( const QDomDocument& doc, const char* filename )
{
  QFile file( filename );
  if( !file.open(QIODevice::WriteOnly) ){
    return false;
  }
  file.write( doc.toByteArray() );
  file.close();

  return true;
}


QDomNode XML::getRoot( const QDomDocument& doc )
{
  return doc.documentElement();
}


QDomElement XML::getFirstElementByTagName( const QDomNode& node, const char* tag )
{
  if( !node.isElement() ) return QDomElement();

  return node.firstChildElement(tag);
}

vector<QDomElement> XML::getElementsByTagName( const QDomNode& node, const char* tag )
{
  vector<QDomElement> velem;

  if( !node.isElement() ) return velem;

  for( QDomElement elem = node.firstChildElement(tag); !elem.isNull();
       elem = elem.nextSiblingElement(tag) ){
    velem.push_back(elem);
  }

  return velem;
}

QString XML::getChildValue( const QDomNode& node )
{
  if( !node.hasChildNodes() || !node.firstChild().isText() )
    return QString("");

  return node.firstChild().nodeValue();
}

QString XML::getAttribute( const QDomElement& elem, const char* attr )
{
  return elem.attribute(attr);
}

extern "C" int vsscanf( const char*, const char*, va_list );

int XML::sscanf( const QString& str, const char* const format, ... )
{
  va_list argptr;
  va_start( argptr, format );

  return vsscanf( qPrintable(str), format, argptr );
}

int XML::sscanf( const QString& str, const char* const format, QString* str2 )
{
  static char value[128];
  int status = sscanf( qPrintable(str), "%s", value );
  *str2 = QString(value);
  return status;
}



