/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

/*!
 \file symbol.h
 \brief fLNX
*/

#ifndef __SYMBOL_H_INCLUDED
#define __SYMBOL_H_INCLUDED

#include <QtCore/QString>

// ffz̃CfbNXɕϊ\
struct ElementSymbol
{
  enum AtomicNumber {
    X=0, // unspecified element
     H, He,
    Li, Be,  B,  C,  N,  O,  F, Ne,
    Na, Mg, Al, Si,  P,  S, Cl, Ar,
     K, Ca, Sc, Ti,  V, Cr, Mn, Fe, Co, Ni, Cu, Zn, Ga, Ge, As, Se, Br, Kr,
    Rb, Sr,  Y, Zr, Nb, Mo, Tc, Ru, Rh, Pd, Ag, Cd, In, Sn, Sb, Te,  I, Xe,
    Cs, Ba,
    La, Ce, Pr, Nd, Pm, Sm, Eu, Gd, Tb, Dy, Ho, Er, Tm, Yb, Lu,
    Hf, Ta,  W, Re, Os, Ir, Pt, Au, Hg, Tl, Pb, Bi, Po, At, Rn,
    Fr, Ra,
    Ac, Th, Pa, U, Np, Pu, Am, Cm, Bk, Cf, Es, Fm, Md, No, Lr
  };

  inline static unsigned int  getAtomicNumber( const QString& name );
  inline static const QString  getAtomicName( const int number );
  inline static double        getAtomicMass( const int number );
  inline static double        getAtomicValence( const int number );

  inline static const QString  getAtomicName( const QString& name ){
    return getAtomicName(getAtomicNumber(name));
  }
  inline static double        getAtomicMass( const QString& name ){
    return getAtomicMass(getAtomicNumber(name));
  }
  inline static double        getAtomicValence( const QString& name ){
    return getAtomicValence(getAtomicNumber(name));
  }
};

inline double       ElementSymbol::getAtomicValence( const int number )
{
  switch( number ){
  case H  : return 1;
  case He : return 2;
  case Li : return 1;
  case Be : return 2;
  case B  : return 2+1;
  case C  : return 2+2;
  case N  : return 2+3;
  case O  : return 2+4;
  case F  : return 2+5;
  case Ne : return 2+6;
  case Na : return 1;
  case Mg : return 2;
  case Al : return 2+1;
  case Si : return 2+2;
  case P  : return 2+3;
  case S  : return 2+4;
  case Cl : return 2+5;
  case Ar : return 2+6;
  case K  : return 1;
  case Ca : return 2;
  case Sc : return 1+2;
  case Ti : return 2+2;
  case V  : return 3+2;
  case Cr : return 5+1;
  case Mn : return 2+5;
  case Fe : return 6+2;
  case Co : return 2+7;
  case Ni : return 8+2;
  case Cu : return 10+1;
  case Zn : return 10+2;
  case Ga : return 2+1;
  case Ge : return 2+2;
  case As : return 2+3;
  case Se : return 2+4;
  case Br : return 2+5;
  case Kr : return 2+6;
  case Rb : return 1;
  case Sr : return 2;
  case Y  : return 1+2;
  case Zr : return 2+2;
  case Nb : return 4+1;
  case Mo : return 1+5;
  case Tc : return 5+2;
  case Ru : return 7+1;
  case Rh : return 1+8;
  case Pd : return 10;
  case Ag : return 10+1;
  case Cd : return 2+10;
  case In : return 2+1;
  case Sn : return 2+2;
  case Sb : return 2+3;
  case Te : return 2+4;
  case I  : return 2+5;
  case Xe : return 2+6;
  case Cs : return 1;
  case Ba : return 2;
  case La : return 9; // 5p6s5d4f
  case Ce : return 10;
  case Pr : return 9; // undef
  case Nd : return 9; // 5p5d6s
  case Pm : return 9; // undef
  case Sm : return 9; // undef
  case Eu : return 9; // undef
  case Gd : return 9; // 5p5d6s
  case Tb : return 9; // undef
  case Dy : return 9; // 5p5d6s
  case Ho : return 9; // undef
  case Er : return 9; // undef
  case Tm : return 9; // undef
  case Yb : return 9; // undef
  case Lu : return 2+14+1;
  case Hf : return 2+2;
  case Ta : return 3+2;
  case W  : return 4+2;
  case Re : return 5+2;
  case Os : return 6+2;
  case Ir : return 7+2;
  case Pt : return 9+1;
  case Au : return 10+1;
  case Hg : return 10+2;
  case Tl : return 2+1;
  case Pb : return 2+2;
  case Bi : return 2+3;
  case Po : return 2+4; // undef
  case At : return 2+5; // undef
  case Rn : return 2+6; // undef
  case Fr : return 1; // undef
  case Ra : return 2; // undef
  case Ac : return 1+2; // undef
  case Th : return 2+2; // undef
  case Pa : return 2+1+2; // undef
  case U  : return 3+1+2; // undef
  case Np : return 2+1+4; // undef
  case Pu : return 6+2; // undef
  case Am : return 7+2; // undef
  case Cm : return 7+1+2; // undef
  case Bk : return 9+2; // undef
  case Cf : return 10+2; // undef
  case Es : return 11+2; // undef
  case Fm : return 12+2; // undef
  case Md : return 13+2; // undef
  case No : return 14+2; // undef
  case Lr : return 14+1+2; // undef
  default : return 0.0;
  }
  return 0.0;
}

inline double       ElementSymbol::getAtomicMass( const int number )
{
  switch( number ){
  case H  : return  1.007947;
  case He : return  4.0026022;
  case Li : return  6.9412;
  case Be : return  9.0121823;
  case B  : return 10.8117;
  case C  : return 12.01078;
  case N  : return 14.00672;
  case O  : return 15.99943;
  case F  : return 18.99840325;
  case Ne : return 20.17976;
  case Na : return 22.989769282;
  case Mg : return 24.30506;
  case Al : return 26.98153868;
  case Si : return 28.08553;
  case P  : return 30.9737622;
  case S  : return 32.0655;
  case Cl : return 35.4532;
  case Ar : return 39.9481;
  case K  : return 39.09831;
  case Ca : return 40.0784;
  case Sc : return 44.9559126;
  case Ti : return 47.8671;
  case V  : return 50.94151;
  case Cr : return 51.99616;
  case Mn : return 54.9380455;
  case Fe : return 55.8452;
  case Co : return 58.9331955;
  case Ni : return 58.69344;
  case Cu : return 63.5463;
  case Zn : return 65.382;
  case Ga : return 69.7231;
  case Ge : return 72.641;
  case As : return 74.921602;
  case Se : return 78.963;
  case Br : return 79.9041;
  case Kr : return 83.7982;
  case Rb : return 85.46783;
  case Sr : return 87.621;
  case Y  : return 88.905852;
  case Zr : return 91.2242;
  case Nb : return 92.906382;
  case Mo : return 95.962;
  case Tc : return 98;
  case Ru : return 101.072;
  case Rh : return 102.905502;
  case Pd : return 106.421;
  case Ag : return 107.86822;
  case Cd : return 112.4118;
  case In : return 114.8183;
  case Sn : return 118.7107;
  case Sb : return 121.7601;
  case Te : return 127.603;
  case I  : return 126.904473;
  case Xe : return 131.2936;
  case Cs : return 132.90545192;
  case Ba : return 137.3277;
  case La : return 138.905477;
  case Ce : return 140.1161;
  case Pr : return 140.907652;
  case Nd : return 144.2423;
  case Pm : return 145;
  case Sm : return 150.362;
  case Eu : return 151.9641;
  case Gd : return 157.253;
  case Tb : return 158.925352;
  case Dy : return 162.5001;
  case Ho : return 164.930322;
  case Er : return 167.2593;
  case Tm : return 168.934212;
  case Yb : return 173.0545;
  case Lu : return 174.96681;
  case Hf : return 178.492;
  case Ta : return 180.947882;
  case W  : return 183.841;
  case Re : return 186.2071;
  case Os : return 190.233;
  case Ir : return 192.2173;
  case Pt : return 195.0849;
  case Au : return 196.9665694;
  case Hg : return 200.592;
  case Tl : return 204.38332;
  case Pb : return 207.21;
  case Bi : return 208.980401;
  case Po : return 209;
  case At : return 210;
  case Rn : return 222;
  case Fr : return 223;
  case Ra : return 226;
  case Ac : return 227;
  case Th : return 232.038062;
  case Pa : return 231.035882;
  case U  : return 238.028913;
  case Np : return 237;
  case Pu : return 244;
  case Am : return 243;
  case Cm : return 247;
  case Bk : return 247;
  case Cf : return 251;
  case Es : return 252;
  case Fm : return 257;
  case Md : return 258;
  case No : return 259;
  case Lr : return 262;
  default : return 0.0;
  }
  return 0.0;
}

inline const QString ElementSymbol::getAtomicName( const int number )
{
  switch( number ){
  case H  : return QString("H");
  case He : return QString("He");
  case Li : return QString("Li");
  case Be : return QString("Be");
  case B  : return QString("B");
  case C  : return QString("C");
  case N  : return QString("N");
  case O  : return QString("O");
  case F  : return QString("F");
  case Ne : return QString("Ne");
  case Na : return QString("Na");
  case Mg : return QString("Mg");
  case Al : return QString("Al");
  case Si : return QString("Si");
  case P  : return QString("P");
  case S  : return QString("S");
  case Cl : return QString("Cl");
  case Ar : return QString("Ar");
  case K  : return QString("K");
  case Ca : return QString("Ca");
  case Sc : return QString("Sc");
  case Ti : return QString("Ti");
  case V  : return QString("V");
  case Cr : return QString("Cr");
  case Mn : return QString("Mn");
  case Fe : return QString("Fe");
  case Co : return QString("Co");
  case Ni : return QString("Ni");
  case Cu : return QString("Cu");
  case Zn : return QString("Zn");
  case Ga : return QString("Ga");
  case Ge : return QString("Ge");
  case As : return QString("As");
  case Se : return QString("Se");
  case Br : return QString("Br");
  case Kr : return QString("Kr");
  case Rb : return QString("Rb");
  case Sr : return QString("Sr");
  case Y  : return QString("Y");
  case Zr : return QString("Zr");
  case Nb : return QString("Nb");
  case Mo : return QString("Mo");
  case Tc : return QString("Tc");
  case Ru : return QString("Ru");
  case Rh : return QString("Rh");
  case Pd : return QString("Pd");
  case Ag : return QString("Ag");
  case Cd : return QString("Cd");
  case In : return QString("In");
  case Sn : return QString("Sn");
  case Sb : return QString("Sb");
  case Te : return QString("Te");
  case I  : return QString("I");
  case Xe : return QString("Xe");
  case Cs : return QString("Cs");
  case Ba : return QString("Ba");
  case La : return QString("La");
  case Ce : return QString("Ce");
  case Pr : return QString("Pr");
  case Nd : return QString("Nd");
  case Pm : return QString("Pm");
  case Sm : return QString("Sm");
  case Eu : return QString("Eu");
  case Gd : return QString("Gd");
  case Tb : return QString("Tb");
  case Dy : return QString("Dy");
  case Ho : return QString("Ho");
  case Er : return QString("Er");
  case Tm : return QString("Tm");
  case Yb : return QString("Yb");
  case Lu : return QString("Lu");
  case Hf : return QString("Hf");
  case Ta : return QString("Ta");
  case W  : return QString("W");
  case Re : return QString("Re");
  case Os : return QString("Os");
  case Ir : return QString("Ir");
  case Pt : return QString("Pt");
  case Au : return QString("Au");
  case Hg : return QString("Hg");
  case Tl : return QString("Tl");
  case Pb : return QString("Pb");
  case Bi : return QString("Bi");
  case Po : return QString("Po");
  case At : return QString("At");
  case Rn : return QString("Rn");
  case Fr : return QString("Fr");
  case Ra : return QString("Ra");
  case Ac : return QString("Ac");
  case Th : return QString("Th");
  case Pa : return QString("Pa");
  case U  : return QString("U");
  case Np : return QString("Np");
  case Pu : return QString("Pu");
  case Am : return QString("Am");
  case Cm : return QString("Cm");
  case Bk : return QString("Bk");
  case Cf : return QString("Cf");
  case Es : return QString("Es");
  case Fm : return QString("Fm");
  case Md : return QString("Md");
  case No : return QString("No");
  case Lr : return QString("Lr");
  default : return QString("X");
  }

  return QString("X");
}

inline unsigned int ElementSymbol::getAtomicNumber( const QString& name )
{
  if( name.size() != 1 && name.size() != 2 ){
    return X;
  }

  const char* const tmp = qPrintable(name);
  const char c0 = tmp[0];
  const char c1 = tmp[1];
  
  switch( c0 ){
  case 'A' : {
    if( false );
    else if( c1 == "Ac"[1] ) return Ac;
    else if( c1 == "Ag"[1] ) return Ag;
    else if( c1 == "Al"[1] ) return Al;
    else if( c1 == "Am"[1] ) return Am;
    else if( c1 == "Ar"[1] ) return Ar;
    else if( c1 == "As"[1] ) return As;
    else if( c1 == "At"[1] ) return At;
    else if( c1 == "Au"[1] ) return Au;
    else                          return  X; // unknown element
  } break;
  case 'B' : {
    if( false );
    else if( c1 == "B"[1]  ) return  B;
    else if( c1 == "Ba"[1] ) return Ba;
    else if( c1 == "Be"[1] ) return Be;
    else if( c1 == "Bi"[1] ) return Bi;
    else if( c1 == "Bk"[1] ) return Bk;
    else if( c1 == "Br"[1] ) return Br;
    else                          return  X; // unknown element
  } break;
  case 'C' : {
    if( false );
    else if( c1 == "C"[1]  ) return  C;
    else if( c1 == "Ca"[1] ) return Ca;
    else if( c1 == "Cd"[1] ) return Cd;
    else if( c1 == "Ce"[1] ) return Ce;
    else if( c1 == "Cf"[1] ) return Cf;
    else if( c1 == "Cl"[1] ) return Cl;
    else if( c1 == "Cm"[1] ) return Cm;
    else if( c1 == "Co"[1] ) return Co;
    else if( c1 == "Cr"[1] ) return Cr;
    else if( c1 == "Cs"[1] ) return Cs;
    else if( c1 == "Cu"[1] ) return Cu;
    else                          return  X; // unknown element
  } break;
  case 'D' : {
    if( false );
    else if( c1 == "Dy"[1] ) return Dy;
    else                          return  X; // unknown element
  } break;
  case 'E' : {
    if( false );
    else if( c1 == "Er"[1] ) return Er;
    else if( c1 == "Es"[1] ) return Es;
    else if( c1 == "Eu"[1] ) return Eu;
    else                          return  X; // unknown element
  } break;
  case 'F' : {
    if( false );
    else if( c1 == "F"[1]  ) return  F;
    else if( c1 == "Fe"[1] ) return Fe;
    else if( c1 == "Fm"[1] ) return Fm;
    else if( c1 == "Fr"[1] ) return Fr;
    else                          return  X; // unknown element
  } break;
  case 'G' : {
    if( false );
    else if( c1 == "Ga"[1] ) return Ga;
    else if( c1 == "Gd"[1] ) return Gd;
    else if( c1 == "Ge"[1] ) return Ge;
    else                          return  X; // unknown element
  } break;
  case 'H' : {
    if( false );
    else if( c1 == "H"[1]  ) return  H;
    else if( c1 == "He"[1] ) return He;
    else if( c1 == "Hf"[1] ) return Hf;
    else if( c1 == "Hg"[1] ) return Hg;
    else if( c1 == "Ho"[1] ) return Ho;
    else                          return  X; // unknown element
  } break;
  case 'I' : {
    if( false );
    else if( c1 == "I"[1]  ) return  I;
    else if( c1 == "In"[1] ) return In;
    else if( c1 == "Ir"[1] ) return Ir;
    else                          return  X; // unknown element
  } break;
  case 'J' : {
    if( false );
    else                          return  X; // unknown element
  } break;
  case 'K' : {
    if( false );
    else if( c1 == "K"[1]  ) return  K;
    else if( c1 == "Kr"[1] ) return Kr;
    else                          return  X; // unknown element
  } break;
  case 'L' : {
    if( false );
    else if( c1 == "La"[1] ) return La;
    else if( c1 == "Li"[1] ) return Li;
    else if( c1 == "Lr"[1] ) return Lr;
    else if( c1 == "Lu"[1] ) return Lu;
    else                          return  X; // unknown element
  } break;
  case 'M' : {
    if( false );
    else if( c1 == "Md"[1] ) return Md;
    else if( c1 == "Mg"[1] ) return Mg;
    else if( c1 == "Mn"[1] ) return Mn;
    else if( c1 == "Mo"[1] ) return Mo;
    else                          return  X; // unknown element
  } break;
  case 'N' : {
    if( false );
    else if( c1 == "N"[1]  ) return  N;
    else if( c1 == "Na"[1] ) return Na;
    else if( c1 == "Nb"[1] ) return Nb;
    else if( c1 == "Nd"[1] ) return Nd;
    else if( c1 == "Ne"[1] ) return Ne;
    else if( c1 == "Ni"[1] ) return Ni;
    else if( c1 == "No"[1] ) return No;
    else if( c1 == "Np"[1] ) return Np;
    else                          return  X; // unknown element
  } break;
  case 'O' : {
    if( false );
    else if( c1 == "O"[1]  ) return  O;
    else if( c1 == "Os"[1] ) return Os;
    else                          return  X; // unknown element
  } break;
  case 'P' : {
    if( false );
    else if( c1 == "P"[1]  ) return  P;
    else if( c1 == "Pa"[1] ) return Pa;
    else if( c1 == "Pb"[1] ) return Pb;
    else if( c1 == "Pd"[1] ) return Pd;
    else if( c1 == "Pm"[1] ) return Pm;
    else if( c1 == "Po"[1] ) return Po;
    else if( c1 == "Pr"[1] ) return Pr;
    else if( c1 == "Pt"[1] ) return Pt;
    else if( c1 == "Pu"[1] ) return Pu;
    else                          return  X; // unknown element
  } break;
  case 'Q' : {
    if( false );
    else                          return  X; // unknown element
  } break;
  case 'R' : {
    if( false );
    else if( c1 == "Ra"[1] ) return Ra;
    else if( c1 == "Rb"[1] ) return Rb;
    else if( c1 == "Re"[1] ) return Re;
    else if( c1 == "Rh"[1] ) return Rh;
    else if( c1 == "Rn"[1] ) return Rn;
    else if( c1 == "Ru"[1] ) return Ru;
    else                          return  X; // unknown element
  } break;
  case 'S' : {
    if( false );
    else if( c1 == "S"[1]  ) return  S;
    else if( c1 == "Sb"[1] ) return Sb;
    else if( c1 == "Sc"[1] ) return Sc;
    else if( c1 == "Se"[1] ) return Se;
    else if( c1 == "Si"[1] ) return Si;
    else if( c1 == "Sm"[1] ) return Sm;
    else if( c1 == "Sn"[1] ) return Sn;
    else if( c1 == "Sr"[1] ) return Sr;
    else                          return  X; // unknown element
  } break;
  case 'T' : {
    if( false );
    else if( c1 == "Ta"[1] ) return Ta;
    else if( c1 == "Tb"[1] ) return Tb;
    else if( c1 == "Tc"[1] ) return Tc;
    else if( c1 == "Te"[1] ) return Te;
    else if( c1 == "Th"[1] ) return Th;
    else if( c1 == "Ti"[1] ) return Ti;
    else if( c1 == "Tl"[1] ) return Tl;
    else if( c1 == "Tm"[1] ) return Tm;
    else                          return  X; // unknown element
  } break;
  case 'U' : {
    if( false );
    else if( c1 == "U"[1]  ) return  U;
    else                          return  X; // unknown element
  } break;
  case 'V' : {
    if( false );
    else if( c1 == "V"[1]  ) return  V;
    else                          return  X; // unknown element
  } break;
  case 'W' : {
    if( false );
    else if( c1 == "W"[1]  ) return  W;
    else                          return  X; // unknown element
  } break;
  case 'X' : {
    if( false );
    else if( c1 == "X"[1]  ) return  X; // unknown element
    else if( c1 == "Xe"[1] ) return Xe;
    else                          return  X; // unknown element
  } break;
  case 'Y' : {
    if( false );
    else if( c1 == "Y"[1]  ) return  Y;
    else if( c1 == "Yb"[1] ) return Yb;
    else                          return  X; // unknown element
  } break;
  case 'Z' : {
    if( false );
    else if( c1 == "Zn"[1] ) return Zn;
    else if( c1 == "Zr"[1] ) return Zr;
    else                          return X; // unknown element
  } break;
  default : {
    return X; // unknown element
  } break;
  }

  return X;
}


#endif // __SYMBOL_H_INCLUDED
