/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

#include <stdio.h>
#include <vector>
using namespace std;

#include "imatrix.h"

static inline int abs( const int n )
{
  return n<0 ? -n : n;
}

class DTElement
{
public:
  int    number; // $B86;RHV9f(B
};

class DTAtom
{
public:
  DTElement   element; // $B85AG(B
  Coordinates coords;  // $B%;%k:BI8(B(A,B,C)

  DTAtom( void ){}
  DTAtom( const int kd,
	const double a, const double b, const double c ){
    this->element.number = kd;
    this->coords = Coordinates(a,b,c);
  }
};




bool match( const double matrixA[3][3], const double matrixB[3][3] )
{
  const double tolerance = 1.0e-8;
  double residue = 0.0;
  for( int i=0; i<3; i++ ){
    for( int j=0; j<3; j++ ){
      const double diff = matrixA[i][j] - matrixB[i][j];
      residue += diff*diff;
    }
  }
  return residue<tolerance*tolerance;
}

bool match( const Coordinates& coordsA, const Coordinates& coordsB )
{
  const double tolerance = 1.0e-8;
  const double residue = Coordinates::normalize5(coordsA-coordsB).norm();
  return residue<tolerance*tolerance;
}

Position latvec[3];
vector<DTAtom> vatom;
vector<IMatrix> vmatrix;

void findSymmetry( void )
{
  double metric_original[3][3];
  double metric_rotated[3][3];

  for( int i=0; i<3; i++ ){
    for( int j=0; j<3; j++ ){
      metric_original[i][j] = latvec[i] * latvec[j];
    }
  }

  IMatrix matrix;

  matrix.initCanditate();
  do{
    if( abs(matrix.determinant()) != 1 ){
      continue;
    }

    matrix.transformUnitary( metric_rotated, metric_original );

    if( !match( metric_rotated, metric_original ) ){
      continue;
    }

    for( int step=0;step<3;step++ ){
      for( int i=0; i<3; i++ ){
	for( int j=0; j<3; j++ ){
	  metric_original[i][j]
	    = 0.5*(metric_original[i][j] + metric_rotated[i][j]);
	}
      }
      matrix.transformUnitary( metric_rotated, metric_original );
    }

    const int iat = 0;
    const Coordinates coords0_transformed = matrix * vatom[iat].coords;

    for( int jat=0; jat<(int)vatom.size(); jat++ ){
      if( vatom[jat].element.number != vatom[0].element.number ) continue;

      const Coordinates difference =
	Coordinates::normalize5(vatom[jat].coords - coords0_transformed);

      matrix.T[0] = fraction(difference.a);
      matrix.T[1] = fraction(difference.b);
      matrix.T[2] = fraction(difference.c);

      bool is_acceptable = true;

      for( int kat=0; kat<(int)vatom.size(); kat++ ){
	const Coordinates coords_transformed = matrix * vatom[kat].coords;
	const int element_number_transformed = vatom[kat].element.number;

	bool is_match = false;
	for( int lat=0; lat<(int)vatom.size(); lat++ ){
	  if( vatom[lat].element.number != element_number_transformed ) continue;
	  if( match( vatom[lat].coords, coords_transformed ) ){
	    is_match = true;
	    break;
	  }
	}
	if( !is_match ){
	  is_acceptable = false;
	  break;
	}
      }

      if( is_acceptable ){
	vmatrix.push_back(matrix);
      }
    }
  }while( matrix.nextCanditate() );

  /*
  for( int m=0; m<(int)vmatrix.size(); m++ ){
    const IMatrix& matrix = vmatrix[m];
    matrix.transformUnitary( metric_rotated, metric_original );

    double err=0.0;
    for( int i=0; i<3; i++ ){
      for( int j=0; j<3; j++ ){
	err += fabs(metric_original[i][j]-metric_rotated[i][j]);
      }
    }
    printf("Metric %2d   %e\n", m, err );
  }

  for( int m=0; m<(int)vmatrix.size(); m++ ){
    const IMatrix& matrix = vmatrix[m];
    matrix.transformUnitary( metric_rotated, metric_original );

    for( int step=0; step<3; step++ ){
      for( int kat=0; kat<(int)vatom.size(); kat++ ){
	const Coordinates coords_transformed = matrix * vatom[kat].coords;
	const int element_number_transformed = vatom[kat].element.number;

	for( int lat=0; lat<(int)vatom.size(); lat++ ){
	  if( vatom[lat].element.number != element_number_transformed ) continue;
	  if( match( vatom[lat].coords, coords_transformed ) ){
	    const Position diff = Coordinates::normalize5
	      (vatom[lat].coords - coords_transformed);
	    vatom[lat].coords -= diff;
	    break;
	  }
	}
      }
    }
  }

  for( int m=0; m<(int)vmatrix.size(); m++ ){
    const IMatrix& matrix = vmatrix[m];
    matrix.transformUnitary( metric_rotated, metric_original );

    for( int kat=0; kat<(int)vatom.size(); kat++ ){
      const Coordinates coords_transformed = matrix * vatom[kat].coords;
      const int element_number_transformed = vatom[kat].element.number;

      for( int lat=0; lat<(int)vatom.size(); lat++ ){
	if( vatom[lat].element.number != element_number_transformed ) continue;
	if( match( vatom[lat].coords, coords_transformed ) ){
	  const Position diff = Coordinates::normalize5
	    (vatom[lat].coords - coords_transformed);

	  double err=0.0;
	  for( int i=0; i<3; i++ ){
	    err += fabs(diff[i]);
	  }
	  if( err > 0.0 ){
	    printf("Atom   %2d %d %e\n", m, kat, err );
	  }
	  break;
	}
      }
    }
  }
  */
}




int main(int argc, char *argv[])
{
  latvec[0] = Position( 3.2488000393, 0.0000000000, 0.0000000000 );
  latvec[1] = Position(-1.6244000196, 2.8135433658, 0.0000000000 );
  latvec[2] = Position( 0.0000000000, 0.0000000000, 5.2053999901 );

  vatom.push_back( DTAtom( 1, 0.666666666667, 0.333333333333, 0.000000000 ) );
  vatom.push_back( DTAtom( 1, 0.333333333333, 0.666666666667, 0.500000000 ) );
  vatom.push_back( DTAtom( 2, 0.666666666667, 0.333333333333, 0.875000000 ) );
  vatom.push_back( DTAtom( 2, 0.333333333333, 0.666666666667, 0.375000000 ) );

  findSymmetry();

  for( int m=0; m<(int)vmatrix.size(); m++ ){
    for( int i=0; i<3; i++ ){
      for( int j=0; j<3; j++ ){
	printf(" %+2d", vmatrix[m].M[i][j] );
      }
      printf(" ");
    }
    for( int i=0; i<3; i++ ){
      printf(" %+.1f", vmatrix[m].T[i].toDouble() );
    }
    printf("\n");
  }


  for( int i=0; i<3; i++ ){
    printf("LatVec %d %+.15f %+.15f %+.15f\n", i,
	   latvec[i].x,
	   latvec[i].y,
	   latvec[i].z );
  }
  for( int ia=0; ia<(int)vatom.size(); ia++ ){
    printf("Atom   %d %+.15f %+.15f %+.15f\n", ia,
	   vatom[ia].coords.a,
	   vatom[ia].coords.b,
	   vatom[ia].coords.c
	   );
  }

  return 0;
}
