/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

/*!
 \file vector3d.h
 \brief OtB[hNX
*/

#ifndef __VECTOR3D_H_INCLUDED
#define __VECTOR3D_H_INCLUDED

#include <vector>
using namespace std;

// 3iq_f[^NX
// 1 STL vector ^Ŏ
template <typename T> class vector3d
{
protected:
  vector<T> data;
  T data_max, data_min;
  int nx, ny, nz;

public:
  bool periodic;
  T kx, ky, kz;

public:
  // RXgN^
  vector3d( void ){
    nx = ny = nz = 0;
    periodic = false;
  }
  // RXgN^ TCYw
  vector3d( const int _nx, const int _ny, const int _nz ) {
    resize( _nx, _ny, _nz );
    periodic = false;
  }
  // TCYύX
  void resize( const int _nx, const int _ny, const int _nz ) {
    nx = _nx, ny = _ny, nz = _nz;
    data.resize(nx*ny*nz);
  }
  // [
  void zero( void ){
    for( int i=0; i<size(); i++ ){
      data[i] = T(0);
    }
  }
  // 
  void clear( void ){
    data.clear(); // just set the size of array to 0.
    vector<T>().swap( data ); // really free the memory.
    nx = ny = nz = 0;
    periodic = false;
  }
  // TCY
  int size( void ) const {
    return (int)data.size();
  }
  int sizeX( void ) const {
    return periodic ? nx-1 : nx;
  }
  int sizeY( void ) const {
    return periodic ? ny-1 : ny;
  }
  int sizeZ( void ) const {
    return periodic ? nz-1 : nz;
  }
  // 
  bool empty( void ) const {
    return data.empty();
  }
  // ݃ANZXZq
  inline T& operator () ( const int i ){
    return data[i];
  }
  // ǂݍ݃ANZXZq
  inline const T& operator () ( const int i ) const {
    return data[i];
  }
  // ݃ANZXZq
  inline T& operator () ( const int ix, const int iy, const int iz ){
    return data[ (ix*ny + iy)*nz + iz ];
  }
  // ǂݍ݃ANZXZq
  inline const T& operator () ( const int ix, const int iy, const int iz ) const {
    return data[ (ix*ny + iy)*nz + iz ];
  }
  // ݗpLXgZq
  inline operator const T* ( void ) const {
    return &data[0];
  }
  // ǂݍݗpLXgZq
  inline operator T* ( void ){
    return &data[0];
  }

  // vfׂĂ悸郁o֐
  void mul( const double d ){
    for( int i=0; i<(int)data.size(); i++ ){
      data[i] *= d;
    }
  }

  void range( void ){
    data_min = data.front();
    data_max = data.front();

    for( int i=0; i<(int)data.size(); i++ ){
      if( data_max < data[i] ) data_max = data[i];
      if( data_min > data[i] ) data_min = data[i];
    }
  }
  T min( void ) const {
    return data_min;
  }
  T max( void ) const {
    return data_max;
  }

  void setPeriodic( const bool periodic){
    this->periodic = periodic;
    if( periodic ){
      kx = (*this)(nx-1,0,0) - (*this)(0,0,0);
      ky = (*this)(0,ny-1,0) - (*this)(0,0,0);
      kz = (*this)(0,0,nz-1) - (*this)(0,0,0);
    }
    else{
      kx = T(0.0);
      ky = T(0.0);
      kz = T(0.0);
    }
  }
};

#endif // __VECTOR3D_H_INCLUDED
