/*
 Copyright (c) 2014 Shinji Tsuneyuki
 This file is distributed under the terms of the GNU General Public License version 3.
 */

/*!
 \file window.cc
 \brief CEBhENX
*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <QtGui/QtGui>
#include <QtOpenGL/QtOpenGL>
#include <GL/glu.h>

#include "window.h"
#include "qtwidgets.h"
#include "qtmisc.h"
#include "qtexception.h"

MainWindow::MainWindow( void ) :
  model(), view(model,this), view2d(model), ctrl(model,view)
{
  setFocusPolicy( Qt::ClickFocus );
  setWindowTitle( model.gloption.window.title );
  setMinimumSize( model.gloption.window.width, model.gloption.window.height );
  setSizePolicy( QSizePolicy::Expanding, QSizePolicy::Expanding );

  setCentralWidget( &view );

  resize( model.gloption.window.width, model.gloption.window.height );

  if( model.gloption.window.fullscreen ){
    showFullScreen();
  }

  // j[o[̐ݒ

  // file
  {
    QMenu* menu = menuBar()->addMenu(tr("&File  "));

    {
      QAction* action = new MyQAction
	(" &Clear all", this, SLOT(edit(const MyEvent&)), ID_FILE_NEW );
      menu->addAction(action);
    }
    menu->addSeparator();

    {
      QAction* action = new MyQAction
	(" Load &Molecule", this, SLOT(edit(const MyEvent&)), ID_FILE_LOAD_MOLECULE );
      menu->addAction(action);
    }
    {
      QAction* action = new MyQAction
	(" Load &xTAPP config file", this, SLOT(edit(const MyEvent&)), ID_FILE_LOAD_XTAPP );
      menu->addAction(action);
    }
    {
      QAction* action = new MyQAction
	(" Load &Field file...", this, SLOT(edit(const MyEvent&)), ID_FILE_LOAD_FIELD );
      menu->addAction(action);
    }
    {
      QAction* action = new MyQAction
	(" Load &Band data...", this, SLOT(edit(const MyEvent&)), ID_FILE_LOAD_BAND );
      menu->addAction(action);
    }
    {
      QAction* action = new MyQAction
	(" Load &Dos data...", this, SLOT(edit(const MyEvent&)), ID_FILE_LOAD_DOS );
      menu->addAction(action);
    }
    menu->addSeparator();
    {
      QAction* action = new MyQAction
	(" Save Molecule", this, SLOT(edit(const MyEvent&)), ID_FILE_SAVE_MOLECULE );
      menu->addAction(action);
    }
    {
      QAction* action = new MyQAction
	(" &Save xTAPP input file", this, SLOT(edit(const MyEvent&)), ID_FILE_SAVE_XTAPP );
      menu->addAction(action);
    }
    menu->addSeparator();
    {
      QAction* action = new MyQAction
	(" &Quit this software", this, SLOT(edit(const MyEvent&)), ID_FILE_QUIT );
      menu->addAction(action);
    }
  }

  //---- lattice
  {
    QMenu* menu = menuBar()->addMenu(tr("&Lattice  "));
    QAction* action = new MyQAction
      ("&Lattice  ", this, SLOT(edit(const MyEvent&)), ID_LATTICE );
    menu->addAction(action);
      //    menuBar()->addAction(action);
  }

  //---- atom list
  {
    QMenu* menu = menuBar()->addMenu(tr("&Atoms "));
    QAction* action = new MyQAction
      ("&Atoms ", this, SLOT(edit(const MyEvent&)), ID_ATOMS );
    menu->addAction(action);
    //    menuBar()->addAction(action);
  }

  //---- TAPP
  {
    QMenu* menu = menuBar()->addMenu(tr("&Tapp   "));
    {
      QAction* action = new MyQAction
	("&xTapp     ", this, SLOT(edit(const MyEvent&)), ID_XTAPP );
      menu->addAction(action);
      //    menuBar()->addAction(action);
    }
  }

  //---- graphics
  {
    QMenu* menu = menuBar()->addMenu(tr("&Graphics "));
    QAction* action = new MyQAction
      ("&Graphics ", this, SLOT(edit(const MyEvent&)), ID_GRAPHICS );
    menu->addAction(action);
    //    menuBar()->addAction(action);
  }

  //---- undo
  {
    QMenu* menu = menuBar()->addMenu(tr("&Undo "));
    QAction* action = new MyQAction
      ("&Undo ", this, SLOT(edit(const MyEvent&)), ID_UNDO );
    menu->addAction(action);
    //    menuBar()->addAction(action);
  }

#if 0
  //---- execute
  {
    QMenu* menu = menuBar()->addMenu(tr("&Execute"));
    // XTAPP
    {
      QAction* action = new MyQAction
	("Execute XTAPP", this, SLOT(edit(const MyEvent&)), ID_EXECUTE_XTAPP );
      menu->addAction(action);
    }
  }
#endif

  //---- view2d
  {
    QMenu* menu = menuBar()->addMenu(tr("&Viewer2D "));
    QAction* action = new MyQAction
      ("&Viewer2D ", this, SLOT(edit(const MyEvent&)), ID_VIEW2D );
    menu->addAction(action);
  }
}

MainWindow::~MainWindow( void )
{
}


void MainWindow::idle( void )
{
}

void MainWindow::edit( const MyEvent& ev )
{
  static bool first = true;
  if( first ){
    path = QDir::currentPath();
    first = false;
  }

  switch(ev.id){

  case ID_FILE_NEW : {
    model.clear();
  } break;


  case ID_FILE_LOAD_MOLECULE : {
    QString fname = QFileDialog::getOpenFileName
      ( this, tr("Load a Molecule file"), path,
	tr("XYZ format files(*.xyz);;"
	   "PDB format files(*.pdb);;"
	   "CIF format files(*.cif);;"
	   "All files(*)"));
    try{
      if( fname != "" && model.loadMolecule(fname) ){
	path = getDirName(fname);
      }
    }
    catch( const MyException& e ){
      MyException::critical(e);
    }
  } break;

  case ID_FILE_SAVE_MOLECULE : {
    QString fname = QFileDialog::getSaveFileName
      ( this, tr("Save a Molecule file"), path,
	tr("XYZ format files(*.xyz)"));
    
    try{
      if( fname != "" && model.saveMolecule(fname) ){
	path = getDirName(fname);
      }
    }
    catch( const MyException& e ){
      MyException::critical(e);
    }
  } break;

  case ID_FILE_LOAD_XTAPP : {
    QString fname = QFileDialog::getOpenFileName
      ( this, tr("Load a xTAPP config file"), path,
	tr("xTAPP config files(*.cg);;"
	   "xTAPP band config files(*.pef);;"
	   "xTAPP dos config files(*.w2c);;"
	   "xTAPP struct files(*.str);;"
	   "All files(*)"));
    try{
      if( fname != "" && model.loadXTAPP(fname) ){
	path = getDirName(fname);
      }
    }
    catch( const MyException& e ){
      MyException::critical(e);
    }
  } break;

  case ID_FILE_SAVE_XTAPP : {
    QString fname = QFileDialog::getSaveFileName
      ( this, tr("Save a xTAPP config file"), path,
	tr("xTAPP config files(*.cg);;"
	   "xTAPP band config files(*.pef);;"
	   "xTAPP dos files(*.w2c)"));
    
    try{
      if( fname != "" && model.saveXTAPP(fname) ){
	path = getDirName(fname);
      }
    }
    catch( const MyException& e ){
      MyException::critical(e);
    }
  } break;

  case ID_FILE_LOAD_FIELD : {
    QStringList vfname = QFileDialog::getOpenFileNames
      ( this, tr("Load field files"), path,
	tr("OpenDX files(*.dx);;"
	   "Gaussian Cube files(*.cube);;"
	   "TAPP VBWFN files(*.56;*.57);;All files(*)"));
    
    try{
      for( int n=0; n<vfname.size(); n++ ){
	if( vfname[n] != "" && model.loadField(vfname[n]) ){
	  path = getDirName(vfname[n]);
	}
      }
    }
    catch( const MyException& e ){
      MyException::critical(e);
    }
  } break;

  case ID_FILE_LOAD_BAND : {
    QString fname = QFileDialog::getOpenFileName
      ( this, tr("Load a band data file"), path,
	tr("TAPP band files(*.band);;All files(*)"));

    try{
      if( fname != "" && model.loadBand(fname) ){
	path = getDirName(fname);
	view2d.show();
	view2d.raise();
	view2d.tab->setCurrentIndex(0);
      }
    }
    catch( const MyException& e ){
      MyException::critical(e);
    }
  } break;

  case ID_FILE_LOAD_DOS : {
    QStringList vfname = QFileDialog::getOpenFileNames
      ( this, tr("Load dos files"), path,
	tr("xTAPP pdos files(*.pdos);;All files(*)"));
    
    try{
      for( int n=0; n<vfname.size(); n++ ){
	if( vfname[n] != "" && model.loadDos(vfname[n]) ){
	  path = getDirName(vfname[n]);
	  view2d.show();
	  view2d.raise();
	  view2d.tab->setCurrentIndex(1);
	}
      }
    }
    catch( const MyException& e ){
      MyException::critical(e);
    }
  } break;

  case ID_FILE_QUIT : {
    qApp->closeAllWindows();
  } break;

  case ID_LATTICE : {
    ctrl.showLattice();
  } break;

  case ID_XTAPP : {
    ctrl.showXTapp();
  } break;

  case ID_GRAPHICS : {
    ctrl.showGraphics();
  } break;

  case ID_ATOMS : {
    ctrl.showAtoms();
  } break;

  case ID_UNDO: {
    model.undo();
  } break;

  case ID_EXECUTE_XTAPP : {
    if( model.getFileNameSaved() == "" ){
      MyException::warning("Please save config before exec.");
    }
    else{
      QString command = "launch.bat xtapp " + model.getFileNameSaved();
      system(qPrintable(command));
    }

  } break;


  case ID_VIEW2D : {
    view2d.show();
    view2d.raise();
  } break;

  }
}


void MainWindow::update( void )
{
}

bool MainWindow::load( const QString& fname )
{
  try{
    if( false );
    else if( fname.endsWith(".xyz" ) ){
      return model.loadMolecule( fname );
    }
    else if( fname.endsWith(".pdb" ) ){
      return model.loadMolecule( fname );
    }
    else if( fname.endsWith(".cif" ) ){
      return model.loadMolecule( fname );
    }
    else if( fname.endsWith(".cg" ) ){
      return model.loadXTAPP( fname );
    }
    else if( fname.endsWith(".pef" ) ){
      return model.loadXTAPP( fname );
    }
    else if( fname.endsWith(".w2c" ) ){
      return model.loadXTAPP( fname );
    }
    else if( fname.endsWith(".str" ) ){
      return model.loadXTAPP( fname );
    }
    else if( fname.endsWith(".dx" ) ){
      return model.loadField(fname);
    }
    else if( fname.endsWith(".cube" ) ){
      return model.loadField(fname);
    }
    else if( fname.endsWith(".56" ) ){
      return model.loadField(fname);
    }
    else if( fname.endsWith(".57" ) ){
      return model.loadField(fname);
    }
    else if( fname.endsWith(".band" ) ){
      if( model.loadBand(fname) ){
	view2d.show();
	view2d.raise();
	view2d.tab->setCurrentIndex(0);
	return true;
      }
      return false;
    }
    else if( fname.endsWith(".pdos" ) ){
      if( model.loadDos(fname) ){
	view2d.show();
	view2d.raise();
	view2d.tab->setCurrentIndex(1);
	return true;
      }
      return false;
    }
    else{}
  }
  catch( const MyException& e ){
    MyException::critical(e);
  }

  return false;
}
